<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectTeamMember;
use App\Models\ProjectTeamMemberWorkHour;
use App\Models\ProjectPhase;
use App\Models\ProjectTask;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProjectTeamMemberWorkHourController extends Controller
{
    /**
     * عرض قائمة ساعات العمل لعضو الفريق
     */
    public function index(InstallationProject $installationProject, ProjectTeamMember $teamMember)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        $workHours = $teamMember->workHours()->with(['phase', 'task', 'approvedByUser'])->orderBy('date', 'desc')->get();

        return view('installation-projects.team-members.work-hours.index', compact('installationProject', 'teamMember', 'workHours'));
    }

    /**
     * عرض نموذج إضافة ساعات عمل جديدة
     */
    public function create(InstallationProject $installationProject, ProjectTeamMember $teamMember)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.team-members.show', [$installationProject->id, $teamMember->id])
                ->with('error', 'ليس لديك صلاحية لإضافة ساعات عمل');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        $phases = $installationProject->phases;
        $tasks = collect(); // سيتم تحميل المهام بناءً على المرحلة المختارة عبر Ajax

        return view('installation-projects.team-members.work-hours.create', compact('installationProject', 'teamMember', 'phases', 'tasks'));
    }

    /**
     * حفظ ساعات عمل جديدة
     */
    public function store(Request $request, InstallationProject $installationProject, ProjectTeamMember $teamMember)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.team-members.show', [$installationProject->id, $teamMember->id])
                ->with('error', 'ليس لديك صلاحية لإضافة ساعات عمل');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'date' => 'required|date',
            'hours' => 'required|numeric|min:0.5|max:24',
            'description' => 'required|string',
            'project_phase_id' => 'required|exists:project_phases,id',
            'project_task_id' => 'nullable|exists:project_tasks,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        $phase = ProjectPhase::findOrFail($request->project_phase_id);
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->back()->with('error', 'المرحلة المحددة لا تنتمي لهذا المشروع')->withInput();
        }

        // التحقق من أن المهمة تنتمي للمرحلة (إذا تم تحديدها)
        if ($request->project_task_id) {
            $task = ProjectTask::findOrFail($request->project_task_id);
            if ($task->project_phase_id !== $phase->id) {
                return redirect()->back()->with('error', 'المهمة المحددة لا تنتمي للمرحلة المحددة')->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // إضافة ساعات العمل
            $workHour = ProjectTeamMemberWorkHour::create([
                'project_team_member_id' => $teamMember->id,
                'date' => $request->date,
                'hours' => $request->hours,
                'description' => $request->description,
                'project_phase_id' => $request->project_phase_id,
                'project_task_id' => $request->project_task_id,
                'approved' => false,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('success', 'تم إضافة ساعات العمل بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة ساعات العمل: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض نموذج تعديل ساعات العمل
     */
    public function edit(InstallationProject $installationProject, ProjectTeamMember $teamMember, ProjectTeamMemberWorkHour $workHour)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.team-members.show', [$installationProject->id, $teamMember->id])
                ->with('error', 'ليس لديك صلاحية لتعديل ساعات العمل');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من أن ساعات العمل تنتمي لعضو الفريق
        if ($workHour->project_team_member_id !== $teamMember->id) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'ساعات العمل المطلوبة غير موجودة لهذا العضو');
        }

        // لا يمكن تعديل ساعات العمل المعتمدة
        if ($workHour->approved) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'لا يمكن تعديل ساعات العمل المعتمدة');
        }

        $phases = $installationProject->phases;
        $tasks = $workHour->project_phase_id ? ProjectPhase::find($workHour->project_phase_id)->tasks : collect();

        return view('installation-projects.team-members.work-hours.edit', compact('installationProject', 'teamMember', 'workHour', 'phases', 'tasks'));
    }

    /**
     * تحديث ساعات العمل
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectTeamMember $teamMember, ProjectTeamMemberWorkHour $workHour)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.team-members.show', [$installationProject->id, $teamMember->id])
                ->with('error', 'ليس لديك صلاحية لتعديل ساعات العمل');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من أن ساعات العمل تنتمي لعضو الفريق
        if ($workHour->project_team_member_id !== $teamMember->id) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'ساعات العمل المطلوبة غير موجودة لهذا العضو');
        }

        // لا يمكن تعديل ساعات العمل المعتمدة
        if ($workHour->approved) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'لا يمكن تعديل ساعات العمل المعتمدة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'date' => 'required|date',
            'hours' => 'required|numeric|min:0.5|max:24',
            'description' => 'required|string',
            'project_phase_id' => 'required|exists:project_phases,id',
            'project_task_id' => 'nullable|exists:project_tasks,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        $phase = ProjectPhase::findOrFail($request->project_phase_id);
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->back()->with('error', 'المرحلة المحددة لا تنتمي لهذا المشروع')->withInput();
        }

        // التحقق من أن المهمة تنتمي للمرحلة (إذا تم تحديدها)
        if ($request->project_task_id) {
            $task = ProjectTask::findOrFail($request->project_task_id);
            if ($task->project_phase_id !== $phase->id) {
                return redirect()->back()->with('error', 'المهمة المحددة لا تنتمي للمرحلة المحددة')->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // تحديث ساعات العمل
            $workHour->update([
                'date' => $request->date,
                'hours' => $request->hours,
                'description' => $request->description,
                'project_phase_id' => $request->project_phase_id,
                'project_task_id' => $request->project_task_id,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('success', 'تم تحديث ساعات العمل بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث ساعات العمل: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف ساعات العمل
     */
    public function destroy(InstallationProject $installationProject, ProjectTeamMember $teamMember, ProjectTeamMemberWorkHour $workHour)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.team-members.show', [$installationProject->id, $teamMember->id])
                ->with('error', 'ليس لديك صلاحية لحذف ساعات العمل');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من أن ساعات العمل تنتمي لعضو الفريق
        if ($workHour->project_team_member_id !== $teamMember->id) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'ساعات العمل المطلوبة غير موجودة لهذا العضو');
        }

        // لا يمكن حذف ساعات العمل المعتمدة
        if ($workHour->approved) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'لا يمكن حذف ساعات العمل المعتمدة');
        }

        try {
            DB::beginTransaction();

            // حذف ساعات العمل
            $workHour->delete();

            DB::commit();

            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('success', 'تم حذف ساعات العمل بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف ساعات العمل: ' . $e->getMessage());
        }
    }

    /**
     * اعتماد ساعات العمل
     */
    public function approve(InstallationProject $installationProject, ProjectTeamMember $teamMember, ProjectTeamMemberWorkHour $workHour)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('approve_work_hours')) {
            return redirect()->route('installation-projects.team-members.show', [$installationProject->id, $teamMember->id])
                ->with('error', 'ليس لديك صلاحية لاعتماد ساعات العمل');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من أن ساعات العمل تنتمي لعضو الفريق
        if ($workHour->project_team_member_id !== $teamMember->id) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'ساعات العمل المطلوبة غير موجودة لهذا العضو');
        }

        // لا يمكن اعتماد ساعات العمل المعتمدة مسبقاً
        if ($workHour->approved) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'ساعات العمل معتمدة بالفعل');
        }

        try {
            DB::beginTransaction();

            // اعتماد ساعات العمل
            $workHour->approve(Auth::id());

            DB::commit();

            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('success', 'تم اعتماد ساعات العمل بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء اعتماد ساعات العمل: ' . $e->getMessage());
        }
    }

    /**
     * إلغاء اعتماد ساعات العمل
     */
    public function unapprove(InstallationProject $installationProject, ProjectTeamMember $teamMember, ProjectTeamMemberWorkHour $workHour)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('approve_work_hours')) {
            return redirect()->route('installation-projects.team-members.show', [$installationProject->id, $teamMember->id])
                ->with('error', 'ليس لديك صلاحية لإلغاء اعتماد ساعات العمل');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من أن ساعات العمل تنتمي لعضو الفريق
        if ($workHour->project_team_member_id !== $teamMember->id) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'ساعات العمل المطلوبة غير موجودة لهذا العضو');
        }

        // لا يمكن إلغاء اعتماد ساعات العمل غير المعتمدة
        if (!$workHour->approved) {
            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('error', 'ساعات العمل غير معتمدة بالفعل');
        }

        try {
            DB::beginTransaction();

            // إلغاء اعتماد ساعات العمل
            $workHour->unapprove();

            DB::commit();

            return redirect()->route('installation-projects.team-members.work-hours.index', [$installationProject->id, $teamMember->id])
                ->with('success', 'تم إلغاء اعتماد ساعات العمل بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إلغاء اعتماد ساعات العمل: ' . $e->getMessage());
        }
    }

    /**
     * الحصول على المهام بناءً على المرحلة المحددة (للاستخدام مع Ajax)
     */
    public function getTasksByPhase(InstallationProject $installationProject, $phaseId)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية للوصول إلى هذه الصفحة']);
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية للوصول إلى هذا المشروع']);
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        $phase = ProjectPhase::find($phaseId);
        if (!$phase || $phase->installation_project_id !== $installationProject->id) {
            return response()->json(['success' => false, 'message' => 'المرحلة المطلوبة غير موجودة في هذا المشروع']);
        }

        $tasks = $phase->tasks;

        return response()->json([
            'success' => true,
            'tasks' => $tasks,
        ]);
    }
}
