<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectTeamMember;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProjectTeamMemberController extends Controller
{
    /**
     * عرض قائمة أعضاء فريق المشروع
     */
    public function index(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $teamMembers = $installationProject->teamMembers()->with('user')->get();

        return view('installation-projects.team-members.index', compact('installationProject', 'teamMembers'));
    }

    /**
     * عرض نموذج إضافة عضو جديد للفريق
     */
    public function create(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة عضو للفريق');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // الحصول على المستخدمين المتاحين
        $users = User::where('company_id', Auth::user()->company_id)
            ->whereHas('roles', function ($query) {
                $query->whereIn('name', ['admin', 'manager', 'project_manager', 'engineer', 'technician', 'employee']);
            })
            ->get();

        // الحصول على أعضاء الفريق الحاليين
        $currentTeamMemberIds = $installationProject->teamMembers()->pluck('user_id')->toArray();

        return view('installation-projects.team-members.create', compact('installationProject', 'users', 'currentTeamMemberIds'));
    }

    /**
     * حفظ عضو جديد للفريق
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة عضو للفريق');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'role' => 'required|string|max:100',
            'responsibilities' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'hourly_rate' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المستخدم ينتمي لنفس الشركة
        $user = User::findOrFail($request->user_id);
        if ($user->company_id !== Auth::user()->company_id) {
            return redirect()->back()->with('error', 'المستخدم المحدد لا ينتمي لشركتك')->withInput();
        }

        // التحقق من أن المستخدم ليس عضواً بالفعل في الفريق
        $existingMember = $installationProject->teamMembers()->where('user_id', $request->user_id)->first();
        if ($existingMember) {
            return redirect()->back()->with('error', 'المستخدم المحدد عضو بالفعل في فريق المشروع')->withInput();
        }

        try {
            DB::beginTransaction();

            // إضافة عضو جديد للفريق
            $teamMember = ProjectTeamMember::create([
                'installation_project_id' => $installationProject->id,
                'user_id' => $request->user_id,
                'role' => $request->role,
                'responsibilities' => $request->responsibilities,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'hourly_rate' => $request->hourly_rate,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('success', 'تم إضافة عضو الفريق بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة عضو الفريق: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل عضو الفريق
     */
    public function show(InstallationProject $installationProject, ProjectTeamMember $teamMember)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        $teamMember->load(['user', 'workHours.phase', 'workHours.task', 'workHours.approvedByUser']);

        return view('installation-projects.team-members.show', compact('installationProject', 'teamMember'));
    }

    /**
     * عرض نموذج تعديل عضو الفريق
     */
    public function edit(InstallationProject $installationProject, ProjectTeamMember $teamMember)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل عضو الفريق');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        return view('installation-projects.team-members.edit', compact('installationProject', 'teamMember'));
    }

    /**
     * تحديث عضو الفريق
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectTeamMember $teamMember)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل عضو الفريق');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'role' => 'required|string|max:100',
            'responsibilities' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'hourly_rate' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث عضو الفريق
            $teamMember->update([
                'role' => $request->role,
                'responsibilities' => $request->responsibilities,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'hourly_rate' => $request->hourly_rate,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.team-members.show', [$installationProject->id, $teamMember->id])
                ->with('success', 'تم تحديث عضو الفريق بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث عضو الفريق: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف عضو الفريق
     */
    public function destroy(InstallationProject $installationProject, ProjectTeamMember $teamMember)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لحذف عضو الفريق');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن عضو الفريق ينتمي للمشروع
        if ($teamMember->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('error', 'عضو الفريق المطلوب غير موجود في هذا المشروع');
        }

        try {
            DB::beginTransaction();

            // حذف عضو الفريق (سيتم حذف ساعات العمل تلقائياً بسبب العلاقات)
            $teamMember->delete();

            DB::commit();

            return redirect()->route('installation-projects.team-members.index', $installationProject->id)
                ->with('success', 'تم حذف عضو الفريق بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف عضو الفريق: ' . $e->getMessage());
        }
    }
}
