<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectPhase;
use App\Models\ProjectTask;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProjectTaskController extends Controller
{
    /**
     * عرض قائمة مهام المرحلة
     */
    public function index(InstallationProject $installationProject, ProjectPhase $phase)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        $tasks = $phase->tasks()->orderBy('order_number')->get();

        return view('installation-projects.tasks.index', compact('installationProject', 'phase', 'tasks'));
    }

    /**
     * عرض نموذج إضافة مهمة جديدة للمرحلة
     */
    public function create(InstallationProject $installationProject, ProjectPhase $phase)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('error', 'ليس لديك صلاحية لإضافة مهمة للمرحلة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // الحصول على المستخدمين المسؤولين المحتملين
        $assignableUsers = User::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->whereHas('roles', function ($query) {
                // استخدام الأدوار الموجودة فعلياً في النظام
                $query->whereIn('slug', ['admin', 'branch_manager', 'technician', 'user'])
                    ->orWhere('is_system_role', true);
            })
            ->orderBy('name')
            ->get();

        // الحصول على الرقم الترتيبي للمهمة الجديدة
        $nextOrderNumber = $phase->tasks()->max('order_number') + 1 ?? 1;

        return view('installation-projects.tasks.create', compact('installationProject', 'phase', 'assignableUsers', 'nextOrderNumber'));
    }

    /**
     * حفظ مهمة جديدة للمرحلة
     */
    public function store(Request $request, InstallationProject $installationProject, ProjectPhase $phase)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('error', 'ليس لديك صلاحية لإضافة مهمة للمرحلة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:start_date',
            'order_number' => 'required|integer|min:1',
            'priority' => 'required|string|in:low,medium,high,urgent',
            'status' => 'required|string|in:not_started,in_progress,completed,delayed,cancelled',
            'completion_percentage' => 'required|integer|min:0|max:100',
            'assigned_to' => 'nullable|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // إنشاء مهمة جديدة للمرحلة
            $task = ProjectTask::create([
                'project_phase_id' => $phase->id,
                'name' => $request->name,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'due_date' => $request->due_date,
                'order_number' => $request->order_number,
                'priority' => $request->priority,
                'status' => $request->status,
                'completion_percentage' => $request->completion_percentage,
                'assigned_to' => $request->assigned_to,
                'notes' => $request->notes,
            ]);

            // تحديث نسبة الإنجاز في المرحلة
            $phase->updateProgress();

            DB::commit();

            return redirect()->route('installation-projects.phases.tasks.index', [$installationProject->id, $phase->id])
                ->with('success', 'تم إضافة المهمة للمرحلة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة المهمة: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل المهمة
     */
    public function show(InstallationProject $installationProject, ProjectPhase $phase, ProjectTask $task)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من أن المهمة تنتمي للمرحلة
        if ($task->project_phase_id !== $phase->id) {
            return redirect()->route('installation-projects.phases.tasks.index', [$installationProject->id, $phase->id])
                ->with('error', 'المهمة المطلوبة غير موجودة في هذه المرحلة');
        }

        // تحميل العلاقات
        $task->load(['assignedUser', 'issues']);

        return view('installation-projects.tasks.show', compact('installationProject', 'phase', 'task'));
    }

    /**
     * عرض نموذج تعديل المهمة
     */
    public function edit(InstallationProject $installationProject, ProjectPhase $phase, ProjectTask $task)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('error', 'ليس لديك صلاحية لتعديل المهمة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من أن المهمة تنتمي للمرحلة
        if ($task->project_phase_id !== $phase->id) {
            return redirect()->route('installation-projects.phases.tasks.index', [$installationProject->id, $phase->id])
                ->with('error', 'المهمة المطلوبة غير موجودة في هذه المرحلة');
        }

        // الحصول على المستخدمين المسؤولين المحتملين
        $assignableUsers = User::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->whereHas('roles', function ($query) {
                // استخدام الأدوار الموجودة فعلياً في النظام
                $query->whereIn('slug', ['admin', 'branch_manager', 'technician', 'user'])
                    ->orWhere('is_system_role', true);
            })
            ->orderBy('name')
            ->get();

        return view('installation-projects.tasks.edit', compact('installationProject', 'phase', 'task', 'assignableUsers'));
    }

    /**
     * تحديث المهمة
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectPhase $phase, ProjectTask $task)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('error', 'ليس لديك صلاحية لتعديل المهمة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من أن المهمة تنتمي للمرحلة
        if ($task->project_phase_id !== $phase->id) {
            return redirect()->route('installation-projects.phases.tasks.index', [$installationProject->id, $phase->id])
                ->with('error', 'المهمة المطلوبة غير موجودة في هذه المرحلة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:start_date',
            'order_number' => 'required|integer|min:1',
            'priority' => 'required|string|in:low,medium,high,urgent',
            'status' => 'required|string|in:not_started,in_progress,completed,delayed,cancelled',
            'completion_percentage' => 'required|integer|min:0|max:100',
            'assigned_to' => 'nullable|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث المهمة
            $task->update([
                'name' => $request->name,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'due_date' => $request->due_date,
                'order_number' => $request->order_number,
                'priority' => $request->priority,
                'status' => $request->status,
                'completion_percentage' => $request->completion_percentage,
                'assigned_to' => $request->assigned_to,
                'notes' => $request->notes,
            ]);

            // تحديث نسبة الإنجاز في المرحلة
            $phase->updateProgress();

            DB::commit();

            return redirect()->route('installation-projects.phases.tasks.show', [$installationProject->id, $phase->id, $task->id])
                ->with('success', 'تم تحديث المهمة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث المهمة: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف المهمة
     */
    public function destroy(InstallationProject $installationProject, ProjectPhase $phase, ProjectTask $task)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('error', 'ليس لديك صلاحية لحذف المهمة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من أن المهمة تنتمي للمرحلة
        if ($task->project_phase_id !== $phase->id) {
            return redirect()->route('installation-projects.phases.tasks.index', [$installationProject->id, $phase->id])
                ->with('error', 'المهمة المطلوبة غير موجودة في هذه المرحلة');
        }

        try {
            DB::beginTransaction();

            // حذف المهمة
            $task->delete();

            // تحديث نسبة الإنجاز في المرحلة
            $phase->updateProgress();

            DB::commit();

            return redirect()->route('installation-projects.phases.tasks.index', [$installationProject->id, $phase->id])
                ->with('success', 'تم حذف المهمة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف المهمة: ' . $e->getMessage());
        }
    }

    /**
     * تحديث نسبة الإنجاز للمهمة
     */
    public function updateProgress(Request $request, InstallationProject $installationProject, ProjectPhase $phase, ProjectTask $task)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('error', 'ليس لديك صلاحية لتحديث نسبة الإنجاز');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من أن المهمة تنتمي للمرحلة
        if ($task->project_phase_id !== $phase->id) {
            return redirect()->route('installation-projects.phases.tasks.index', [$installationProject->id, $phase->id])
                ->with('error', 'المهمة المطلوبة غير موجودة في هذه المرحلة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'completion_percentage' => 'required|integer|min:0|max:100',
            'status' => 'required|string|in:not_started,in_progress,completed,delayed,cancelled',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث نسبة الإنجاز والحالة
            $task->update([
                'completion_percentage' => $request->completion_percentage,
                'status' => $request->status,
                'notes' => $request->has('notes') ? $task->notes . "\n" . now()->format('Y-m-d') . ": " . $request->notes : $task->notes,
            ]);

            // تحديث نسبة الإنجاز في المرحلة
            $phase->updateProgress();

            DB::commit();

            return redirect()->route('installation-projects.phases.tasks.show', [$installationProject->id, $phase->id, $task->id])
                ->with('success', 'تم تحديث نسبة الإنجاز بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث نسبة الإنجاز: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * تعيين المهمة لمستخدم
     */
    public function assignTask(Request $request, InstallationProject $installationProject, ProjectPhase $phase, ProjectTask $task)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('error', 'ليس لديك صلاحية لتعيين المهمة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من أن المهمة تنتمي للمرحلة
        if ($task->project_phase_id !== $phase->id) {
            return redirect()->route('installation-projects.phases.tasks.index', [$installationProject->id, $phase->id])
                ->with('error', 'المهمة المطلوبة غير موجودة في هذه المرحلة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'assigned_to' => 'required|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تعيين المهمة للمستخدم
            $task->update([
                'assigned_to' => $request->assigned_to,
                'notes' => $request->has('notes') ? $task->notes . "\n" . now()->format('Y-m-d') . ": تم تعيين المهمة لـ " . User::find($request->assigned_to)->name . ". " . $request->notes : $task->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.phases.tasks.show', [$installationProject->id, $phase->id, $task->id])
                ->with('success', 'تم تعيين المهمة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تعيين المهمة: ' . $e->getMessage())->withInput();
        }
    }
}
