<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectPhase;
use App\Models\ProjectRisk;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ProjectRiskController extends Controller
{
    /**
     * عرض قائمة المخاطر للمشروع
     */
    public function index(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $risks = ProjectRisk::where('installation_project_id', $installationProject->id)
            ->orderBy('risk_score', 'desc')
            ->get();

        $risksByLevel = $risks->groupBy(function ($risk) {
            return $risk->getRiskLevel();
        });

        $risksByStatus = $risks->groupBy('status');

        return view('installation-projects.risks.index', compact('installationProject', 'risks', 'risksByLevel', 'risksByStatus'));
    }

    /**
     * عرض نموذج إنشاء مخاطرة جديدة
     */
    public function create(InstallationProject $installationProject)
    {
        $this->authorize('update', $installationProject);

        $phases = $installationProject->phases;
        $users = User::where('company_id', Auth::user()->company_id)->get();

        return view('installation-projects.risks.create', compact('installationProject', 'phases', 'users'));
    }

    /**
     * تخزين مخاطرة جديدة
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        $this->authorize('update', $installationProject);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'probability' => 'required|in:low,medium,high,very_high',
            'impact' => 'required|in:low,medium,high,very_high',
            'status' => 'required|in:identified,analyzing,mitigating,monitoring,closed,occurred',
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'responsible_user_id' => 'nullable|exists:users,id',
            'identification_date' => 'required|date',
            'target_resolution_date' => 'nullable|date',
            'mitigation_plan' => 'nullable|string',
            'contingency_plan' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $risk = new ProjectRisk($request->all());
        $risk->installation_project_id = $installationProject->id;
        $risk->save();

        // حساب درجة المخاطرة
        $risk->calculateRiskScore();

        return redirect()->route('installation-projects.risks.index', $installationProject->id)
            ->with('success', 'تم إضافة المخاطرة بنجاح');
    }

    /**
     * عرض تفاصيل المخاطرة
     */
    public function show(InstallationProject $installationProject, ProjectRisk $risk)
    {
        $this->authorize('view', $installationProject);

        return view('installation-projects.risks.show', compact('installationProject', 'risk'));
    }

    /**
     * عرض نموذج تعديل المخاطرة
     */
    public function edit(InstallationProject $installationProject, ProjectRisk $risk)
    {
        $this->authorize('update', $installationProject);

        $phases = $installationProject->phases;
        $users = User::where('company_id', Auth::user()->company_id)->get();

        return view('installation-projects.risks.edit', compact('installationProject', 'risk', 'phases', 'users'));
    }

    /**
     * تحديث المخاطرة
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectRisk $risk)
    {
        $this->authorize('update', $installationProject);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'probability' => 'required|in:low,medium,high,very_high',
            'impact' => 'required|in:low,medium,high,very_high',
            'status' => 'required|in:identified,analyzing,mitigating,monitoring,closed,occurred',
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'responsible_user_id' => 'nullable|exists:users,id',
            'identification_date' => 'required|date',
            'target_resolution_date' => 'nullable|date',
            'actual_resolution_date' => 'nullable|date',
            'mitigation_plan' => 'nullable|string',
            'contingency_plan' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $risk->update($request->all());

        // حساب درجة المخاطرة
        $risk->calculateRiskScore();

        return redirect()->route('installation-projects.risks.index', $installationProject->id)
            ->with('success', 'تم تحديث المخاطرة بنجاح');
    }

    /**
     * حذف المخاطرة
     */
    public function destroy(InstallationProject $installationProject, ProjectRisk $risk)
    {
        $this->authorize('update', $installationProject);

        $risk->delete();

        return redirect()->route('installation-projects.risks.index', $installationProject->id)
            ->with('success', 'تم حذف المخاطرة بنجاح');
    }

    /**
     * عرض لوحة تحكم المخاطر
     */
    public function dashboard(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $risks = ProjectRisk::where('installation_project_id', $installationProject->id)->get();

        $risksByLevel = $risks->groupBy(function ($risk) {
            return $risk->getRiskLevel();
        });

        $risksByStatus = $risks->groupBy('status');

        $highPriorityRisks = $risks->filter(function ($risk) {
            return in_array($risk->getRiskLevel(), ['high', 'critical']);
        });

        $overdueRisks = $risks->filter(function ($risk) {
            return $risk->isOverdue();
        });

        $risksByPhase = $risks->groupBy('project_phase_id');

        return view('installation-projects.risks.dashboard', compact(
            'installationProject',
            'risks',
            'risksByLevel',
            'risksByStatus',
            'highPriorityRisks',
            'overdueRisks',
            'risksByPhase'
        ));
    }
}
