<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectResource;
use App\Models\ProjectResourceAllocation;
use App\Models\ProjectPhase;
use App\Models\ProjectTask;
use App\Models\InventorySupplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ProjectResourceController extends Controller
{
    /**
     * عرض قائمة الموارد للمشروع
     */
    public function index(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $resources = ProjectResource::where('installation_project_id', $installationProject->id)
            ->orderBy('name')
            ->get();

        $resourcesByType = $resources->groupBy('type');

        return view('installation-projects.resources.index', compact('installationProject', 'resources', 'resourcesByType'));
    }

    /**
     * عرض نموذج إنشاء مورد جديد
     */
    public function create(InstallationProject $installationProject)
    {
        $this->authorize('update', $installationProject);

        $suppliers = InventorySupplier::where('company_id', Auth::user()->company_id)->get();

        return view('installation-projects.resources.create', compact('installationProject', 'suppliers'));
    }

    /**
     * تخزين مورد جديد
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        $this->authorize('update', $installationProject);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'type' => 'required|in:human,material,equipment,other',
            'description' => 'nullable|string',
            'cost_per_unit' => 'required|numeric|min:0',
            'unit' => 'required|string|max:50',
            'available_units' => 'required|integer|min:0',
            'available_from' => 'nullable|date',
            'available_to' => 'nullable|date',
            'supplier_id' => 'nullable|exists:inventory_suppliers,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $resource = new ProjectResource($request->all());
        $resource->installation_project_id = $installationProject->id;
        $resource->allocated_units = 0;
        $resource->save();

        return redirect()->route('installation-projects.resources.index', $installationProject->id)
            ->with('success', 'تم إضافة المورد بنجاح');
    }

    /**
     * عرض تفاصيل المورد
     */
    public function show(InstallationProject $installationProject, ProjectResource $resource)
    {
        $this->authorize('view', $installationProject);

        $allocations = $resource->allocations;

        return view('installation-projects.resources.show', compact('installationProject', 'resource', 'allocations'));
    }

    /**
     * عرض نموذج تعديل المورد
     */
    public function edit(InstallationProject $installationProject, ProjectResource $resource)
    {
        $this->authorize('update', $installationProject);

        $suppliers = InventorySupplier::where('company_id', Auth::user()->company_id)->get();

        return view('installation-projects.resources.edit', compact('installationProject', 'resource', 'suppliers'));
    }

    /**
     * تحديث المورد
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectResource $resource)
    {
        $this->authorize('update', $installationProject);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'type' => 'required|in:human,material,equipment,other',
            'description' => 'nullable|string',
            'cost_per_unit' => 'required|numeric|min:0',
            'unit' => 'required|string|max:50',
            'available_units' => 'required|integer|min:0',
            'available_from' => 'nullable|date',
            'available_to' => 'nullable|date',
            'supplier_id' => 'nullable|exists:inventory_suppliers,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $resource->update($request->all());

        return redirect()->route('installation-projects.resources.index', $installationProject->id)
            ->with('success', 'تم تحديث المورد بنجاح');
    }

    /**
     * حذف المورد
     */
    public function destroy(InstallationProject $installationProject, ProjectResource $resource)
    {
        $this->authorize('update', $installationProject);

        // التحقق من عدم وجود تخصيصات للمورد
        if ($resource->allocations->count() > 0) {
            return redirect()->route('installation-projects.resources.index', $installationProject->id)
                ->with('error', 'لا يمكن حذف المورد لأنه مخصص لمهام أو مراحل');
        }

        $resource->delete();

        return redirect()->route('installation-projects.resources.index', $installationProject->id)
            ->with('success', 'تم حذف المورد بنجاح');
    }

    /**
     * عرض نموذج تخصيص المورد
     */
    public function allocateForm(InstallationProject $installationProject, ProjectResource $resource)
    {
        $this->authorize('update', $installationProject);

        $phases = $installationProject->phases;
        $tasks = ProjectTask::whereIn('project_phase_id', $phases->pluck('id'))->get();

        return view('installation-projects.resources.allocate', compact('installationProject', 'resource', 'phases', 'tasks'));
    }

    /**
     * تخصيص المورد
     */
    public function allocate(Request $request, InstallationProject $installationProject, ProjectResource $resource)
    {
        $this->authorize('update', $installationProject);

        $validator = Validator::make($request->all(), [
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'project_task_id' => 'nullable|exists:project_tasks,id',
            'allocated_units' => 'required|integer|min:1',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من توفر الوحدات الكافية
        if ($request->allocated_units > $resource->getRemainingUnits()) {
            return redirect()->back()
                ->with('error', 'عدد الوحدات المطلوبة أكبر من المتاح')
                ->withInput();
        }

        // التحقق من تحديد إما المرحلة أو المهمة
        if (empty($request->project_phase_id) && empty($request->project_task_id)) {
            return redirect()->back()
                ->with('error', 'يجب تحديد إما المرحلة أو المهمة')
                ->withInput();
        }

        $allocation = new ProjectResourceAllocation($request->all());
        $allocation->project_resource_id = $resource->id;
        $allocation->save();

        // حساب التكلفة
        $allocation->calculateCost();

        // تحديث عدد الوحدات المخصصة في المورد
        $resource->updateAllocatedUnits();

        return redirect()->route('installation-projects.resources.show', [$installationProject->id, $resource->id])
            ->with('success', 'تم تخصيص المورد بنجاح');
    }

    /**
     * إلغاء تخصيص المورد
     */
    public function deallocate(InstallationProject $installationProject, ProjectResource $resource, ProjectResourceAllocation $allocation)
    {
        $this->authorize('update', $installationProject);

        $allocation->delete();

        // تحديث عدد الوحدات المخصصة في المورد
        $resource->updateAllocatedUnits();

        return redirect()->route('installation-projects.resources.show', [$installationProject->id, $resource->id])
            ->with('success', 'تم إلغاء تخصيص المورد بنجاح');
    }

    /**
     * عرض تقرير استخدام الموارد
     */
    public function report(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $resources = ProjectResource::where('installation_project_id', $installationProject->id)->get();

        $resourcesByType = $resources->groupBy('type');

        $totalCost = $resources->sum(function ($resource) {
            return $resource->calculateTotalCost();
        });

        $resourceUtilization = $resources->map(function ($resource) {
            return [
                'name' => $resource->name,
                'type' => $resource->type,
                'available_units' => $resource->available_units,
                'allocated_units' => $resource->allocated_units,
                'utilization_percentage' => $resource->available_units > 0 ? round(($resource->allocated_units / $resource->available_units) * 100, 2) : 0,
                'total_cost' => $resource->calculateTotalCost(),
            ];
        });

        return view('installation-projects.resources.report', compact(
            'installationProject',
            'resources',
            'resourcesByType',
            'totalCost',
            'resourceUtilization'
        ));
    }
}
