<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectReport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use PDF;

class ProjectReportController extends Controller
{
    /**
     * عرض قائمة التقارير للمشروع
     */
    public function index(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $reports = ProjectReport::where('installation_project_id', $installationProject->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $reportsByType = $reports->groupBy('report_type');

        return view('installation-projects.reports.index', compact('installationProject', 'reports', 'reportsByType'));
    }

    /**
     * عرض نموذج إنشاء تقرير جديد
     */
    public function create(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $compareProjects = InstallationProject::where('company_id', Auth::user()->company_id)
            ->where('id', '!=', $installationProject->id)
            ->get();

        return view('installation-projects.reports.create', compact('installationProject', 'compareProjects'));
    }

    /**
     * تخزين تقرير جديد
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'report_type' => 'required|in:progress,financial,resource,earned_value,risk,comparison,custom',
            'description' => 'nullable|string',
            'period_start' => 'nullable|date',
            'period_end' => 'nullable|date|after_or_equal:period_start',
            'compare_project_ids' => 'nullable|array',
            'compare_project_ids.*' => 'exists:installation_projects,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $report = new ProjectReport();
        $report->installation_project_id = $installationProject->id;
        $report->title = $request->title;
        $report->report_type = $request->report_type;
        $report->description = $request->description;
        $report->report_date = now();
        $report->period_start = $request->period_start;
        $report->period_end = $request->period_end;
        $report->created_by = Auth::id();

        // تخزين معلمات التقرير
        $parameters = [];
        if ($request->report_type == 'comparison' && !empty($request->compare_project_ids)) {
            $parameters['compare_project_ids'] = $request->compare_project_ids;
        }
        $report->parameters = $parameters;

        $report->save();

        // توليد بيانات التقرير
        $report->generateReport();

        // إنشاء ملف PDF للتقرير
        $this->generatePdfReport($report);

        return redirect()->route('installation-projects.reports.show', [$installationProject->id, $report->id])
            ->with('success', 'تم إنشاء التقرير بنجاح');
    }

    /**
     * عرض تفاصيل التقرير
     */
    public function show(InstallationProject $installationProject, ProjectReport $report)
    {
        $this->authorize('view', $installationProject);

        return view('installation-projects.reports.show', compact('installationProject', 'report'));
    }

    /**
     * حذف التقرير
     */
    public function destroy(InstallationProject $installationProject, ProjectReport $report)
    {
        $this->authorize('update', $installationProject);

        // حذف ملف التقرير إذا كان موجوداً
        if ($report->file_path && Storage::exists($report->file_path)) {
            Storage::delete($report->file_path);
        }

        $report->delete();

        return redirect()->route('installation-projects.reports.index', $installationProject->id)
            ->with('success', 'تم حذف التقرير بنجاح');
    }

    /**
     * تنزيل التقرير كملف PDF
     */
    public function download(InstallationProject $installationProject, ProjectReport $report)
    {
        $this->authorize('view', $installationProject);

        if (!$report->file_path || !Storage::exists($report->file_path)) {
            // إنشاء ملف PDF للتقرير إذا لم يكن موجوداً
            $this->generatePdfReport($report);
        }

        return Storage::download($report->file_path, $report->title . '.pdf');
    }

    /**
     * إنشاء ملف PDF للتقرير
     */
    private function generatePdfReport(ProjectReport $report)
    {
        $data = $report->data;
        $view = 'installation-projects.reports.pdf.' . $report->report_type;

        $pdf = PDF::loadView($view, [
            'report' => $report,
            'data' => $data,
        ]);

        $fileName = 'reports/project_' . $report->installation_project_id . '_report_' . $report->id . '.pdf';
        Storage::put($fileName, $pdf->output());

        $report->file_path = $fileName;
        $report->save();

        return $fileName;
    }

    /**
     * عرض لوحة تحكم التقارير
     */
    public function dashboard(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $reports = ProjectReport::where('installation_project_id', $installationProject->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $reportsByType = $reports->groupBy('report_type');

        $latestReports = $reportsByType->map(function ($typeReports) {
            return $typeReports->first();
        });

        return view('installation-projects.reports.dashboard', compact(
            'installationProject',
            'reports',
            'reportsByType',
            'latestReports'
        ));
    }
}
