<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectPhase;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProjectPhaseController extends Controller
{
    /**
     * عرض قائمة مراحل المشروع
     */
    public function index(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $phases = $installationProject->phases()->orderBy('order_number')->get();

        return view('installation-projects.phases.index', compact('installationProject', 'phases'));
    }

    /**
     * عرض نموذج إضافة مرحلة جديدة للمشروع
     */
    public function create(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة مرحلة للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // الحصول على المستخدمين المسؤولين المحتملين
        $responsibleUsers = User::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->whereHas('roles', function ($query) {
                // استخدام الأدوار الموجودة فعلياً في النظام
                $query->whereIn('slug', ['admin', 'branch_manager', 'technician', 'user'])
                    ->orWhere('is_system_role', true);
            })
            ->orderBy('name')
            ->get();

        // الحصول على الرقم الترتيبي للمرحلة الجديدة
        $nextOrderNumber = $installationProject->phases()->max('order_number') + 1 ?? 1;

        return view('installation-projects.phases.create', compact('installationProject', 'responsibleUsers', 'nextOrderNumber'));
    }

    /**
     * حفظ مرحلة جديدة للمشروع
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة مرحلة للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'order_number' => 'required|integer|min:1',
            'budget' => 'required|numeric|min:0',
            'status' => 'required|string|in:not_started,in_progress,completed,delayed,cancelled',
            'completion_percentage' => 'required|integer|min:0|max:100',
            'responsible_user_id' => 'nullable|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // إنشاء مرحلة جديدة للمشروع
            $phase = ProjectPhase::create([
                'installation_project_id' => $installationProject->id,
                'name' => $request->name,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'order_number' => $request->order_number,
                'budget' => $request->budget,
                'status' => $request->status,
                'completion_percentage' => $request->completion_percentage,
                'responsible_user_id' => $request->responsible_user_id,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('success', 'تم إضافة المرحلة للمشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة المرحلة: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل مرحلة المشروع
     */
    public function show(InstallationProject $installationProject, ProjectPhase $phase)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // تحميل العلاقات
        $phase->load(['tasks', 'responsibleUser', 'expenses', 'issues', 'checklists']);

        return view('installation-projects.phases.show', compact('installationProject', 'phase'));
    }

    /**
     * عرض نموذج تعديل مرحلة المشروع
     */
    public function edit(InstallationProject $installationProject, ProjectPhase $phase)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل مرحلة المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // الحصول على المستخدمين المسؤولين المحتملين
        $responsibleUsers = User::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->whereHas('roles', function ($query) {
                // استخدام الأدوار الموجودة فعلياً في النظام
                $query->whereIn('slug', ['admin', 'branch_manager', 'technician', 'user'])
                    ->orWhere('is_system_role', true);
            })
            ->orderBy('name')
            ->get();

        return view('installation-projects.phases.edit', compact('installationProject', 'phase', 'responsibleUsers'));
    }

    /**
     * تحديث مرحلة المشروع
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectPhase $phase)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل مرحلة المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'order_number' => 'required|integer|min:1',
            'budget' => 'required|numeric|min:0',
            'status' => 'required|string|in:not_started,in_progress,completed,delayed,cancelled',
            'completion_percentage' => 'required|integer|min:0|max:100',
            'responsible_user_id' => 'nullable|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث مرحلة المشروع
            $phase->update([
                'name' => $request->name,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'order_number' => $request->order_number,
                'budget' => $request->budget,
                'status' => $request->status,
                'completion_percentage' => $request->completion_percentage,
                'responsible_user_id' => $request->responsible_user_id,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('success', 'تم تحديث المرحلة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث المرحلة: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف مرحلة المشروع
     */
    public function destroy(InstallationProject $installationProject, ProjectPhase $phase)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لحذف مرحلة المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        try {
            DB::beginTransaction();

            // حذف المرحلة (سيتم حذف المهام المرتبطة بها تلقائياً بسبب العلاقات)
            $phase->delete();

            DB::commit();

            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('success', 'تم حذف المرحلة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف المرحلة: ' . $e->getMessage());
        }
    }

    /**
     * تحديث نسبة الإنجاز للمرحلة
     */
    public function updateProgress(Request $request, InstallationProject $installationProject, ProjectPhase $phase)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتحديث نسبة الإنجاز');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المرحلة تنتمي للمشروع
        if ($phase->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.phases.index', $installationProject->id)
                ->with('error', 'المرحلة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'completion_percentage' => 'required|integer|min:0|max:100',
            'status' => 'required|string|in:not_started,in_progress,completed,delayed,cancelled',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث نسبة الإنجاز والحالة
            $phase->update([
                'completion_percentage' => $request->completion_percentage,
                'status' => $request->status,
                'notes' => $request->has('notes') ? $phase->notes . "\n" . now()->format('Y-m-d') . ": " . $request->notes : $phase->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.phases.show', [$installationProject->id, $phase->id])
                ->with('success', 'تم تحديث نسبة الإنجاز بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث نسبة الإنجاز: ' . $e->getMessage())->withInput();
        }
    }
}
