<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectPayment;
use App\Models\ProjectMilestone;
use App\Models\Invoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ProjectPaymentController extends Controller
{
    /**
     * عرض قائمة مدفوعات المشروع
     */
    public function index(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $payments = $installationProject->payments()->with(['milestone', 'invoice', 'createdBy', 'approvedBy'])->get();

        // حساب إجمالي المدفوعات
        $totalPayments = $payments->sum('total_amount');
        $totalApprovedPayments = $payments->where('status', 'approved')->sum('total_amount');
        $totalPendingPayments = $payments->where('status', 'pending')->sum('total_amount');

        // حساب النسبة المئوية للمدفوعات من إجمالي الميزانية
        $paymentPercentage = 0;
        if ($installationProject->total_budget > 0) {
            $paymentPercentage = round(($totalApprovedPayments / $installationProject->total_budget) * 100);
        }

        // الحصول على المعالم المالية المتبقية
        $remainingMilestones = $installationProject->milestones()
            ->where('is_financial', true)
            ->where(function ($query) {
                $query->where('status', '!=', 'completed')
                    ->orWhereNull('status');
            })
            ->get();

        return view('installation-projects.payments.index', compact(
            'installationProject',
            'payments',
            'totalPayments',
            'totalApprovedPayments',
            'totalPendingPayments',
            'paymentPercentage',
            'remainingMilestones'
        ));
    }

    /**
     * عرض نموذج إضافة دفعة جديدة
     */
    public function create(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة دفعة للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // الحصول على المعالم المالية
        $milestones = $installationProject->milestones()->where('is_financial', true)->get();

        // الحصول على الفواتير المرتبطة بالمشروع
        $invoices = Invoice::where('installation_project_id', $installationProject->id)
            ->where('status', 'issued')
            ->get();

        return view('installation-projects.payments.create', compact('installationProject', 'milestones', 'invoices'));
    }

    /**
     * حفظ دفعة جديدة
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة دفعة للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'payment_date' => 'required|date',
            'amount' => 'required|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'payment_method' => 'required|string|in:cash,bank_transfer,check,credit_card,online_payment,other',
            'reference_number' => 'nullable|string|max:100',
            'description' => 'required|string',
            'project_milestone_id' => 'nullable|exists:project_milestones,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'receipt_file' => 'nullable|file|max:5120', // 5MB كحد أقصى
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المعلم ينتمي للمشروع (إذا تم تحديده)
        if ($request->project_milestone_id) {
            $milestone = ProjectMilestone::findOrFail($request->project_milestone_id);
            if ($milestone->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'المعلم المحدد لا ينتمي لهذا المشروع')->withInput();
            }
        }

        // التحقق من أن الفاتورة تنتمي للمشروع (إذا تم تحديدها)
        if ($request->invoice_id) {
            $invoice = Invoice::findOrFail($request->invoice_id);
            if ($invoice->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'الفاتورة المحددة لا تنتمي لهذا المشروع')->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // حساب المبلغ الإجمالي
            $amount = $request->amount;
            $taxAmount = $request->tax_amount ?? 0;
            $totalAmount = $amount + $taxAmount;

            // معالجة ملف الإيصال إذا تم تحميله
            $receiptFilePath = null;
            if ($request->hasFile('receipt_file')) {
                $file = $request->file('receipt_file');
                $fileName = Str::uuid() . '_' . $file->getClientOriginalName();
                $receiptFilePath = 'projects/' . $installationProject->id . '/payments/' . $fileName;
                Storage::put($receiptFilePath, file_get_contents($file));
            }

            // إنشاء الدفعة
            $payment = ProjectPayment::create([
                'installation_project_id' => $installationProject->id,
                'project_milestone_id' => $request->project_milestone_id,
                'payment_date' => $request->payment_date,
                'amount' => $amount,
                'tax_amount' => $taxAmount,
                'total_amount' => $totalAmount,
                'payment_method' => $request->payment_method,
                'reference_number' => $request->reference_number,
                'description' => $request->description,
                'status' => 'pending',
                'invoice_id' => $request->invoice_id,
                'receipt_file' => $receiptFilePath,
                'created_by' => Auth::id(),
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('success', 'تم إضافة الدفعة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة الدفعة: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل الدفعة
     */
    public function show(InstallationProject $installationProject, ProjectPayment $payment)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن الدفعة تنتمي للمشروع
        if ($payment->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('error', 'الدفعة المطلوبة غير موجودة في هذا المشروع');
        }

        $payment->load(['milestone', 'invoice', 'createdBy', 'approvedBy']);

        return view('installation-projects.payments.show', compact('installationProject', 'payment'));
    }

    /**
     * عرض نموذج تعديل الدفعة
     */
    public function edit(InstallationProject $installationProject, ProjectPayment $payment)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل الدفعة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن الدفعة تنتمي للمشروع
        if ($payment->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('error', 'الدفعة المطلوبة غير موجودة في هذا المشروع');
        }

        // لا يمكن تعديل الدفعات المعتمدة
        if ($payment->isApproved()) {
            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('error', 'لا يمكن تعديل الدفعات المعتمدة');
        }

        // الحصول على المعالم المالية
        $milestones = $installationProject->milestones()->where('is_financial', true)->get();

        // الحصول على الفواتير المرتبطة بالمشروع
        $invoices = Invoice::where('installation_project_id', $installationProject->id)
            ->where('status', 'issued')
            ->get();

        return view('installation-projects.payments.edit', compact('installationProject', 'payment', 'milestones', 'invoices'));
    }

    /**
     * تحديث الدفعة
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectPayment $payment)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل الدفعة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن الدفعة تنتمي للمشروع
        if ($payment->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('error', 'الدفعة المطلوبة غير موجودة في هذا المشروع');
        }

        // لا يمكن تعديل الدفعات المعتمدة
        if ($payment->isApproved()) {
            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('error', 'لا يمكن تعديل الدفعات المعتمدة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'payment_date' => 'required|date',
            'amount' => 'required|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'payment_method' => 'required|string|in:cash,bank_transfer,check,credit_card,online_payment,other',
            'reference_number' => 'nullable|string|max:100',
            'description' => 'required|string',
            'project_milestone_id' => 'nullable|exists:project_milestones,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'receipt_file' => 'nullable|file|max:5120', // 5MB كحد أقصى
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المعلم ينتمي للمشروع (إذا تم تحديده)
        if ($request->project_milestone_id) {
            $milestone = ProjectMilestone::findOrFail($request->project_milestone_id);
            if ($milestone->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'المعلم المحدد لا ينتمي لهذا المشروع')->withInput();
            }
        }

        // التحقق من أن الفاتورة تنتمي للمشروع (إذا تم تحديدها)
        if ($request->invoice_id) {
            $invoice = Invoice::findOrFail($request->invoice_id);
            if ($invoice->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'الفاتورة المحددة لا تنتمي لهذا المشروع')->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // حساب المبلغ الإجمالي
            $amount = $request->amount;
            $taxAmount = $request->tax_amount ?? 0;
            $totalAmount = $amount + $taxAmount;

            // تحديث بيانات الدفعة
            $updateData = [
                'project_milestone_id' => $request->project_milestone_id,
                'payment_date' => $request->payment_date,
                'amount' => $amount,
                'tax_amount' => $taxAmount,
                'total_amount' => $totalAmount,
                'payment_method' => $request->payment_method,
                'reference_number' => $request->reference_number,
                'description' => $request->description,
                'invoice_id' => $request->invoice_id,
                'notes' => $request->notes,
            ];

            // معالجة ملف الإيصال إذا تم تحميله
            if ($request->hasFile('receipt_file')) {
                // حذف الملف القديم إذا وجد
                if ($payment->receipt_file) {
                    Storage::delete($payment->receipt_file);
                }

                $file = $request->file('receipt_file');
                $fileName = Str::uuid() . '_' . $file->getClientOriginalName();
                $receiptFilePath = 'projects/' . $installationProject->id . '/payments/' . $fileName;
                Storage::put($receiptFilePath, file_get_contents($file));

                $updateData['receipt_file'] = $receiptFilePath;
            }

            // تحديث الدفعة
            $payment->update($updateData);

            DB::commit();

            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('success', 'تم تحديث الدفعة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث الدفعة: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف الدفعة
     */
    public function destroy(InstallationProject $installationProject, ProjectPayment $payment)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لحذف الدفعة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن الدفعة تنتمي للمشروع
        if ($payment->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('error', 'الدفعة المطلوبة غير موجودة في هذا المشروع');
        }

        // لا يمكن حذف الدفعات المعتمدة
        if ($payment->isApproved()) {
            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('error', 'لا يمكن حذف الدفعات المعتمدة');
        }

        try {
            DB::beginTransaction();

            // حذف ملف الإيصال إذا وجد
            if ($payment->receipt_file) {
                Storage::delete($payment->receipt_file);
            }

            // حذف الدفعة
            $payment->delete();

            DB::commit();

            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('success', 'تم حذف الدفعة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف الدفعة: ' . $e->getMessage());
        }
    }

    /**
     * اعتماد الدفعة
     */
    public function approve(Request $request, InstallationProject $installationProject, ProjectPayment $payment)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('approve_payments')) {
            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('error', 'ليس لديك صلاحية لاعتماد الدفعات');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن الدفعة تنتمي للمشروع
        if ($payment->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('error', 'الدفعة المطلوبة غير موجودة في هذا المشروع');
        }

        // لا يمكن اعتماد الدفعات المعتمدة مسبقاً
        if ($payment->isApproved()) {
            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('error', 'الدفعة معتمدة بالفعل');
        }

        try {
            DB::beginTransaction();

            // اعتماد الدفعة
            $payment->approve(Auth::id());

            // إذا كانت الدفعة مرتبطة بمعلم مالي، قم بتحديث حالة المعلم
            if ($payment->project_milestone_id) {
                $milestone = $payment->milestone;
                $milestone->status = 'completed';
                $milestone->completed_at = now();
                $milestone->completed_by = Auth::id();
                $milestone->save();
            }

            // إذا كانت الدفعة مرتبطة بفاتورة، قم بتحديث حالة الفاتورة
            if ($payment->invoice_id) {
                $invoice = $payment->invoice;
                $invoice->status = 'paid';
                $invoice->payment_date = $payment->payment_date;
                $invoice->save();
            }

            DB::commit();

            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('success', 'تم اعتماد الدفعة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء اعتماد الدفعة: ' . $e->getMessage());
        }
    }

    /**
     * رفض الدفعة
     */
    public function reject(Request $request, InstallationProject $installationProject, ProjectPayment $payment)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('approve_payments')) {
            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('error', 'ليس لديك صلاحية لرفض الدفعات');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن الدفعة تنتمي للمشروع
        if ($payment->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('error', 'الدفعة المطلوبة غير موجودة في هذا المشروع');
        }

        // لا يمكن رفض الدفعات المعتمدة
        if ($payment->isApproved()) {
            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('error', 'لا يمكن رفض الدفعات المعتمدة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'rejection_notes' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // رفض الدفعة
            $payment->reject(Auth::id(), $request->rejection_notes);

            DB::commit();

            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('success', 'تم رفض الدفعة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء رفض الدفعة: ' . $e->getMessage());
        }
    }

    /**
     * تنزيل ملف الإيصال
     */
    public function downloadReceipt(InstallationProject $installationProject, ProjectPayment $payment)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن الدفعة تنتمي للمشروع
        if ($payment->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.payments.index', $installationProject->id)
                ->with('error', 'الدفعة المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من وجود ملف الإيصال
        if (!$payment->receipt_file || !Storage::exists($payment->receipt_file)) {
            return redirect()->route('installation-projects.payments.show', [$installationProject->id, $payment->id])
                ->with('error', 'ملف الإيصال غير موجود');
        }

        // تنزيل الملف
        $fileName = pathinfo($payment->receipt_file, PATHINFO_BASENAME);
        return Storage::download($payment->receipt_file, $fileName);
    }
}
