<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectMilestone;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProjectMilestoneController extends Controller
{
    /**
     * عرض قائمة المعالم الرئيسية للمشروع
     */
    public function index(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $milestones = $installationProject->milestones()->orderBy('order_number')->get();

        return view('installation-projects.milestones.index', compact('installationProject', 'milestones'));
    }

    /**
     * عرض نموذج إضافة معلم رئيسي جديد للمشروع
     */
    public function create(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة معلم رئيسي للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // الحصول على الرقم الترتيبي للمعلم الرئيسي الجديد
        $nextOrderNumber = $installationProject->milestones()->max('order_number') + 1 ?? 1;

        return view('installation-projects.milestones.create', compact('installationProject', 'nextOrderNumber'));
    }

    /**
     * حفظ معلم رئيسي جديد للمشروع
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة معلم رئيسي للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'due_date' => 'required|date',
            'payment_amount' => 'nullable|numeric|min:0',
            'status' => 'required|string|in:pending,completed,delayed,cancelled',
            'order_number' => 'required|integer|min:1',
            'deliverables' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // إنشاء معلم رئيسي جديد للمشروع
            $milestone = ProjectMilestone::create([
                'installation_project_id' => $installationProject->id,
                'title' => $request->title,
                'description' => $request->description,
                'due_date' => $request->due_date,
                'completion_date' => $request->status === 'completed' ? now() : null,
                'payment_amount' => $request->payment_amount,
                'status' => $request->status,
                'order_number' => $request->order_number,
                'deliverables' => $request->deliverables,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.milestones.index', $installationProject->id)
                ->with('success', 'تم إضافة المعلم الرئيسي للمشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة المعلم الرئيسي: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل المعلم الرئيسي
     */
    public function show(InstallationProject $installationProject, ProjectMilestone $milestone)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المعلم الرئيسي ينتمي للمشروع
        if ($milestone->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.milestones.index', $installationProject->id)
                ->with('error', 'المعلم الرئيسي المطلوب غير موجود في هذا المشروع');
        }

        return view('installation-projects.milestones.show', compact('installationProject', 'milestone'));
    }

    /**
     * عرض نموذج تعديل المعلم الرئيسي
     */
    public function edit(InstallationProject $installationProject, ProjectMilestone $milestone)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل المعلم الرئيسي');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المعلم الرئيسي ينتمي للمشروع
        if ($milestone->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.milestones.index', $installationProject->id)
                ->with('error', 'المعلم الرئيسي المطلوب غير موجود في هذا المشروع');
        }

        return view('installation-projects.milestones.edit', compact('installationProject', 'milestone'));
    }

    /**
     * تحديث المعلم الرئيسي
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectMilestone $milestone)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل المعلم الرئيسي');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المعلم الرئيسي ينتمي للمشروع
        if ($milestone->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.milestones.index', $installationProject->id)
                ->with('error', 'المعلم الرئيسي المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'due_date' => 'required|date',
            'completion_date' => 'nullable|date',
            'payment_amount' => 'nullable|numeric|min:0',
            'status' => 'required|string|in:pending,completed,delayed,cancelled',
            'order_number' => 'required|integer|min:1',
            'deliverables' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث المعلم الرئيسي
            $milestone->update([
                'title' => $request->title,
                'description' => $request->description,
                'due_date' => $request->due_date,
                'completion_date' => $request->status === 'completed' ? ($request->completion_date ?? now()) : null,
                'payment_amount' => $request->payment_amount,
                'status' => $request->status,
                'order_number' => $request->order_number,
                'deliverables' => $request->deliverables,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.milestones.show', [$installationProject->id, $milestone->id])
                ->with('success', 'تم تحديث المعلم الرئيسي بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث المعلم الرئيسي: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف المعلم الرئيسي
     */
    public function destroy(InstallationProject $installationProject, ProjectMilestone $milestone)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لحذف المعلم الرئيسي');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المعلم الرئيسي ينتمي للمشروع
        if ($milestone->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.milestones.index', $installationProject->id)
                ->with('error', 'المعلم الرئيسي المطلوب غير موجود في هذا المشروع');
        }

        try {
            DB::beginTransaction();

            // حذف المعلم الرئيسي
            $milestone->delete();

            DB::commit();

            return redirect()->route('installation-projects.milestones.index', $installationProject->id)
                ->with('success', 'تم حذف المعلم الرئيسي بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف المعلم الرئيسي: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة المعلم الرئيسي
     */
    public function updateStatus(Request $request, InstallationProject $installationProject, ProjectMilestone $milestone)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتحديث حالة المعلم الرئيسي');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المعلم الرئيسي ينتمي للمشروع
        if ($milestone->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.milestones.index', $installationProject->id)
                ->with('error', 'المعلم الرئيسي المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'status' => 'required|string|in:pending,completed,delayed,cancelled',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث حالة المعلم الرئيسي
            $milestone->update([
                'status' => $request->status,
                'completion_date' => $request->status === 'completed' ? now() : null,
                'notes' => $request->notes ? $milestone->notes . "\n" . now()->format('Y-m-d H:i') . ": " . $request->notes : $milestone->notes,
            ]);

            DB::commit();

            return redirect()->back()->with('success', 'تم تحديث حالة المعلم الرئيسي بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث حالة المعلم الرئيسي: ' . $e->getMessage());
        }
    }
}
