<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectPhase;
use App\Models\ProjectTask;
use App\Models\ProjectTaskDependency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ProjectGanttController extends Controller
{
    /**
     * عرض مخطط جانت للمشروع
     */
    public function index(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $phases = $installationProject->phases;
        $tasks = ProjectTask::whereIn('project_phase_id', $phases->pluck('id'))->get();
        $dependencies = ProjectTaskDependency::whereIn('task_id', $tasks->pluck('id'))->get();

        // تحضير البيانات لمخطط جانت
        $ganttData = $this->prepareGanttData($phases, $tasks, $dependencies);

        return view('installation-projects.gantt.index', compact('installationProject', 'ganttData'));
    }

    /**
     * تحديث مهمة من مخطط جانت
     */
    public function updateTask(Request $request, InstallationProject $installationProject)
    {
        $this->authorize('update', $installationProject);

        $validator = Validator::make($request->all(), [
            'task_id' => 'required|exists:project_tasks,id',
            'start_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:start_date',
            'progress' => 'required|integer|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        $task = ProjectTask::findOrFail($request->task_id);

        // التحقق من أن المهمة تنتمي للمشروع
        $phase = $task->phase;
        if ($phase->installation_project_id != $installationProject->id) {
            return response()->json(['error' => 'المهمة لا تنتمي للمشروع'], 403);
        }

        $task->start_date = $request->start_date;
        $task->due_date = $request->due_date;
        $task->completion_percentage = $request->progress;

        if ($request->progress == 100) {
            $task->status = 'completed';
        } elseif ($request->progress > 0) {
            $task->status = 'in_progress';
        } else {
            $task->status = 'not_started';
        }

        $task->save();

        // تحديث نسبة الإنجاز في المرحلة والمشروع
        $phase->updateProgress();
        $installationProject->updateProgress();

        return response()->json(['success' => true]);
    }

    /**
     * إضافة اعتمادية بين المهام
     */
    public function addDependency(Request $request, InstallationProject $installationProject)
    {
        $this->authorize('update', $installationProject);

        $validator = Validator::make($request->all(), [
            'task_id' => 'required|exists:project_tasks,id',
            'dependency_task_id' => 'required|exists:project_tasks,id|different:task_id',
            'dependency_type' => 'required|in:finish_to_start,start_to_start,finish_to_finish,start_to_finish',
            'lag_days' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        $task = ProjectTask::findOrFail($request->task_id);
        $dependencyTask = ProjectTask::findOrFail($request->dependency_task_id);

        // التحقق من أن المهام تنتمي للمشروع
        $phase = $task->phase;
        $dependencyPhase = $dependencyTask->phase;

        if (
            $phase->installation_project_id != $installationProject->id ||
            $dependencyPhase->installation_project_id != $installationProject->id
        ) {
            return response()->json(['error' => 'المهام لا تنتمي للمشروع'], 403);
        }

        // التحقق من عدم وجود اعتمادية مسبقة
        $existingDependency = ProjectTaskDependency::where('task_id', $request->task_id)
            ->where('dependency_task_id', $request->dependency_task_id)
            ->first();

        if ($existingDependency) {
            return response()->json(['error' => 'توجد اعتمادية مسبقة بين هاتين المهمتين'], 422);
        }

        $dependency = new ProjectTaskDependency();
        $dependency->task_id = $request->task_id;
        $dependency->dependency_task_id = $request->dependency_task_id;
        $dependency->dependency_type = $request->dependency_type;
        $dependency->lag_days = $request->lag_days ?? 0;
        $dependency->save();

        return response()->json(['success' => true, 'dependency_id' => $dependency->id]);
    }

    /**
     * حذف اعتمادية بين المهام
     */
    public function removeDependency(Request $request, InstallationProject $installationProject)
    {
        $this->authorize('update', $installationProject);

        $validator = Validator::make($request->all(), [
            'dependency_id' => 'required|exists:project_task_dependencies,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        $dependency = ProjectTaskDependency::findOrFail($request->dependency_id);

        // التحقق من أن الاعتمادية تنتمي للمشروع
        $task = $dependency->task;
        $phase = $task->phase;

        if ($phase->installation_project_id != $installationProject->id) {
            return response()->json(['error' => 'الاعتمادية لا تنتمي للمشروع'], 403);
        }

        $dependency->delete();

        return response()->json(['success' => true]);
    }

    /**
     * تحضير البيانات لمخطط جانت
     */
    private function prepareGanttData($phases, $tasks, $dependencies)
    {
        $ganttData = [
            'data' => [],
            'links' => []
        ];

        // إضافة المراحل
        foreach ($phases as $phase) {
            $ganttData['data'][] = [
                'id' => 'phase_' . $phase->id,
                'text' => $phase->name,
                'start_date' => $phase->start_date->format('Y-m-d'),
                'end_date' => $phase->end_date->format('Y-m-d'),
                'progress' => $phase->completion_percentage / 100,
                'type' => 'project',
                'open' => true,
                'parent' => 0
            ];
        }

        // إضافة المهام
        foreach ($tasks as $task) {
            $ganttData['data'][] = [
                'id' => 'task_' . $task->id,
                'text' => $task->name,
                'start_date' => $task->start_date->format('Y-m-d'),
                'end_date' => $task->due_date->format('Y-m-d'),
                'progress' => $task->completion_percentage / 100,
                'parent' => 'phase_' . $task->project_phase_id,
                'priority' => $task->priority,
                'responsible' => $task->assignedUser ? $task->assignedUser->name : null
            ];
        }

        // إضافة الاعتماديات
        foreach ($dependencies as $dependency) {
            $type = 0; // finish_to_start (default)

            switch ($dependency->dependency_type) {
                case 'start_to_start':
                    $type = 1;
                    break;
                case 'finish_to_finish':
                    $type = 2;
                    break;
                case 'start_to_finish':
                    $type = 3;
                    break;
            }

            $ganttData['links'][] = [
                'id' => $dependency->id,
                'source' => 'task_' . $dependency->dependency_task_id,
                'target' => 'task_' . $dependency->task_id,
                'type' => $type,
                'lag' => $dependency->lag_days
            ];
        }

        return $ganttData;
    }

    /**
     * عرض المسار الحرج للمشروع
     */
    public function criticalPath(InstallationProject $installationProject)
    {
        $this->authorize('view', $installationProject);

        $phases = $installationProject->phases;
        $tasks = ProjectTask::whereIn('project_phase_id', $phases->pluck('id'))->get();
        $dependencies = ProjectTaskDependency::whereIn('task_id', $tasks->pluck('id'))->get();

        // حساب المسار الحرج
        $criticalPath = $this->calculateCriticalPath($tasks, $dependencies);

        return view('installation-projects.gantt.critical-path', compact('installationProject', 'criticalPath'));
    }

    /**
     * حساب المسار الحرج للمشروع
     */
    private function calculateCriticalPath($tasks, $dependencies)
    {
        // تنفيذ خوارزمية المسار الحرج (CPM)
        // هذه نسخة مبسطة من الخوارزمية

        $taskData = [];

        // تهيئة بيانات المهام
        foreach ($tasks as $task) {
            $taskData[$task->id] = [
                'task' => $task,
                'duration' => $task->start_date->diffInDays($task->due_date) + 1,
                'es' => 0, // Early Start
                'ef' => 0, // Early Finish
                'ls' => 0, // Late Start
                'lf' => 0, // Late Finish
                'slack' => 0,
                'is_critical' => false,
                'predecessors' => [],
                'successors' => []
            ];
        }

        // بناء علاقات الاعتماد
        foreach ($dependencies as $dependency) {
            if ($dependency->dependency_type == 'finish_to_start') {
                $taskData[$dependency->task_id]['predecessors'][] = [
                    'task_id' => $dependency->dependency_task_id,
                    'lag' => $dependency->lag_days
                ];

                $taskData[$dependency->dependency_task_id]['successors'][] = [
                    'task_id' => $dependency->task_id,
                    'lag' => $dependency->lag_days
                ];
            }
        }

        // حساب البداية المبكرة والنهاية المبكرة (Forward Pass)
        $startTasks = array_filter($taskData, function ($data) {
            return empty($data['predecessors']);
        });

        foreach ($startTasks as $taskId => $data) {
            $this->forwardPass($taskId, $taskData);
        }

        // تحديد أقصى نهاية مبكرة
        $maxEf = 0;
        foreach ($taskData as $taskId => $data) {
            $maxEf = max($maxEf, $data['ef']);
        }

        // حساب البداية المتأخرة والنهاية المتأخرة (Backward Pass)
        $endTasks = array_filter($taskData, function ($data) {
            return empty($data['successors']);
        });

        foreach ($endTasks as $taskId => $data) {
            $taskData[$taskId]['lf'] = $maxEf;
            $taskData[$taskId]['ls'] = $taskData[$taskId]['lf'] - $taskData[$taskId]['duration'];
            $this->backwardPass($taskId, $taskData);
        }

        // حساب الفائض وتحديد المسار الحرج
        foreach ($taskData as $taskId => $data) {
            $taskData[$taskId]['slack'] = $data['ls'] - $data['es'];
            $taskData[$taskId]['is_critical'] = ($data['slack'] == 0);
        }

        // استخراج المهام على المسار الحرج
        $criticalTasks = array_filter($taskData, function ($data) {
            return $data['is_critical'];
        });

        return $criticalTasks;
    }

    /**
     * حساب البداية المبكرة والنهاية المبكرة (Forward Pass)
     */
    private function forwardPass($taskId, &$taskData)
    {
        $task = $taskData[$taskId];

        foreach ($task['successors'] as $successor) {
            $successorId = $successor['task_id'];
            $lag = $successor['lag'];

            $newEs = $task['ef'] + $lag;

            if ($newEs > $taskData[$successorId]['es']) {
                $taskData[$successorId]['es'] = $newEs;
                $taskData[$successorId]['ef'] = $newEs + $taskData[$successorId]['duration'];
            }

            $this->forwardPass($successorId, $taskData);
        }
    }

    /**
     * حساب البداية المتأخرة والنهاية المتأخرة (Backward Pass)
     */
    private function backwardPass($taskId, &$taskData)
    {
        $task = $taskData[$taskId];

        foreach ($task['predecessors'] as $predecessor) {
            $predecessorId = $predecessor['task_id'];
            $lag = $predecessor['lag'];

            $newLf = $task['ls'] - $lag;

            if ($taskData[$predecessorId]['lf'] == 0 || $newLf < $taskData[$predecessorId]['lf']) {
                $taskData[$predecessorId]['lf'] = $newLf;
                $taskData[$predecessorId]['ls'] = $newLf - $taskData[$predecessorId]['duration'];
            }

            $this->backwardPass($predecessorId, $taskData);
        }
    }
}
