<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectExpense;
use App\Models\ProjectPhase;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ProjectExpenseController extends Controller
{
    /**
     * عرض قائمة مصروفات المشروع
     */
    public function index(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $expenses = $installationProject->expenses()->with(['phase', 'supplier', 'approvedBy', 'createdBy'])->get();

        // تجميع المصروفات حسب النوع
        $expensesByType = $expenses->groupBy('expense_type');

        // حساب إجمالي المصروفات
        $totalExpenses = $expenses->sum('total_amount');
        $totalApprovedExpenses = $expenses->where('status', 'approved')->sum('total_amount');
        $totalPendingExpenses = $expenses->where('status', 'pending')->sum('total_amount');

        return view('installation-projects.expenses.index', compact(
            'installationProject',
            'expenses',
            'expensesByType',
            'totalExpenses',
            'totalApprovedExpenses',
            'totalPendingExpenses'
        ));
    }

    /**
     * عرض نموذج إضافة مصروف جديد
     */
    public function create(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة مصروف للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $phases = $installationProject->phases;
        $suppliers = Supplier::where('company_id', Auth::user()->company_id)->get();

        return view('installation-projects.expenses.create', compact('installationProject', 'phases', 'suppliers'));
    }

    /**
     * حفظ مصروف جديد
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة مصروف للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'expense_date' => 'required|date',
            'amount' => 'required|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'expense_type' => 'required|string|in:material,labor,equipment,transportation,accommodation,food,permit,insurance,other',
            'description' => 'required|string',
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'reference_number' => 'nullable|string|max:100',
            'payment_method' => 'required|string|in:cash,bank_transfer,check,credit_card,company_account,other',
            'receipt_number' => 'nullable|string|max:100',
            'receipt_file' => 'nullable|file|max:5120', // 5MB كحد أقصى
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المرحلة تنتمي للمشروع (إذا تم تحديدها)
        if ($request->project_phase_id) {
            $phase = ProjectPhase::findOrFail($request->project_phase_id);
            if ($phase->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'المرحلة المحددة لا تنتمي لهذا المشروع')->withInput();
            }
        }

        // التحقق من أن المورد ينتمي لنفس الشركة (إذا تم تحديده)
        if ($request->supplier_id) {
            $supplier = Supplier::findOrFail($request->supplier_id);
            if ($supplier->company_id !== Auth::user()->company_id) {
                return redirect()->back()->with('error', 'المورد المحدد لا ينتمي لشركتك')->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // حساب المبلغ الإجمالي
            $amount = $request->amount;
            $taxAmount = $request->tax_amount ?? 0;
            $totalAmount = $amount + $taxAmount;

            // معالجة ملف الإيصال إذا تم تحميله
            $receiptFilePath = null;
            if ($request->hasFile('receipt_file')) {
                $file = $request->file('receipt_file');
                $fileName = Str::uuid() . '_' . $file->getClientOriginalName();
                $receiptFilePath = 'projects/' . $installationProject->id . '/expenses/' . $fileName;
                Storage::put($receiptFilePath, file_get_contents($file));
            }

            // إنشاء المصروف
            $expense = ProjectExpense::create([
                'installation_project_id' => $installationProject->id,
                'project_phase_id' => $request->project_phase_id,
                'expense_date' => $request->expense_date,
                'amount' => $amount,
                'tax_amount' => $taxAmount,
                'total_amount' => $totalAmount,
                'expense_type' => $request->expense_type,
                'description' => $request->description,
                'supplier_id' => $request->supplier_id,
                'reference_number' => $request->reference_number,
                'payment_method' => $request->payment_method,
                'receipt_number' => $request->receipt_number,
                'receipt_file' => $receiptFilePath,
                'status' => 'pending',
                'created_by' => Auth::id(),
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('success', 'تم إضافة المصروف بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة المصروف: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل المصروف
     */
    public function show(InstallationProject $installationProject, ProjectExpense $expense)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصروف ينتمي للمشروع
        if ($expense->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('error', 'المصروف المطلوب غير موجود في هذا المشروع');
        }

        $expense->load(['phase', 'supplier', 'approvedBy', 'createdBy']);

        return view('installation-projects.expenses.show', compact('installationProject', 'expense'));
    }

    /**
     * عرض نموذج تعديل المصروف
     */
    public function edit(InstallationProject $installationProject, ProjectExpense $expense)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل المصروف');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصروف ينتمي للمشروع
        if ($expense->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('error', 'المصروف المطلوب غير موجود في هذا المشروع');
        }

        // لا يمكن تعديل المصروفات المعتمدة
        if ($expense->isApproved()) {
            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('error', 'لا يمكن تعديل المصروفات المعتمدة');
        }

        $phases = $installationProject->phases;
        $suppliers = Supplier::where('company_id', Auth::user()->company_id)->get();

        return view('installation-projects.expenses.edit', compact('installationProject', 'expense', 'phases', 'suppliers'));
    }

    /**
     * تحديث المصروف
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectExpense $expense)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل المصروف');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصروف ينتمي للمشروع
        if ($expense->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('error', 'المصروف المطلوب غير موجود في هذا المشروع');
        }

        // لا يمكن تعديل المصروفات المعتمدة
        if ($expense->isApproved()) {
            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('error', 'لا يمكن تعديل المصروفات المعتمدة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'expense_date' => 'required|date',
            'amount' => 'required|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'expense_type' => 'required|string|in:material,labor,equipment,transportation,accommodation,food,permit,insurance,other',
            'description' => 'required|string',
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'reference_number' => 'nullable|string|max:100',
            'payment_method' => 'required|string|in:cash,bank_transfer,check,credit_card,company_account,other',
            'receipt_number' => 'nullable|string|max:100',
            'receipt_file' => 'nullable|file|max:5120', // 5MB كحد أقصى
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المرحلة تنتمي للمشروع (إذا تم تحديدها)
        if ($request->project_phase_id) {
            $phase = ProjectPhase::findOrFail($request->project_phase_id);
            if ($phase->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'المرحلة المحددة لا تنتمي لهذا المشروع')->withInput();
            }
        }

        // التحقق من أن المورد ينتمي لنفس الشركة (إذا تم تحديده)
        if ($request->supplier_id) {
            $supplier = Supplier::findOrFail($request->supplier_id);
            if ($supplier->company_id !== Auth::user()->company_id) {
                return redirect()->back()->with('error', 'المورد المحدد لا ينتمي لشركتك')->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // حساب المبلغ الإجمالي
            $amount = $request->amount;
            $taxAmount = $request->tax_amount ?? 0;
            $totalAmount = $amount + $taxAmount;

            // تحديث بيانات المصروف
            $updateData = [
                'project_phase_id' => $request->project_phase_id,
                'expense_date' => $request->expense_date,
                'amount' => $amount,
                'tax_amount' => $taxAmount,
                'total_amount' => $totalAmount,
                'expense_type' => $request->expense_type,
                'description' => $request->description,
                'supplier_id' => $request->supplier_id,
                'reference_number' => $request->reference_number,
                'payment_method' => $request->payment_method,
                'receipt_number' => $request->receipt_number,
                'notes' => $request->notes,
            ];

            // معالجة ملف الإيصال إذا تم تحميله
            if ($request->hasFile('receipt_file')) {
                // حذف الملف القديم إذا وجد
                if ($expense->receipt_file) {
                    Storage::delete($expense->receipt_file);
                }

                $file = $request->file('receipt_file');
                $fileName = Str::uuid() . '_' . $file->getClientOriginalName();
                $receiptFilePath = 'projects/' . $installationProject->id . '/expenses/' . $fileName;
                Storage::put($receiptFilePath, file_get_contents($file));

                $updateData['receipt_file'] = $receiptFilePath;
            }

            // تحديث المصروف
            $expense->update($updateData);

            DB::commit();

            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('success', 'تم تحديث المصروف بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث المصروف: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف المصروف
     */
    public function destroy(InstallationProject $installationProject, ProjectExpense $expense)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لحذف المصروف');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصروف ينتمي للمشروع
        if ($expense->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('error', 'المصروف المطلوب غير موجود في هذا المشروع');
        }

        // لا يمكن حذف المصروفات المعتمدة
        if ($expense->isApproved()) {
            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('error', 'لا يمكن حذف المصروفات المعتمدة');
        }

        try {
            DB::beginTransaction();

            // حذف ملف الإيصال إذا وجد
            if ($expense->receipt_file) {
                Storage::delete($expense->receipt_file);
            }

            // حذف المصروف
            $expense->delete();

            DB::commit();

            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('success', 'تم حذف المصروف بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف المصروف: ' . $e->getMessage());
        }
    }

    /**
     * اعتماد المصروف
     */
    public function approve(Request $request, InstallationProject $installationProject, ProjectExpense $expense)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('approve_expenses')) {
            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('error', 'ليس لديك صلاحية لاعتماد المصروفات');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصروف ينتمي للمشروع
        if ($expense->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('error', 'المصروف المطلوب غير موجود في هذا المشروع');
        }

        // لا يمكن اعتماد المصروفات المعتمدة مسبقاً
        if ($expense->isApproved()) {
            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('error', 'المصروف معتمد بالفعل');
        }

        try {
            DB::beginTransaction();

            // اعتماد المصروف
            $expense->approve(Auth::id());

            DB::commit();

            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('success', 'تم اعتماد المصروف بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء اعتماد المصروف: ' . $e->getMessage());
        }
    }

    /**
     * رفض المصروف
     */
    public function reject(Request $request, InstallationProject $installationProject, ProjectExpense $expense)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('approve_expenses')) {
            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('error', 'ليس لديك صلاحية لرفض المصروفات');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصروف ينتمي للمشروع
        if ($expense->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('error', 'المصروف المطلوب غير موجود في هذا المشروع');
        }

        // لا يمكن رفض المصروفات المعتمدة
        if ($expense->isApproved()) {
            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('error', 'لا يمكن رفض المصروفات المعتمدة');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'rejection_notes' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // رفض المصروف
            $expense->reject(Auth::id(), $request->rejection_notes);

            DB::commit();

            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('success', 'تم رفض المصروف بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء رفض المصروف: ' . $e->getMessage());
        }
    }

    /**
     * تنزيل ملف الإيصال
     */
    public function downloadReceipt(InstallationProject $installationProject, ProjectExpense $expense)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصروف ينتمي للمشروع
        if ($expense->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.expenses.index', $installationProject->id)
                ->with('error', 'المصروف المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من وجود ملف الإيصال
        if (!$expense->receipt_file || !Storage::exists($expense->receipt_file)) {
            return redirect()->route('installation-projects.expenses.show', [$installationProject->id, $expense->id])
                ->with('error', 'ملف الإيصال غير موجود');
        }

        // تنزيل الملف
        $fileName = pathinfo($expense->receipt_file, PATHINFO_BASENAME);
        return Storage::download($expense->receipt_file, $fileName);
    }
}
