<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectElevator;
use App\Models\Elevator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProjectElevatorController extends Controller
{
    /**
     * عرض قائمة مصاعد المشروع
     */
    public function index(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $elevators = $installationProject->elevators()->paginate(10);

        return view('installation-projects.elevators.index', compact('installationProject', 'elevators'));
    }

    /**
     * عرض نموذج إضافة مصعد جديد للمشروع
     */
    public function create(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة مصعد للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        return view('installation-projects.elevators.create', compact('installationProject'));
    }

    /**
     * حفظ مصعد جديد للمشروع
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة مصعد للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'elevator_type' => 'required|string|in:passenger,freight,hospital,panoramic,residential,commercial,other',
            'model' => 'required|string|max:255',
            'manufacturer' => 'required|string|max:255',
            'capacity' => 'required|integer|min:1',
            'floors_count' => 'required|integer|min:2',
            'speed' => 'required|numeric|min:0.1',
            'unit_price' => 'required|numeric|min:0',
            'status' => 'required|string|in:pending,ordered,delivered,installed,tested,completed',
            'expected_delivery_date' => 'nullable|date',
            'actual_delivery_date' => 'nullable|date',
            'expected_installation_date' => 'nullable|date',
            'actual_installation_date' => 'nullable|date',
            'specifications' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // إنشاء مصعد جديد للمشروع
            $projectElevator = ProjectElevator::create([
                'installation_project_id' => $installationProject->id,
                'elevator_type' => $request->elevator_type,
                'model' => $request->model,
                'manufacturer' => $request->manufacturer,
                'capacity' => $request->capacity,
                'floors_count' => $request->floors_count,
                'speed' => $request->speed,
                'unit_price' => $request->unit_price,
                'status' => $request->status,
                'expected_delivery_date' => $request->expected_delivery_date,
                'actual_delivery_date' => $request->actual_delivery_date,
                'expected_installation_date' => $request->expected_installation_date,
                'actual_installation_date' => $request->actual_installation_date,
                'specifications' => $request->specifications,
                'notes' => $request->notes,
            ]);

            // إذا كان المصعد مكتمل، قم بإنشاء سجل مصعد في النظام
            if ($request->status === 'completed' && $request->actual_installation_date) {
                $projectElevator->createElevatorRecord();
            }

            // تحديث إجمالي تكلفة المشروع
            $totalElevatorsCost = $installationProject->elevators()->sum('unit_price');
            $installationProject->update([
                'total_cost' => $totalElevatorsCost + $installationProject->expenses()->sum('amount')
            ]);

            DB::commit();

            return redirect()->route('installation-projects.elevators.index', $installationProject->id)
                ->with('success', 'تم إضافة المصعد للمشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة المصعد: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل مصعد المشروع
     */
    public function show(InstallationProject $installationProject, ProjectElevator $projectElevator)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصعد ينتمي للمشروع
        if ($projectElevator->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.elevators.index', $installationProject->id)
                ->with('error', 'المصعد المطلوب غير موجود في هذا المشروع');
        }

        // الحصول على قوائم الفحص المرتبطة بهذا المصعد
        $checklists = $projectElevator->checklists()->with('items')->get();

        return view('installation-projects.elevators.show', compact('installationProject', 'projectElevator', 'checklists'));
    }

    /**
     * عرض نموذج تعديل مصعد المشروع
     */
    public function edit(InstallationProject $installationProject, ProjectElevator $projectElevator)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل مصعد المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصعد ينتمي للمشروع
        if ($projectElevator->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.elevators.index', $installationProject->id)
                ->with('error', 'المصعد المطلوب غير موجود في هذا المشروع');
        }

        return view('installation-projects.elevators.edit', compact('installationProject', 'projectElevator'));
    }

    /**
     * تحديث مصعد المشروع
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectElevator $projectElevator)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل مصعد المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصعد ينتمي للمشروع
        if ($projectElevator->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.elevators.index', $installationProject->id)
                ->with('error', 'المصعد المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'elevator_type' => 'required|string|in:passenger,freight,hospital,panoramic,residential,commercial,other',
            'model' => 'required|string|max:255',
            'manufacturer' => 'required|string|max:255',
            'capacity' => 'required|integer|min:1',
            'floors_count' => 'required|integer|min:2',
            'speed' => 'required|numeric|min:0.1',
            'unit_price' => 'required|numeric|min:0',
            'status' => 'required|string|in:pending,ordered,delivered,installed,tested,completed',
            'expected_delivery_date' => 'nullable|date',
            'actual_delivery_date' => 'nullable|date',
            'expected_installation_date' => 'nullable|date',
            'actual_installation_date' => 'nullable|date',
            'specifications' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث مصعد المشروع
            $projectElevator->update([
                'elevator_type' => $request->elevator_type,
                'model' => $request->model,
                'manufacturer' => $request->manufacturer,
                'capacity' => $request->capacity,
                'floors_count' => $request->floors_count,
                'speed' => $request->speed,
                'unit_price' => $request->unit_price,
                'status' => $request->status,
                'expected_delivery_date' => $request->expected_delivery_date,
                'actual_delivery_date' => $request->actual_delivery_date,
                'expected_installation_date' => $request->expected_installation_date,
                'actual_installation_date' => $request->actual_installation_date,
                'specifications' => $request->specifications,
                'notes' => $request->notes,
            ]);

            // إذا كان المصعد مكتمل، قم بإنشاء سجل مصعد في النظام إذا لم يكن موجوداً بالفعل
            if ($request->status === 'completed' && $request->actual_installation_date && !$projectElevator->elevator_id) {
                $projectElevator->createElevatorRecord();
            }

            // تحديث إجمالي تكلفة المشروع
            $totalElevatorsCost = $installationProject->elevators()->sum('unit_price');
            $installationProject->update([
                'total_cost' => $totalElevatorsCost + $installationProject->expenses()->sum('amount')
            ]);

            DB::commit();

            return redirect()->route('installation-projects.elevators.show', [$installationProject->id, $projectElevator->id])
                ->with('success', 'تم تحديث المصعد بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث المصعد: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف مصعد المشروع
     */
    public function destroy(InstallationProject $installationProject, ProjectElevator $projectElevator)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لحذف مصعد المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصعد ينتمي للمشروع
        if ($projectElevator->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.elevators.index', $installationProject->id)
                ->with('error', 'المصعد المطلوب غير موجود في هذا المشروع');
        }

        try {
            DB::beginTransaction();

            // حذف المصعد
            $projectElevator->delete();

            // تحديث إجمالي تكلفة المشروع
            $totalElevatorsCost = $installationProject->elevators()->sum('unit_price');
            $installationProject->update([
                'total_cost' => $totalElevatorsCost + $installationProject->expenses()->sum('amount')
            ]);

            DB::commit();

            return redirect()->route('installation-projects.elevators.index', $installationProject->id)
                ->with('success', 'تم حذف المصعد بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف المصعد: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة المصعد
     */
    public function updateStatus(Request $request, InstallationProject $installationProject, ProjectElevator $projectElevator)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتحديث حالة المصعد');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المصعد ينتمي للمشروع
        if ($projectElevator->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.elevators.index', $installationProject->id)
                ->with('error', 'المصعد المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'status' => 'required|string|in:pending,ordered,delivered,installed,tested,completed',
            'actual_delivery_date' => 'nullable|date|required_if:status,delivered,installed,tested,completed',
            'actual_installation_date' => 'nullable|date|required_if:status,installed,tested,completed',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث حالة المصعد
            $projectElevator->update([
                'status' => $request->status,
                'actual_delivery_date' => $request->actual_delivery_date,
                'actual_installation_date' => $request->actual_installation_date,
                'notes' => $request->has('notes') ? $projectElevator->notes . "\n" . now()->format('Y-m-d') . ": " . $request->notes : $projectElevator->notes,
            ]);

            // إذا كان المصعد مكتمل، قم بإنشاء سجل مصعد في النظام إذا لم يكن موجوداً بالفعل
            if ($request->status === 'completed' && $request->actual_installation_date && !$projectElevator->elevator_id) {
                $projectElevator->createElevatorRecord();
            }

            DB::commit();

            return redirect()->route('installation-projects.elevators.show', [$installationProject->id, $projectElevator->id])
                ->with('success', 'تم تحديث حالة المصعد بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث حالة المصعد: ' . $e->getMessage())->withInput();
        }
    }
}
