<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectDocument;
use App\Models\ProjectPhase;
use App\Models\ProjectElevator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ProjectDocumentController extends Controller
{
    /**
     * عرض قائمة مستندات المشروع
     */
    public function index(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $documents = $installationProject->documents()->with(['phase', 'elevator', 'uploader'])->get();

        // تجميع المستندات حسب النوع
        $documentsByType = $documents->groupBy('document_type');

        return view('installation-projects.documents.index', compact('installationProject', 'documents', 'documentsByType'));
    }

    /**
     * عرض نموذج رفع مستند جديد
     */
    public function create(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لرفع مستند للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $phases = $installationProject->phases;
        $elevators = $installationProject->elevators;

        return view('installation-projects.documents.create', compact('installationProject', 'phases', 'elevators'));
    }

    /**
     * حفظ مستند جديد
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لرفع مستند للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'document_file' => 'required|file|max:20480', // 20MB كحد أقصى
            'document_type' => 'required|string|in:contract,drawing,specification,certificate,report,invoice,photo,other',
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'project_elevator_id' => 'nullable|exists:project_elevators,id',
            'status' => 'required|string|in:draft,pending,approved,rejected,archived',
            'version' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المرحلة تنتمي للمشروع (إذا تم تحديدها)
        if ($request->project_phase_id) {
            $phase = ProjectPhase::findOrFail($request->project_phase_id);
            if ($phase->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'المرحلة المحددة لا تنتمي لهذا المشروع')->withInput();
            }
        }

        // التحقق من أن المصعد ينتمي للمشروع (إذا تم تحديده)
        if ($request->project_elevator_id) {
            $elevator = ProjectElevator::findOrFail($request->project_elevator_id);
            if ($elevator->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'المصعد المحدد لا ينتمي لهذا المشروع')->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // رفع الملف
            $file = $request->file('document_file');
            $originalFileName = $file->getClientOriginalName();
            $fileSize = $file->getSize();
            $fileType = $file->getMimeType();

            // إنشاء اسم فريد للملف
            $fileName = Str::uuid() . '_' . $originalFileName;

            // تحديد مسار التخزين
            $filePath = 'projects/' . $installationProject->id . '/documents/' . $fileName;

            // تخزين الملف
            Storage::put($filePath, file_get_contents($file));

            // إنشاء سجل المستند
            $document = ProjectDocument::create([
                'installation_project_id' => $installationProject->id,
                'project_phase_id' => $request->project_phase_id,
                'project_elevator_id' => $request->project_elevator_id,
                'title' => $request->title,
                'description' => $request->description,
                'file_path' => $filePath,
                'file_name' => $originalFileName,
                'file_size' => $fileSize,
                'file_type' => $fileType,
                'document_type' => $request->document_type,
                'uploaded_by' => Auth::id(),
                'version' => $request->version,
                'status' => $request->status,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.documents.index', $installationProject->id)
                ->with('success', 'تم رفع المستند بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء رفع المستند: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل المستند
     */
    public function show(InstallationProject $installationProject, ProjectDocument $document)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المستند ينتمي للمشروع
        if ($document->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.documents.index', $installationProject->id)
                ->with('error', 'المستند المطلوب غير موجود في هذا المشروع');
        }

        $document->load(['phase', 'elevator', 'uploader']);

        return view('installation-projects.documents.show', compact('installationProject', 'document'));
    }

    /**
     * عرض نموذج تعديل المستند
     */
    public function edit(InstallationProject $installationProject, ProjectDocument $document)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل المستند');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المستند ينتمي للمشروع
        if ($document->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.documents.index', $installationProject->id)
                ->with('error', 'المستند المطلوب غير موجود في هذا المشروع');
        }

        $phases = $installationProject->phases;
        $elevators = $installationProject->elevators;

        return view('installation-projects.documents.edit', compact('installationProject', 'document', 'phases', 'elevators'));
    }

    /**
     * تحديث المستند
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectDocument $document)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل المستند');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المستند ينتمي للمشروع
        if ($document->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.documents.index', $installationProject->id)
                ->with('error', 'المستند المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'document_file' => 'nullable|file|max:20480', // 20MB كحد أقصى
            'document_type' => 'required|string|in:contract,drawing,specification,certificate,report,invoice,photo,other',
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'project_elevator_id' => 'nullable|exists:project_elevators,id',
            'status' => 'required|string|in:draft,pending,approved,rejected,archived',
            'version' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // التحقق من أن المرحلة تنتمي للمشروع (إذا تم تحديدها)
        if ($request->project_phase_id) {
            $phase = ProjectPhase::findOrFail($request->project_phase_id);
            if ($phase->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'المرحلة المحددة لا تنتمي لهذا المشروع')->withInput();
            }
        }

        // التحقق من أن المصعد ينتمي للمشروع (إذا تم تحديده)
        if ($request->project_elevator_id) {
            $elevator = ProjectElevator::findOrFail($request->project_elevator_id);
            if ($elevator->installation_project_id !== $installationProject->id) {
                return redirect()->back()->with('error', 'المصعد المحدد لا ينتمي لهذا المشروع')->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // تحديث بيانات المستند
            $updateData = [
                'title' => $request->title,
                'description' => $request->description,
                'document_type' => $request->document_type,
                'project_phase_id' => $request->project_phase_id,
                'project_elevator_id' => $request->project_elevator_id,
                'status' => $request->status,
                'version' => $request->version,
                'notes' => $request->notes,
            ];

            // إذا تم تحديث الملف
            if ($request->hasFile('document_file')) {
                // حذف الملف القديم
                Storage::delete($document->file_path);

                // رفع الملف الجديد
                $file = $request->file('document_file');
                $originalFileName = $file->getClientOriginalName();
                $fileSize = $file->getSize();
                $fileType = $file->getMimeType();

                // إنشاء اسم فريد للملف
                $fileName = Str::uuid() . '_' . $originalFileName;

                // تحديد مسار التخزين
                $filePath = 'projects/' . $installationProject->id . '/documents/' . $fileName;

                // تخزين الملف
                Storage::put($filePath, file_get_contents($file));

                // تحديث بيانات الملف
                $updateData['file_path'] = $filePath;
                $updateData['file_name'] = $originalFileName;
                $updateData['file_size'] = $fileSize;
                $updateData['file_type'] = $fileType;
            }

            // تحديث المستند
            $document->update($updateData);

            DB::commit();

            return redirect()->route('installation-projects.documents.show', [$installationProject->id, $document->id])
                ->with('success', 'تم تحديث المستند بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث المستند: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف المستند
     */
    public function destroy(InstallationProject $installationProject, ProjectDocument $document)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لحذف المستند');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المستند ينتمي للمشروع
        if ($document->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.documents.index', $installationProject->id)
                ->with('error', 'المستند المطلوب غير موجود في هذا المشروع');
        }

        try {
            DB::beginTransaction();

            // حذف الملف من التخزين
            Storage::delete($document->file_path);

            // حذف سجل المستند
            $document->delete();

            DB::commit();

            return redirect()->route('installation-projects.documents.index', $installationProject->id)
                ->with('success', 'تم حذف المستند بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف المستند: ' . $e->getMessage());
        }
    }

    /**
     * تنزيل المستند
     */
    public function download(InstallationProject $installationProject, ProjectDocument $document)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المستند ينتمي للمشروع
        if ($document->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.documents.index', $installationProject->id)
                ->with('error', 'المستند المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من وجود الملف
        if (!Storage::exists($document->file_path)) {
            return redirect()->route('installation-projects.documents.show', [$installationProject->id, $document->id])
                ->with('error', 'الملف غير موجود');
        }

        // تنزيل الملف
        return Storage::download($document->file_path, $document->file_name);
    }

    /**
     * عرض المستند (للصور وملفات PDF)
     */
    public function preview(InstallationProject $installationProject, ProjectDocument $document)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن المستند ينتمي للمشروع
        if ($document->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.documents.index', $installationProject->id)
                ->with('error', 'المستند المطلوب غير موجود في هذا المشروع');
        }

        // التحقق من وجود الملف
        if (!Storage::exists($document->file_path)) {
            return redirect()->route('installation-projects.documents.show', [$installationProject->id, $document->id])
                ->with('error', 'الملف غير موجود');
        }

        // التحقق من نوع الملف (صورة أو PDF)
        if (!$document->isImage() && !$document->isPdf()) {
            return redirect()->route('installation-projects.documents.show', [$installationProject->id, $document->id])
                ->with('error', 'لا يمكن معاينة هذا النوع من الملفات');
        }

        return view('installation-projects.documents.preview', compact('installationProject', 'document'));
    }
}
