<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectChecklist;
use App\Models\ProjectChecklistItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProjectChecklistItemController extends Controller
{
    /**
     * إضافة عنصر جديد لقائمة التحقق
     */
    public function store(Request $request, InstallationProject $installationProject, ProjectChecklist $checklist)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية لإضافة عنصر لقائمة التحقق']);
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية للوصول إلى هذا المشروع']);
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return response()->json(['success' => false, 'message' => 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع']);
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'description' => 'required|string|max:255',
            'order_number' => 'required|integer|min:1',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()]);
        }

        try {
            DB::beginTransaction();

            // إضافة عنصر جديد
            $item = $checklist->items()->create([
                'description' => $request->description,
                'is_checked' => false,
                'order_number' => $request->order_number,
                'notes' => $request->notes,
            ]);

            // تحديث حالة قائمة التحقق
            $checklist->updateStatus();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم إضافة العنصر بنجاح',
                'item' => $item->fresh(),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'حدث خطأ أثناء إضافة العنصر: ' . $e->getMessage()]);
        }
    }

    /**
     * تحديث عنصر في قائمة التحقق
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectChecklist $checklist, ProjectChecklistItem $item)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية لتعديل عنصر قائمة التحقق']);
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية للوصول إلى هذا المشروع']);
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return response()->json(['success' => false, 'message' => 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع']);
        }

        // التحقق من أن العنصر ينتمي لقائمة التحقق
        if ($item->project_checklist_id !== $checklist->id) {
            return response()->json(['success' => false, 'message' => 'العنصر المطلوب غير موجود في قائمة التحقق هذه']);
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'description' => 'required|string|max:255',
            'order_number' => 'required|integer|min:1',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()]);
        }

        try {
            DB::beginTransaction();

            // تحديث العنصر
            $item->update([
                'description' => $request->description,
                'order_number' => $request->order_number,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم تحديث العنصر بنجاح',
                'item' => $item->fresh(),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'حدث خطأ أثناء تحديث العنصر: ' . $e->getMessage()]);
        }
    }

    /**
     * حذف عنصر من قائمة التحقق
     */
    public function destroy(InstallationProject $installationProject, ProjectChecklist $checklist, ProjectChecklistItem $item)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية لحذف عنصر قائمة التحقق']);
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية للوصول إلى هذا المشروع']);
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return response()->json(['success' => false, 'message' => 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع']);
        }

        // التحقق من أن العنصر ينتمي لقائمة التحقق
        if ($item->project_checklist_id !== $checklist->id) {
            return response()->json(['success' => false, 'message' => 'العنصر المطلوب غير موجود في قائمة التحقق هذه']);
        }

        try {
            DB::beginTransaction();

            // حذف العنصر
            $item->delete();

            // تحديث حالة قائمة التحقق
            $checklist->updateStatus();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم حذف العنصر بنجاح',
                'checklist' => $checklist->fresh(['completedBy']),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'حدث خطأ أثناء حذف العنصر: ' . $e->getMessage()]);
        }
    }

    /**
     * تحديث حالة العنصر (مكتمل/غير مكتمل)
     */
    public function toggleStatus(Request $request, InstallationProject $installationProject, ProjectChecklist $checklist, ProjectChecklistItem $item)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية لتحديث حالة العنصر']);
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية للوصول إلى هذا المشروع']);
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return response()->json(['success' => false, 'message' => 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع']);
        }

        // التحقق من أن العنصر ينتمي لقائمة التحقق
        if ($item->project_checklist_id !== $checklist->id) {
            return response()->json(['success' => false, 'message' => 'العنصر المطلوب غير موجود في قائمة التحقق هذه']);
        }

        try {
            DB::beginTransaction();

            // تبديل حالة العنصر
            if ($item->is_checked) {
                $item->uncheck();
            } else {
                $item->check();
            }

            // تحديث الملاحظات إذا وجدت
            if ($request->has('notes')) {
                $item->update(['notes' => $request->notes]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم تحديث حالة العنصر بنجاح',
                'item' => $item->fresh(['checkedByUser']),
                'checklist' => $checklist->fresh(['completedBy']),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'حدث خطأ أثناء تحديث حالة العنصر: ' . $e->getMessage()]);
        }
    }
}
