<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\ProjectChecklist;
use App\Models\ProjectPhase;
use App\Models\ProjectElevator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProjectChecklistController extends Controller
{
    /**
     * عرض قائمة قوائم التحقق للمشروع
     */
    public function index(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $checklists = $installationProject->checklists()->with(['phase', 'elevator', 'creator', 'completedBy'])->get();

        return view('installation-projects.checklists.index', compact('installationProject', 'checklists'));
    }

    /**
     * عرض نموذج إنشاء قائمة تحقق جديدة
     */
    public function create(InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة قائمة تحقق للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $phases = $installationProject->phases;
        $elevators = $installationProject->elevators;

        return view('installation-projects.checklists.create', compact('installationProject', 'phases', 'elevators'));
    }

    /**
     * حفظ قائمة تحقق جديدة
     */
    public function store(Request $request, InstallationProject $installationProject)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)->with('error', 'ليس لديك صلاحية لإضافة قائمة تحقق للمشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'type' => 'required|string|in:pre_installation,installation,testing,safety,handover,other',
            'description' => 'nullable|string',
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'project_elevator_id' => 'nullable|exists:project_elevators,id',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string|max:255',
            'items.*.order_number' => 'required|integer|min:1',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // إنشاء قائمة التحقق
            $checklist = ProjectChecklist::create([
                'installation_project_id' => $installationProject->id,
                'project_phase_id' => $request->project_phase_id,
                'project_elevator_id' => $request->project_elevator_id,
                'title' => $request->title,
                'type' => $request->type,
                'description' => $request->description,
                'status' => 'not_started',
                'created_by' => Auth::id(),
                'notes' => $request->notes,
            ]);

            // إضافة عناصر قائمة التحقق
            foreach ($request->items as $item) {
                $checklist->items()->create([
                    'description' => $item['description'],
                    'is_checked' => false,
                    'order_number' => $item['order_number'],
                    'notes' => $item['notes'] ?? null,
                ]);
            }

            DB::commit();

            return redirect()->route('installation-projects.checklists.show', [$installationProject->id, $checklist->id])
                ->with('success', 'تم إنشاء قائمة التحقق بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء إنشاء قائمة التحقق: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل قائمة التحقق
     */
    public function show(InstallationProject $installationProject, ProjectChecklist $checklist)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('view_installation_projects')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.checklists.index', $installationProject->id)
                ->with('error', 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع');
        }

        $checklist->load(['phase', 'elevator', 'creator', 'completedBy', 'items.checkedByUser']);

        return view('installation-projects.checklists.show', compact('installationProject', 'checklist'));
    }

    /**
     * عرض نموذج تعديل قائمة التحقق
     */
    public function edit(InstallationProject $installationProject, ProjectChecklist $checklist)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل قائمة التحقق');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.checklists.index', $installationProject->id)
                ->with('error', 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع');
        }

        $phases = $installationProject->phases;
        $elevators = $installationProject->elevators;
        $items = $checklist->items()->orderBy('order_number')->get();

        return view('installation-projects.checklists.edit', compact('installationProject', 'checklist', 'phases', 'elevators', 'items'));
    }

    /**
     * تحديث قائمة التحقق
     */
    public function update(Request $request, InstallationProject $installationProject, ProjectChecklist $checklist)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لتعديل قائمة التحقق');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.checklists.index', $installationProject->id)
                ->with('error', 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'type' => 'required|string|in:pre_installation,installation,testing,safety,handover,other',
            'description' => 'nullable|string',
            'project_phase_id' => 'nullable|exists:project_phases,id',
            'project_elevator_id' => 'nullable|exists:project_elevators,id',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.id' => 'nullable|exists:project_checklist_items,id',
            'items.*.description' => 'required|string|max:255',
            'items.*.order_number' => 'required|integer|min:1',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث قائمة التحقق
            $checklist->update([
                'project_phase_id' => $request->project_phase_id,
                'project_elevator_id' => $request->project_elevator_id,
                'title' => $request->title,
                'type' => $request->type,
                'description' => $request->description,
                'notes' => $request->notes,
            ]);

            // تحديث عناصر قائمة التحقق
            $existingItemIds = [];

            foreach ($request->items as $itemData) {
                if (isset($itemData['id']) && $itemData['id']) {
                    // تحديث العنصر الموجود
                    $item = $checklist->items()->find($itemData['id']);
                    if ($item) {
                        $item->update([
                            'description' => $itemData['description'],
                            'order_number' => $itemData['order_number'],
                            'notes' => $itemData['notes'] ?? null,
                        ]);
                        $existingItemIds[] = $item->id;
                    }
                } else {
                    // إنشاء عنصر جديد
                    $item = $checklist->items()->create([
                        'description' => $itemData['description'],
                        'is_checked' => false,
                        'order_number' => $itemData['order_number'],
                        'notes' => $itemData['notes'] ?? null,
                    ]);
                    $existingItemIds[] = $item->id;
                }
            }

            // حذف العناصر التي لم تعد موجودة
            $checklist->items()->whereNotIn('id', $existingItemIds)->delete();

            // تحديث حالة قائمة التحقق
            $checklist->updateStatus();

            DB::commit();

            return redirect()->route('installation-projects.checklists.show', [$installationProject->id, $checklist->id])
                ->with('success', 'تم تحديث قائمة التحقق بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث قائمة التحقق: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف قائمة التحقق
     */
    public function destroy(InstallationProject $installationProject, ProjectChecklist $checklist)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('error', 'ليس لديك صلاحية لحذف قائمة التحقق');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return redirect()->route('installation-projects.checklists.index', $installationProject->id)
                ->with('error', 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع');
        }

        try {
            DB::beginTransaction();

            // حذف قائمة التحقق (سيتم حذف العناصر تلقائياً بسبب العلاقات)
            $checklist->delete();

            DB::commit();

            return redirect()->route('installation-projects.checklists.index', $installationProject->id)
                ->with('success', 'تم حذف قائمة التحقق بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف قائمة التحقق: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة عنصر في قائمة التحقق
     */
    public function updateItemStatus(Request $request, InstallationProject $installationProject, ProjectChecklist $checklist, $itemId)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->hasPermission('edit_installation_project')) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية لتحديث حالة العنصر']);
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if ($installationProject->company_id !== Auth::user()->company_id) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية للوصول إلى هذا المشروع']);
        }

        // التحقق من أن قائمة التحقق تنتمي للمشروع
        if ($checklist->installation_project_id !== $installationProject->id) {
            return response()->json(['success' => false, 'message' => 'قائمة التحقق المطلوبة غير موجودة في هذا المشروع']);
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'is_checked' => 'required|boolean',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()]);
        }

        try {
            DB::beginTransaction();

            // الحصول على العنصر
            $item = $checklist->items()->findOrFail($itemId);

            // تحديث حالة العنصر
            if ($request->is_checked) {
                $item->check();
                if ($request->notes) {
                    $item->update(['notes' => $request->notes]);
                }
            } else {
                $item->uncheck();
                if ($request->notes) {
                    $item->update(['notes' => $request->notes]);
                }
            }

            // تحديث حالة قائمة التحقق
            $checklist->updateStatus();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم تحديث حالة العنصر بنجاح',
                'item' => $item->fresh(['checkedByUser']),
                'checklist' => $checklist->fresh(['completedBy']),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'حدث خطأ أثناء تحديث حالة العنصر: ' . $e->getMessage()]);
        }
    }
}
