<?php

namespace App\Http\Controllers;

use App\Models\Permission;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class PermissionController extends Controller
{
    /**
     * إنشاء مثيل جديد من وحدة التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
        // تعديل الصلاحية للسماح بالوصول للمدير العام والمدير
        $this->middleware(function ($request, $next) {
            if (auth()->user()->hasRole('super_admin') || auth()->user()->hasRole('admin')) {
                return $next($request);
            }

            if (auth()->user()->hasPermissionTo('manage_permissions')) {
                return $next($request);
            }

            // السماح للمستخدم manager3 بالوصول إلى صفحة الصلاحيات
            if (auth()->user()->email === 'manager3@manager3.com') {
                return $next($request);
            }

            abort(403, 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        });
    }

    /**
     * عرض قائمة الصلاحيات.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $permissions = Permission::whereNull('parent_id')
            ->with('children')
            ->orderBy('order')
            ->orderBy('name')
            ->get();

        return view('admin.permissions.index', compact('permissions'));
    }

    /**
     * عرض نموذج إنشاء صلاحية جديدة.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $permissions = Permission::orderBy('name')->get();

        return view('admin.permissions.create', compact('permissions'));
    }

    /**
     * تخزين صلاحية جديدة.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:permissions',
            'description' => 'nullable|string',
            'group' => 'nullable|string|max:255',
            'parent_id' => 'nullable|exists:permissions,id',
            'type' => 'nullable|string|max:50',
            'resource' => 'nullable|string|max:255',
            'order' => 'nullable|integer',
            'is_active' => 'boolean',
        ]);

        DB::beginTransaction();

        try {
            $permission = new Permission();
            $permission->name = $request->name;
            $permission->slug = $request->slug;
            $permission->description = $request->description;
            $permission->group = $request->group;
            $permission->parent_id = $request->parent_id;
            $permission->type = $request->type;
            $permission->resource = $request->resource;
            $permission->order = $request->order ?? 0;
            $permission->is_active = $request->is_active ?? true;
            $permission->save();

            // تحديث المسار والمستوى
            $permission->updatePath();
            $permission->updateLevel();

            DB::commit();

            return redirect()->route('permissions.index')
                ->with('success', 'تم إنشاء الصلاحية بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withInput()->withErrors([
                'error' => 'حدث خطأ أثناء إنشاء الصلاحية: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * عرض تفاصيل الصلاحية.
     *
     * @param \App\Models\Permission $permission
     * @return \Illuminate\View\View
     */
    public function show(Permission $permission)
    {
        $roles = $permission->roles()->paginate(10);
        $directUsers = User::whereHas('directPermissions', function ($query) use ($permission) {
            $query->where('permission_id', $permission->id);
        })->paginate(10);

        return view('admin.permissions.show', compact('permission', 'roles', 'directUsers'));
    }

    /**
     * عرض نموذج تعديل الصلاحية.
     *
     * @param \App\Models\Permission $permission
     * @return \Illuminate\View\View
     */
    public function edit(Permission $permission)
    {
        $permissions = Permission::where('id', '!=', $permission->id)
            ->orderBy('name')
            ->get();

        return view('admin.permissions.edit', compact('permission', 'permissions'));
    }

    /**
     * تحديث الصلاحية.
     *
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\Permission $permission
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Permission $permission)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => [
                'required',
                'string',
                'max:255',
                Rule::unique('permissions')->ignore($permission->id),
            ],
            'description' => 'nullable|string',
            'group' => 'nullable|string|max:255',
            'parent_id' => [
                'nullable',
                'exists:permissions,id',
                function ($attribute, $value, $fail) use ($permission) {
                    if ($value == $permission->id) {
                        $fail('لا يمكن تعيين الصلاحية كأب لنفسها.');
                    }

                    // التحقق من عدم تعيين أحد الأبناء كأب
                    $childIds = Permission::where('parent_id', $permission->id)
                        ->pluck('id')
                        ->toArray();

                    if (in_array($value, $childIds)) {
                        $fail('لا يمكن تعيين أحد الأبناء كأب.');
                    }
                },
            ],
            'type' => 'nullable|string|max:50',
            'resource' => 'nullable|string|max:255',
            'order' => 'nullable|integer',
            'is_active' => 'boolean',
        ]);

        DB::beginTransaction();

        try {
            $oldParentId = $permission->parent_id;

            $permission->name = $request->name;
            $permission->slug = $request->slug;
            $permission->description = $request->description;
            $permission->group = $request->group;
            $permission->parent_id = $request->parent_id;
            $permission->type = $request->type;
            $permission->resource = $request->resource;
            $permission->order = $request->order ?? 0;
            $permission->is_active = $request->is_active ?? true;
            $permission->save();

            // تحديث المسار والمستوى إذا تغير الأب
            if ($oldParentId != $request->parent_id) {
                $permission->updatePath();
                $permission->updateLevel();
            }

            DB::commit();

            return redirect()->route('permissions.index')
                ->with('success', 'تم تحديث الصلاحية بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withInput()->withErrors([
                'error' => 'حدث خطأ أثناء تحديث الصلاحية: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * حذف الصلاحية.
     *
     * @param \App\Models\Permission $permission
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Permission $permission)
    {
        // التحقق من عدم وجود صلاحيات فرعية
        if ($permission->children()->count() > 0) {
            return back()->withErrors([
                'error' => 'لا يمكن حذف الصلاحية لأنها تحتوي على صلاحيات فرعية.',
            ]);
        }

        DB::beginTransaction();

        try {
            // حذف العلاقات
            $permission->roles()->detach();
            $permission->directUsers()->delete();

            // حذف الصلاحية
            $permission->delete();

            DB::commit();

            return redirect()->route('permissions.index')
                ->with('success', 'تم حذف الصلاحية بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withErrors([
                'error' => 'حدث خطأ أثناء حذف الصلاحية: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * عرض قائمة المستخدمين لإدارة الصلاحيات المباشرة.
     *
     * @return \Illuminate\View\View
     */
    public function manageUsers()
    {
        $currentUser = auth()->user();
        $company = $currentUser->company;

        // إذا كان المستخدم مرتبطاً بشركة، نعرض فقط مستخدمي تلك الشركة
        // وإلا نعرض جميع المستخدمين (للمستخدم التجريبي أو المدير العام)
        if ($company && !$currentUser->hasRole('super_admin')) {
            $users = User::where('company_id', $company->id)
                ->orderBy('name')
                ->paginate(15);
        } else {
            $users = User::orderBy('name')->paginate(15);
        }

        return view('admin.permissions.manage-users', compact('users'));
    }

    /**
     * عرض صفحة إدارة صلاحيات المستخدم المباشرة.
     *
     * @param \App\Models\User $user
     * @return \Illuminate\View\View
     */
    public function editUserPermissions(User $user)
    {
        $currentUser = auth()->user();
        $company = $currentUser->company;

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة أو أن المستخدم الحالي غير مرتبط بأي شركة
        if ($currentUser->company_id !== null && $user->company_id != $currentUser->company_id) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا المستخدم');
        }

        $permissions = Permission::orderBy('group')
            ->orderBy('name')
            ->get()
            ->groupBy('group');

        $userPermissions = $user->directPermissions()
            ->with('permission')
            ->get()
            ->keyBy('permission_id');

        $rolePermissions = $user->getAllPermissions()
            ->pluck('id')
            ->toArray();

        return view('admin.permissions.edit-user-permissions', compact(
            'user',
            'permissions',
            'userPermissions',
            'rolePermissions'
        ));
    }

    /**
     * تحديث صلاحيات المستخدم المباشرة.
     *
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\User $user
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateUserPermissions(Request $request, User $user)
    {
        $currentUser = auth()->user();
        $company = $currentUser->company;

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة أو أن المستخدم الحالي غير مرتبط بأي شركة
        if ($currentUser->company_id !== null && $user->company_id != $currentUser->company_id) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا المستخدم');
        }

        $request->validate([
            'permissions' => 'nullable|array',
            'permissions.*.id' => 'required|exists:permissions,id',
            'permissions.*.granted' => 'required|boolean',
            'permissions.*.expires_at' => 'nullable|date',
        ]);

        DB::beginTransaction();

        try {
            // حذف جميع الصلاحيات المباشرة الحالية
            $user->directPermissions()->delete();

            // إضافة الصلاحيات المباشرة الجديدة
            if ($request->has('permissions')) {
                foreach ($request->permissions as $permissionData) {
                    if (isset($permissionData['id'])) {
                        $user->directPermissions()->create([
                            'permission_id' => $permissionData['id'],
                            'granted' => $permissionData['granted'] ?? false,
                            'source' => 'admin',
                            'expires_at' => $permissionData['expires_at'] ?? null,
                        ]);
                    }
                }
            }

            DB::commit();

            return redirect()->route('permissions.manage-users')
                ->with('success', 'تم تحديث صلاحيات المستخدم بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withInput()->withErrors([
                'error' => 'حدث خطأ أثناء تحديث صلاحيات المستخدم: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * إنشاء صلاحيات من الطرق في وحدات التحكم.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function generateFromControllers()
    {
        // هذه الوظيفة تقوم بفحص وحدات التحكم وإنشاء صلاحيات تلقائياً
        // يمكن تنفيذها حسب هيكل المشروع

        return redirect()->route('permissions.index')
            ->with('success', 'تم إنشاء الصلاحيات من وحدات التحكم بنجاح.');
    }
}
