<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Employee;
use App\Models\Payroll;
use App\Models\PayrollItem;
use App\Models\SalaryPolicy;
use App\Models\TaxBracket;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PayrollController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->input('company_id', Auth::user()->company_id);

        $payrolls = Payroll::where('company_id', $companyId)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('hrm.payroll.index', compact('payrolls'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $companies = Company::all();
        $salaryPolicies = SalaryPolicy::all();

        // Add month and year data for dropdowns
        $months = [
            '01' => 'يناير',
            '02' => 'فبراير',
            '03' => 'مارس',
            '04' => 'أبريل',
            '05' => 'مايو',
            '06' => 'يونيو',
            '07' => 'يوليو',
            '08' => 'أغسطس',
            '09' => 'سبتمبر',
            '10' => 'أكتوبر',
            '11' => 'نوفمبر',
            '12' => 'ديسمبر'
        ];

        // Generate years (current year and 5 years back)
        $currentYear = date('Y');
        $years = [];
        for ($i = 0; $i <= 5; $i++) {
            $year = $currentYear - $i;
            $years[$year] = $year;
        }

        // Set current month
        $currentMonth = date('m');

        return view('hrm.payroll.create', compact('companies', 'salaryPolicies', 'months', 'years', 'currentMonth', 'currentYear'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'company_id' => 'required|exists:companies,id',
            'payroll_period' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'payment_date' => 'required|date|after_or_equal:end_date',
        ]);

        try {
            DB::beginTransaction();

            // Create payroll record
            $payroll = new Payroll();
            $payroll->company_id = $request->company_id;
            $payroll->payroll_period = $request->payroll_period;
            $payroll->start_date = $request->start_date;
            $payroll->end_date = $request->end_date;
            $payroll->payment_date = $request->payment_date;
            $payroll->status = 'draft';
            $payroll->created_by = Auth::id();
            $payroll->save();

            // Get employees for this company
            $employees = Employee::where('company_id', $request->company_id)
                ->where('status', 'active')
                ->get();

            // Get salary policy
            $salaryPolicy = SalaryPolicy::where('company_id', $request->company_id)->first();

            // Get tax brackets
            $taxBrackets = TaxBracket::where('company_id', $request->company_id)
                ->orderBy('min_amount')
                ->get();

            // Initialize totals
            $totalBasicSalary = 0;
            $totalAllowances = 0;
            $totalDeductions = 0;
            $totalNetSalary = 0;
            $totalTax = 0;
            $totalSocialInsurance = 0;
            $totalOvertime = 0;

            // Create payroll items for each employee
            foreach ($employees as $employee) {
                // Calculate salary components
                $basicSalary = $employee->basic_salary;
                $allowances = [
                    'housing' => $employee->housing_allowance,
                    'transportation' => $employee->transportation_allowance,
                    'other' => $employee->other_allowances
                ];

                $grossSalary = $basicSalary + $employee->housing_allowance +
                    $employee->transportation_allowance + $employee->other_allowances;

                // Calculate tax
                $taxAmount = 0;
                if ($taxBrackets->count() > 0) {
                    foreach ($taxBrackets as $bracket) {
                        $taxAmount += $bracket->calculateTax($grossSalary);
                    }
                }

                // Calculate social insurance
                $socialInsurance = 0;
                if ($salaryPolicy) {
                    $socialInsurance = $grossSalary * ($salaryPolicy->social_insurance_percentage / 100);
                }

                // Calculate net salary
                $totalDeduction = $taxAmount + $socialInsurance;
                $netSalary = $grossSalary - $totalDeduction;

                // Create payroll item
                $payrollItem = new PayrollItem();
                $payrollItem->company_id = $request->company_id;
                $payrollItem->payroll_id = $payroll->id;
                $payrollItem->employee_id = $employee->id;
                $payrollItem->basic_salary = $basicSalary;
                $payrollItem->allowances = json_encode($allowances);
                $payrollItem->gross_salary = $grossSalary;
                $payrollItem->tax_amount = $taxAmount;
                $payrollItem->social_insurance = $socialInsurance;
                $payrollItem->net_salary = $netSalary;
                $payrollItem->payment_status = 'pending';
                $payrollItem->created_by = Auth::id();
                $payrollItem->save();

                // Add to totals
                $totalBasicSalary += $basicSalary;
                $totalAllowances += ($employee->housing_allowance + $employee->transportation_allowance + $employee->other_allowances);
                $totalDeductions += $totalDeduction;
                $totalNetSalary += $netSalary;
                $totalTax += $taxAmount;
                $totalSocialInsurance += $socialInsurance;
            }

            // Update payroll with totals
            $payroll->total_employees = $employees->count();
            $payroll->total_basic_salary = $totalBasicSalary;
            $payroll->total_allowances = $totalAllowances;
            $payroll->total_deductions = $totalDeductions;
            $payroll->total_net_salary = $totalNetSalary;
            $payroll->total_tax = $totalTax;
            $payroll->total_social_insurance = $totalSocialInsurance;
            $payroll->total_overtime = $totalOvertime;
            $payroll->save();

            DB::commit();

            return redirect()->route('hrm.payroll.show', $payroll->id)
                ->with('success', 'تم إنشاء كشف الرواتب بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ أثناء إنشاء كشف الرواتب: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $payroll = Payroll::with(['payrollItems.employee'])->findOrFail($id);

        // Check if user has access to this payroll
        if (Auth::user()->company_id != $payroll->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('hrm.payroll.index')
                ->with('error', 'ليس لديك صلاحية للوصول إلى كشف الرواتب هذا');
        }

        return view('hrm.payroll.show', compact('payroll'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if user has access to this payroll
        if (Auth::user()->company_id != $payroll->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('payroll.index')
                ->with('error', 'ليس لديك صلاحية لتعديل كشف الرواتب هذا');
        }

        // Check if payroll is in draft status
        if ($payroll->status != 'draft') {
            return redirect()->route('payroll.show', $payroll->id)
                ->with('error', 'لا يمكن تعديل كشف الرواتب بعد اعتماده');
        }

        $companies = Company::all();

        return view('hrm.payroll.edit', compact('payroll', 'companies'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'payroll_period' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'payment_date' => 'required|date|after_or_equal:end_date',
        ]);

        $payroll = Payroll::findOrFail($id);

        // Check if user has access to this payroll
        if (Auth::user()->company_id != $payroll->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('payroll.index')
                ->with('error', 'ليس لديك صلاحية لتعديل كشف الرواتب هذا');
        }

        // Check if payroll is in draft status
        if ($payroll->status != 'draft') {
            return redirect()->route('payroll.show', $payroll->id)
                ->with('error', 'لا يمكن تعديل كشف الرواتب بعد اعتماده');
        }

        try {
            $payroll->payroll_period = $request->payroll_period;
            $payroll->start_date = $request->start_date;
            $payroll->end_date = $request->end_date;
            $payroll->payment_date = $request->payment_date;
            $payroll->notes = $request->notes;
            $payroll->updated_by = Auth::id();
            $payroll->save();

            return redirect()->route('payroll.show', $payroll->id)
                ->with('success', 'تم تحديث كشف الرواتب بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء تحديث كشف الرواتب: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if user has access to this payroll
        if (Auth::user()->company_id != $payroll->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('payroll.index')
                ->with('error', 'ليس لديك صلاحية لحذف كشف الرواتب هذا');
        }

        // Check if payroll is in draft status
        if ($payroll->status != 'draft') {
            return redirect()->route('payroll.show', $payroll->id)
                ->with('error', 'لا يمكن حذف كشف الرواتب بعد اعتماده');
        }

        try {
            DB::beginTransaction();

            // Delete payroll items first
            PayrollItem::where('payroll_id', $id)->delete();

            // Delete payroll
            $payroll->delete();

            DB::commit();

            return redirect()->route('payroll.index')
                ->with('success', 'تم حذف كشف الرواتب بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ أثناء حذف كشف الرواتب: ' . $e->getMessage());
        }
    }

    /**
     * Approve the payroll.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function approve($id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if user has access to this payroll
        if (Auth::user()->company_id != $payroll->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('payroll.index')
                ->with('error', 'ليس لديك صلاحية لاعتماد كشف الرواتب هذا');
        }

        // Check if payroll is in draft status
        if ($payroll->status != 'draft') {
            return redirect()->route('payroll.show', $payroll->id)
                ->with('error', 'تم اعتماد كشف الرواتب بالفعل');
        }

        try {
            $payroll->status = 'approved';
            $payroll->approved_by = Auth::id();
            $payroll->approved_at = now();
            $payroll->save();

            return redirect()->route('payroll.show', $payroll->id)
                ->with('success', 'تم اعتماد كشف الرواتب بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء اعتماد كشف الرواتب: ' . $e->getMessage());
        }
    }

    /**
     * Process payroll payments.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function processPayments($id)
    {
        $payroll = Payroll::findOrFail($id);

        // Check if user has access to this payroll
        if (Auth::user()->company_id != $payroll->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('payroll.index')
                ->with('error', 'ليس لديك صلاحية لمعالجة مدفوعات كشف الرواتب هذا');
        }

        // Check if payroll is approved
        if ($payroll->status != 'approved') {
            return redirect()->route('payroll.show', $payroll->id)
                ->with('error', 'يجب اعتماد كشف الرواتب قبل معالجة المدفوعات');
        }

        try {
            DB::beginTransaction();

            // Update all payroll items to paid
            PayrollItem::where('payroll_id', $id)
                ->update([
                    'payment_status' => 'paid',
                    'payment_date' => now(),
                    'updated_by' => Auth::id()
                ]);

            // Update payroll status
            $payroll->status = 'paid';
            $payroll->updated_by = Auth::id();
            $payroll->save();

            DB::commit();

            return redirect()->route('payroll.show', $payroll->id)
                ->with('success', 'تم معالجة مدفوعات كشف الرواتب بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ أثناء معالجة مدفوعات كشف الرواتب: ' . $e->getMessage());
        }
    }
}
