<?php

namespace App\Http\Controllers;

use App\Models\Opportunity;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class OpportunityController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $company_id = $request->input('company_id', Auth::user()->company_id);
        $customer_id = $request->input('customer_id');
        $stage = $request->input('stage');

        $opportunities = Opportunity::with(['customer', 'user'])
            ->where('company_id', $company_id);

        if ($customer_id) {
            $opportunities->where('customer_id', $customer_id);
        }

        if ($stage) {
            $opportunities->where('stage', $stage);
        }

        $opportunities = $opportunities->orderBy('expected_close_date', 'asc')
            ->paginate(15);

        return view('opportunities.index', compact('opportunities'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $company_id = $request->input('company_id', Auth::user()->company_id);
        $customer_id = $request->input('customer_id');

        $customers = Customer::where('company_id', $company_id)->get();
        $customer = $customer_id ? Customer::findOrFail($customer_id) : null;

        return view('opportunities.create', compact('customers', 'customer', 'customer_id'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:new_sale,upgrade,maintenance,consultation,other',
            'stage' => 'required|in:lead,qualified,proposal,negotiation,closed_won,closed_lost',
            'amount' => 'nullable|numeric|min:0',
            'probability' => 'nullable|numeric|min:0|max:100',
            'expected_close_date' => 'nullable|date',
            'actual_close_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'source' => 'nullable|string|max:255',
            'loss_reason' => 'nullable|string|max:255',
        ]);

        $company_id = $request->input('company_id', Auth::user()->company_id);

        $opportunity = new Opportunity($validated);
        $opportunity->company_id = $company_id;
        $opportunity->user_id = Auth::id();
        $opportunity->save();

        return redirect()->route('opportunities.index')
            ->with('success', 'تم إنشاء الفرصة التجارية بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $opportunity = Opportunity::with(['customer', 'user'])
            ->findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($opportunity->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بعرض هذه الفرصة التجارية');
        }

        return view('opportunities.show', compact('opportunity'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $opportunity = Opportunity::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($opportunity->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بتعديل هذه الفرصة التجارية');
        }

        $company_id = Auth::user()->company_id;
        $customers = Customer::where('company_id', $company_id)->get();

        return view('opportunities.edit', compact('opportunity', 'customers'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $opportunity = Opportunity::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($opportunity->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بتعديل هذه الفرصة التجارية');
        }

        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:new_sale,upgrade,maintenance,consultation,other',
            'stage' => 'required|in:lead,qualified,proposal,negotiation,closed_won,closed_lost',
            'amount' => 'nullable|numeric|min:0',
            'probability' => 'nullable|numeric|min:0|max:100',
            'expected_close_date' => 'nullable|date',
            'actual_close_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'source' => 'nullable|string|max:255',
            'loss_reason' => 'nullable|string|max:255',
        ]);

        $opportunity->update($validated);

        return redirect()->route('opportunities.index')
            ->with('success', 'تم تحديث الفرصة التجارية بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $opportunity = Opportunity::findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($opportunity->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بحذف هذه الفرصة التجارية');
        }

        $opportunity->delete();

        return redirect()->route('opportunities.index')
            ->with('success', 'تم حذف الفرصة التجارية بنجاح');
    }

    /**
     * عرض الفرص التجارية المرتبطة بعميل معين
     *
     * @param  int  $customerId
     * @return \Illuminate\Http\Response
     */
    public function customerOpportunities($customerId)
    {
        $customer = Customer::findOrFail($customerId);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($customer->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بعرض فرص هذا العميل');
        }

        $opportunities = Opportunity::with(['user'])
            ->where('customer_id', $customerId)
            ->orderBy('expected_close_date', 'asc')
            ->paginate(15);

        return view('opportunities.customer_opportunities', compact('opportunities', 'customer'));
    }
}
