<?php

namespace App\Http\Controllers;

use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class NotificationController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');

        // Apply permissions middleware
        $this->middleware('permission:view_maintenance_notifications', ['only' => ['index', 'getUnreadCount', 'getRecentUnread']]);
        $this->middleware('permission:view_maintenance_notifications', ['only' => ['show']]);
        $this->middleware('permission:view_maintenance_notifications', ['only' => ['markAsRead']]);
        $this->middleware('permission:view_maintenance_notifications', ['only' => ['markAllAsRead']]);
        $this->middleware('permission:delete_maintenance_notifications', ['only' => ['destroy']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $query = Notification::query();

        // الحصول على شركة المستخدم الحالي
        $userCompanyId = auth()->user()->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        if ($userCompanyId) {
            $query->where('company_id', $userCompanyId);
        }

        // Filter by type if provided
        if ($request->has('type') && $request->type != '') {
            $query->where('type', $request->type);
        }

        // Filter by status if provided
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        // Filter by priority if provided
        if ($request->has('priority') && $request->priority != '') {
            $query->where('priority', $request->priority);
        }

        // Filter by read/unread if provided
        if ($request->has('read') && $request->read != '') {
            if ($request->read == 'read') {
                $query->whereNotNull('read_at');
            } elseif ($request->read == 'unread') {
                $query->whereNull('read_at');
            }
        }

        // تقييد الإشعارات بالمستخدم الحالي أو الإشعارات العامة
        $notifications = $query->where(function ($q) {
            $q->where('user_id', Auth::id())
                ->orWhereNull('user_id');
        })->orderBy('created_at', 'desc')->paginate(15);

        return view('notifications.index', compact('notifications'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $notification = Notification::findOrFail($id);
        $user = Auth::user();
        $userCompanyId = $user->company_id;

        // التحقق من أن الإشعار ينتمي لشركة المستخدم
        if ($userCompanyId && $notification->company_id !== $userCompanyId) {
            abort(403, 'غير مصرح لك بعرض هذا الإشعار');
        }

        // التحقق من أن الإشعار ينتمي للمستخدم الحالي أو أنه إشعار عام
        if ($notification->user_id && $notification->user_id !== $user->id) {
            abort(403, 'غير مصرح لك بعرض هذا الإشعار');
        }

        // Mark as read if not already read
        if (!$notification->read_at) {
            $notification->markAsRead();
        }

        return view('notifications.show', compact('notification'));
    }

    /**
     * Mark a notification as read.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function markAsRead($id)
    {
        $notification = Notification::findOrFail($id);
        $user = Auth::user();
        $userCompanyId = $user->company_id;

        // التحقق من أن الإشعار ينتمي لشركة المستخدم
        if ($userCompanyId && $notification->company_id !== $userCompanyId) {
            abort(403, 'غير مصرح لك بتحديث هذا الإشعار');
        }

        // التحقق من أن الإشعار ينتمي للمستخدم الحالي أو أنه إشعار عام
        if ($notification->user_id && $notification->user_id !== $user->id) {
            abort(403, 'غير مصرح لك بتحديث هذا الإشعار');
        }

        $notification->markAsRead();

        return redirect()->back()->with('success', 'تم تحديث حالة الإشعار بنجاح.');
    }

    /**
     * Mark all notifications as read.
     *
     * @return \Illuminate\Http\Response
     */
    public function markAllAsRead()
    {
        $user = Auth::user();
        $userCompanyId = $user->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        $query = Notification::whereNull('read_at');

        // تقييد الإشعارات بشركة المستخدم
        if ($userCompanyId) {
            $query->where('company_id', $userCompanyId);
        }

        // تقييد الإشعارات بالمستخدم الحالي أو الإشعارات العامة
        $notifications = $query->where(function ($q) use ($user) {
            $q->where('user_id', $user->id)
                ->orWhereNull('user_id');
        })->get();

        foreach ($notifications as $notification) {
            $notification->markAsRead();
        }

        return redirect()->back()->with('success', 'تم تحديث جميع الإشعارات بنجاح.');
    }

    /**
     * Delete a notification.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $notification = Notification::findOrFail($id);
        $user = Auth::user();
        $userCompanyId = $user->company_id;

        // التحقق من أن الإشعار ينتمي لشركة المستخدم
        if ($userCompanyId && $notification->company_id !== $userCompanyId) {
            abort(403, 'غير مصرح لك بحذف هذا الإشعار');
        }

        // التحقق من أن الإشعار ينتمي للمستخدم الحالي أو أنه إشعار عام
        if ($notification->user_id && $notification->user_id !== $user->id) {
            abort(403, 'غير مصرح لك بحذف هذا الإشعار');
        }

        $notification->delete();

        return redirect()->route('notifications.index')->with('success', 'تم حذف الإشعار بنجاح.');
    }

    /**
     * Get unread notifications count for the current user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getUnreadCount()
    {
        $user = Auth::user();
        $userCompanyId = $user->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        $query = Notification::whereNull('read_at');

        // تقييد الإشعارات بشركة المستخدم
        if ($userCompanyId) {
            $query->where('company_id', $userCompanyId);
        }

        // تقييد الإشعارات بالمستخدم الحالي أو الإشعارات العامة
        $count = $query->where(function ($q) use ($user) {
            $q->where('user_id', $user->id)
                ->orWhereNull('user_id');
        })->count();

        return response()->json(['count' => $count]);
    }

    /**
     * Get recent unread notifications for the current user.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRecentUnread()
    {
        $user = Auth::user();
        $userCompanyId = $user->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        $query = Notification::whereNull('read_at');

        // تقييد الإشعارات بشركة المستخدم
        if ($userCompanyId) {
            $query->where('company_id', $userCompanyId);
        }

        // تقييد الإشعارات بالمستخدم الحالي أو الإشعارات العامة
        $notifications = $query->where(function ($q) use ($user) {
            $q->where('user_id', $user->id)
                ->orWhereNull('user_id');
        })
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        return response()->json(['notifications' => $notifications]);
    }
}
