<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\MaintenanceTeam;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MaintenanceTeamController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $maintenanceTeams = MaintenanceTeam::with(['members', 'teamLeader', 'company'])
            ->orderBy('name')
            ->paginate(15);

        return view('maintenance.teams.index', compact('maintenanceTeams'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $users = User::where('is_active', true)->orderBy('name')->get();
        $companies = Company::orderBy('name')->get();

        return view('maintenance.teams.create', compact('users', 'companies'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'company_id' => 'required|exists:companies,id',
            'team_leader_id' => 'required|exists:users,id',
            'description' => 'nullable|string',
            'members' => 'required|array',
            'members.*' => 'exists:users,id',
            'member_roles' => 'nullable|array',
            'member_roles.*' => 'nullable|string|max:255',
        ]);

        $maintenanceTeam = new MaintenanceTeam();
        $maintenanceTeam->name = $request->name;
        $maintenanceTeam->company_id = $request->company_id;
        $maintenanceTeam->leader_id = $request->team_leader_id;
        $maintenanceTeam->description = $request->description;
        $maintenanceTeam->created_by = Auth::id();
        $maintenanceTeam->save();

        // Attach team members with their roles
        foreach ($request->members as $index => $userId) {
            $role = isset($request->member_roles[$index]) ? $request->member_roles[$index] : 'Member';
            $maintenanceTeam->members()->attach($userId, ['role' => $role]);
        }

        return redirect()->route('maintenance-teams.index')
            ->with('success', 'تم إنشاء فريق الصيانة بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\MaintenanceTeam  $maintenanceTeam
     * @return \Illuminate\Http\Response
     */
    public function show(MaintenanceTeam $maintenanceTeam)
    {
        $maintenanceTeam->load(['members', 'teamLeader', 'company', 'creator', 'schedules', 'maintenanceRequests']);

        return view('maintenance.teams.show', compact('maintenanceTeam'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\MaintenanceTeam  $maintenanceTeam
     * @return \Illuminate\Http\Response
     */
    public function edit(MaintenanceTeam $maintenanceTeam)
    {
        $maintenanceTeam->load(['members', 'teamLeader', 'company']);
        $users = User::where('is_active', true)->orderBy('name')->get();
        $companies = Company::orderBy('name')->get();

        return view('maintenance.teams.edit', compact('maintenanceTeam', 'users', 'companies'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\MaintenanceTeam  $maintenanceTeam
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, MaintenanceTeam $maintenanceTeam)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'company_id' => 'required|exists:companies,id',
            'team_leader_id' => 'required|exists:users,id',
            'description' => 'nullable|string',
            'members' => 'required|array',
            'members.*' => 'exists:users,id',
            'member_roles' => 'nullable|array',
            'member_roles.*' => 'nullable|string|max:255',
        ]);

        $maintenanceTeam->name = $request->name;
        $maintenanceTeam->company_id = $request->company_id;
        $maintenanceTeam->leader_id = $request->team_leader_id;
        $maintenanceTeam->description = $request->description;
        $maintenanceTeam->save();

        // Detach all members first
        $maintenanceTeam->members()->detach();

        // Attach team members with their roles
        foreach ($request->members as $index => $userId) {
            $role = isset($request->member_roles[$index]) ? $request->member_roles[$index] : 'Member';
            $maintenanceTeam->members()->attach($userId, ['role' => $role]);
        }

        return redirect()->route('maintenance-teams.index')
            ->with('success', 'تم تحديث فريق الصيانة بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\MaintenanceTeam  $maintenanceTeam
     * @return \Illuminate\Http\Response
     */
    public function destroy(MaintenanceTeam $maintenanceTeam)
    {
        // Check if team has any assigned maintenance requests
        if ($maintenanceTeam->maintenanceRequests()->count() > 0) {
            return redirect()->route('maintenance-teams.index')
                ->with('error', 'لا يمكن حذف الفريق لأنه مرتبط بطلبات صيانة');
        }

        // Detach all members before deleting
        $maintenanceTeam->members()->detach();
        $maintenanceTeam->delete();

        return redirect()->route('maintenance-teams.index')
            ->with('success', 'تم حذف فريق الصيانة بنجاح');
    }
}
