<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\MaintenanceContract;
use App\Models\MaintenanceSchedule;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MaintenanceScheduleController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
        // Temporarily remove permission check for calendar view
        $this->middleware('permission:view-maintenance-schedule')->only(['index', 'show']);
        $this->middleware('permission:create-maintenance-schedule')->only(['create', 'store', 'generateSchedules']);
        $this->middleware('permission:edit-maintenance-schedule')->only(['edit', 'update']);
        $this->middleware('permission:delete-maintenance-schedule')->only('destroy');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $query = MaintenanceSchedule::with(['elevator', 'maintenance_contract', 'user']);

        // Apply company isolation - only show schedules for the user's company
        if (Auth::user()->company_id) {
            $query->where('company_id', Auth::user()->company_id);
        }

        // Filter by status if provided
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        // Filter by date range if provided
        if ($request->has('start_date') && $request->start_date != '') {
            $query->where('scheduled_date', '>=', $request->start_date);
        }
        if ($request->has('end_date') && $request->end_date != '') {
            $query->where('scheduled_date', '<=', $request->end_date);
        }

        // Filter by elevator if provided
        if ($request->has('elevator_id') && $request->elevator_id != '') {
            $query->where('elevator_id', $request->elevator_id);
        }

        // Filter by contract if provided
        if ($request->has('contract_id') && $request->contract_id != '') {
            $query->where('maintenance_contract_id', $request->contract_id);
        }

        $schedules = $query->orderBy('scheduled_date', 'asc')->paginate(15);

        // Get data for filters - only show elevators and contracts for the user's company
        $elevatorsQuery = Elevator::select('id', 'model', 'serial_number');
        $contractsQuery = MaintenanceContract::select('id', 'contract_number');

        // Apply company isolation to filters
        if (Auth::user()->company_id) {
            $elevatorsQuery->where('company_id', Auth::user()->company_id);
            $contractsQuery->where('company_id', Auth::user()->company_id);
        }

        $elevators = $elevatorsQuery->get();
        $contracts = $contractsQuery->get();

        return view('maintenance.schedules.index', compact('schedules', 'elevators', 'contracts'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // Aplicar aislamiento de empresas a los filtros
        $elevatorsQuery = Elevator::select('id', 'model', 'serial_number');
        $contractsQuery = MaintenanceContract::select('id', 'contract_number');
        $techniciansQuery = User::whereHas('roles', function ($query) {
            $query->where('name', 'Technician');
        });

        // Aplicar aislamiento de empresas
        if (Auth::user()->company_id) {
            $elevatorsQuery->where('company_id', Auth::user()->company_id);
            $contractsQuery->where('company_id', Auth::user()->company_id);
            $techniciansQuery->where('company_id', Auth::user()->company_id);
        }

        $elevators = $elevatorsQuery->get();
        $contracts = $contractsQuery->get();
        $technicians = $techniciansQuery->get();

        return view('maintenance.schedules.create', compact('elevators', 'contracts', 'technicians'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'maintenance_contract_id' => 'required|exists:maintenance_contracts,id',
            'elevator_id' => 'required|exists:elevators,id',
            'scheduled_date' => 'required|date',
            'status' => 'required|in:scheduled,completed,missed,cancelled',
            'notes' => 'nullable|string',
        ]);

        $schedule = new MaintenanceSchedule();
        $schedule->maintenance_contract_id = $request->maintenance_contract_id;
        $schedule->elevator_id = $request->elevator_id;
        $schedule->scheduled_date = $request->scheduled_date;
        $schedule->status = $request->status;
        $schedule->notes = $request->notes;
        $schedule->company_id = Auth::user()->company_id;
        $schedule->user_id = Auth::id();
        $schedule->save();

        return redirect()->route('maintenance-schedules.index')
            ->with('success', 'تم إنشاء جدول الصيانة بنجاح.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $query = MaintenanceSchedule::with(['elevator', 'maintenance_contract', 'user', 'maintenance_logs']);

        // Aplicar aislamiento de empresas - solo mostrar horarios de la empresa del usuario
        if (Auth::user()->company_id) {
            $query->where('company_id', Auth::user()->company_id);
        }

        $schedule = $query->findOrFail($id);

        return view('maintenance.schedules.show', compact('schedule'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        // Aplicar aislamiento de empresas al horario
        $query = MaintenanceSchedule::query();
        if (Auth::user()->company_id) {
            $query->where('company_id', Auth::user()->company_id);
        }
        $schedule = $query->findOrFail($id);

        // Aplicar aislamiento de empresas a los filtros
        $elevatorsQuery = Elevator::select('id', 'model', 'serial_number');
        $contractsQuery = MaintenanceContract::select('id', 'contract_number');

        if (Auth::user()->company_id) {
            $elevatorsQuery->where('company_id', Auth::user()->company_id);
            $contractsQuery->where('company_id', Auth::user()->company_id);
        }

        $elevators = $elevatorsQuery->get();
        $contracts = $contractsQuery->get();

        return view('maintenance.schedules.edit', compact('schedule', 'elevators', 'contracts'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'maintenance_contract_id' => 'required|exists:maintenance_contracts,id',
            'elevator_id' => 'required|exists:elevators,id',
            'scheduled_date' => 'required|date',
            'status' => 'required|in:scheduled,completed,missed,cancelled',
            'notes' => 'nullable|string',
        ]);

        // Aplicar aislamiento de empresas
        $query = MaintenanceSchedule::query();
        if (Auth::user()->company_id) {
            $query->where('company_id', Auth::user()->company_id);
        }
        $schedule = $query->findOrFail($id);

        // Verificar que el contrato y el ascensor pertenecen a la empresa del usuario
        if (Auth::user()->company_id) {
            $contract = MaintenanceContract::where('id', $request->maintenance_contract_id)
                ->where('company_id', Auth::user()->company_id)
                ->first();

            $elevator = Elevator::where('id', $request->elevator_id)
                ->where('company_id', Auth::user()->company_id)
                ->first();

            if (!$contract || !$elevator) {
                return redirect()->back()->with('error', 'لا يمكنك استخدام عقد أو مصعد لا ينتمي إلى شركتك');
            }
        }

        $schedule->maintenance_contract_id = $request->maintenance_contract_id;
        $schedule->elevator_id = $request->elevator_id;
        $schedule->scheduled_date = $request->scheduled_date;
        $schedule->status = $request->status;
        $schedule->notes = $request->notes;
        $schedule->save();

        return redirect()->route('maintenance-schedules.index')
            ->with('success', 'تم تحديث جدول الصيانة بنجاح.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        // Aplicar aislamiento de empresas
        $query = MaintenanceSchedule::query();
        if (Auth::user()->company_id) {
            $query->where('company_id', Auth::user()->company_id);
        }
        $schedule = $query->findOrFail($id);

        $schedule->delete();

        return redirect()->route('maintenance-schedules.index')
            ->with('success', 'تم حذف جدول الصيانة بنجاح.');
    }

    /**
     * Generate maintenance schedules based on contract frequency.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function generateSchedules(Request $request)
    {
        try {
            \Log::info('Starting generateSchedules with data: ' . json_encode($request->all()));

            $request->validate([
                'contract_id' => 'required|exists:maintenance_contracts,id',
                'start_date' => 'required|date',
                'end_date' => 'required|date|after:start_date',
            ]);

            // Obtener el contrato y aplicar aislamiento de empresas
            $query = MaintenanceContract::with('elevators');

            // Aplicar aislamiento de empresas - solo mostrar contratos de la empresa del usuario
            if (Auth::user()->company_id) {
                $query->where('company_id', Auth::user()->company_id);
            }

            $contract = $query->findOrFail($request->contract_id);

            // Verificar que el contrato pertenece a la empresa del usuario
            if (Auth::user()->company_id && $contract->company_id != Auth::user()->company_id) {
                throw new \Exception('لا يمكنك الوصول إلى هذا العقد لأنه لا ينتمي إلى شركتك');
            }
            \Log::info('Contract found: ' . json_encode($contract->toArray()));

            // التحقق من أن العقد نشط
            if ($contract->status !== 'active') {
                throw new \Exception('لا يمكن إنشاء جداول صيانة لعقد غير نشط. حالة العقد الحالية: ' . $contract->status);
            }

            $startDate = $request->start_date;
            $endDate = $request->end_date;

            // التحقق من أن التواريخ ضمن فترة العقد
            $contractStartDate = new \DateTime($contract->start_date);
            $contractEndDate = new \DateTime($contract->end_date);
            $requestStartDate = new \DateTime($startDate);
            $requestEndDate = new \DateTime($endDate);

            if ($requestStartDate < $contractStartDate) {
                throw new \Exception('تاريخ البداية يجب أن يكون بعد تاريخ بداية العقد (' . $contractStartDate->format('Y-m-d') . ')');
            }

            if ($requestEndDate > $contractEndDate) {
                throw new \Exception('تاريخ النهاية يجب أن يكون قبل تاريخ نهاية العقد (' . $contractEndDate->format('Y-m-d') . ')');
            }

            $billing_cycle = $contract->billing_cycle; // monthly, quarterly, semi_annually, annually
            $visits_per_year = $contract->visits_per_year ?: 12; // عدد الزيارات في السنة، الافتراضي 12
            \Log::info('Using billing cycle: ' . $billing_cycle . ', visits per year: ' . $visits_per_year);

            // حساب الفاصل الزمني بناءً على عدد الزيارات في السنة
            $interval = 12 / $visits_per_year;

            // التحقق من أن الفاصل الزمني صحيح
            if ($interval < 1) {
                $interval = 1; // الحد الأدنى شهر واحد
            }

            \Log::info('Calculated interval: ' . $interval . ' months');

            $currentDate = new \DateTime($startDate);
            $endDateTime = new \DateTime($endDate);
            $count = 0;

            // التحقق من وجود مصاعد مرتبطة بالعقد
            if ($contract->elevators->isEmpty()) {
                return redirect()->route('maintenance-schedules.index')
                    ->with('error', "لا توجد مصاعد مرتبطة بهذا العقد. يرجى إضافة مصاعد للعقد أولاً.");
            }

            // حساب إجمالي عدد الزيارات المتوقعة للعقد
            $totalExpectedVisits = $this->calculateTotalExpectedVisits($contract);

            // حساب عدد الزيارات المجدولة بالفعل للعقد
            $existingSchedulesCount = MaintenanceSchedule::where('maintenance_contract_id', $contract->id)->count();

            \Log::info("Total expected visits for contract: {$totalExpectedVisits}, Existing schedules: {$existingSchedulesCount}");

            // التحقق مما إذا كان تم إنشاء جميع الزيارات المتوقعة بالفعل
            if ($existingSchedulesCount >= $totalExpectedVisits) {
                $message = "تم إنشاء جميع جداول الصيانة المسجلة بالعقد مسبقاً. لا يمكن إنشاء المزيد من الجداول.";

                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => $message
                    ], 422);
                }

                return redirect()->route('maintenance-schedules.index')
                    ->with('error', $message);
            }

            // التحقق من عدم وجود تداخل في جداول الصيانة
            $elevatorIds = $contract->elevators->pluck('id')->toArray();

            // جمع التواريخ التي سيتم إنشاء جداول صيانة لها
            $scheduleDates = [];
            $tempDate = clone $currentDate;
            while ($tempDate <= $endDateTime) {
                $scheduleDates[] = $tempDate->format('Y-m-d');
                $days_interval = round(30 * $interval);
                $tempDate->modify("+{$days_interval} days");
            }

            // التحقق من وجود جداول صيانة مسبقة للمصاعد في نفس التواريخ
            $existingSchedules = MaintenanceSchedule::where('maintenance_contract_id', $contract->id)
                ->whereIn('elevator_id', $elevatorIds)
                ->whereIn('scheduled_date', $scheduleDates)
                ->get();

            if ($existingSchedules->isNotEmpty()) {
                // تجميع المصاعد والتواريخ التي لديها جداول صيانة مسبقة
                $existingElevatorDates = [];
                foreach ($existingSchedules as $schedule) {
                    $elevatorId = $schedule->elevator_id;
                    $date = $schedule->scheduled_date->format('Y-m-d');
                    $existingElevatorDates[$elevatorId][] = $date;
                }

                \Log::info('Found existing schedules: ' . json_encode($existingElevatorDates));
            }

            \Log::info('Starting schedule creation from ' . $currentDate->format('Y-m-d') . ' to ' . $endDateTime->format('Y-m-d'));
            $iteration = 1;
            $remainingVisits = $totalExpectedVisits - $existingSchedulesCount;

            while ($currentDate <= $endDateTime && $count < $remainingVisits) {
                \Log::info('Iteration ' . $iteration . ': Creating schedules for ' . $currentDate->format('Y-m-d'));
                $dateStr = $currentDate->format('Y-m-d');

                foreach ($contract->elevators as $elevator) {
                    // التحقق مما إذا كان هناك جدول صيانة موجود بالفعل لهذا المصعد في هذا التاريخ
                    $existingSchedule = isset($existingElevatorDates[$elevator->id]) && in_array($dateStr, $existingElevatorDates[$elevator->id]);

                    if (!$existingSchedule) {
                        $schedule = MaintenanceSchedule::create([
                            'maintenance_contract_id' => $contract->id,
                            'elevator_id' => $elevator->id,
                            'scheduled_date' => $dateStr,
                            'status' => 'scheduled',
                            'company_id' => Auth::user()->company_id,
                            'user_id' => Auth::id(),
                        ]);

                        \Log::info('Created schedule ID: ' . $schedule->id . ' for elevator ID: ' . $elevator->id . ' on date: ' . $dateStr);
                        $count++;

                        // إضافة هذا الجدول إلى قائمة الجداول الموجودة لتجنب التكرار
                        $existingElevatorDates[$elevator->id][] = $dateStr;
                    } else {
                        \Log::info('Skipping duplicate schedule for elevator ID: ' . $elevator->id . ' on date: ' . $dateStr);
                    }

                    // التحقق مما إذا وصلنا إلى الحد الأقصى من الزيارات المتبقية
                    if ($count >= $remainingVisits) {
                        break 2; // الخروج من كلا الحلقتين
                    }
                }

                $iteration++;

                // حساب الفاصل الزمني بالأيام
                $days_interval = round(30 * $interval); // تقريب عدد الأيام
                \Log::info('Adding ' . $days_interval . ' days to current date: ' . $currentDate->format('Y-m-d'));

                // إضافة الفاصل الزمني بالأيام
                $currentDate->modify("+{$days_interval} days");
            }

            \Log::info("Successfully created {$count} maintenance schedules");

            $message = $count > 0
                ? "تم إنشاء {$count} جدول صيانة بنجاح."
                : "لم يتم إنشاء أي جداول صيانة جديدة. جميع الجداول المطلوبة موجودة بالفعل.";

            // التحقق مما إذا كان الطلب AJAX
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => $message
                ]);
            }

            return redirect()->route('maintenance-schedules.index')
                ->with('success', $message);
        } catch (\Exception $e) {
            \Log::error('Error in generateSchedules: ' . $e->getMessage() . "\n" . $e->getTraceAsString());

            // التحقق مما إذا كان الطلب AJAX
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => "حدث خطأ أثناء إنشاء جداول الصيانة: " . $e->getMessage()
                ], 422);
            }

            return redirect()->route('maintenance-schedules.index')
                ->with('error', "حدث خطأ أثناء إنشاء جداول الصيانة: " . $e->getMessage());
        }
    }

    /**
     * حساب إجمالي عدد الزيارات المتوقعة للعقد
     *
     * @param MaintenanceContract $contract
     * @return int
     */
    private function calculateTotalExpectedVisits(MaintenanceContract $contract)
    {
        // حساب عدد الأشهر في العقد
        $startDate = new \DateTime($contract->start_date);
        $endDate = new \DateTime($contract->end_date);
        $interval = $startDate->diff($endDate);
        $totalMonths = ($interval->y * 12) + $interval->m + ($interval->d > 0 ? 1 : 0);

        // حساب عدد الزيارات لكل مصعد
        $visitsPerElevator = ceil(($totalMonths / 12) * $contract->visits_per_year);

        // حساب إجمالي عدد الزيارات لجميع المصاعد
        $totalVisits = $visitsPerElevator * $contract->elevators->count();

        \Log::info("Contract duration: {$totalMonths} months, Visits per elevator: {$visitsPerElevator}, Total visits: {$totalVisits}");

        return $totalVisits;
    }

    /**
     * Display the maintenance calendar view.
     *
     * @return \Illuminate\Http\Response
     */
    public function calendar()
    {
        return view('maintenance-schedules.calendar');
    }
}
