<?php

namespace App\Http\Controllers;

use App\Models\Notification;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MaintenanceNotificationsController extends Controller
{
    /**
     * خدمة الإشعارات.
     *
     * @var \App\Services\NotificationService
     */
    protected $notificationService;

    /**
     * إنشاء مثيل جديد للتحكم.
     *
     * @param  \App\Services\NotificationService  $notificationService
     * @return void
     */
    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * عرض قائمة إشعارات الصيانة.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $user = Auth::user();
        $companyId = $user->company_id;

        // الحصول على إشعارات الصيانة للمستخدم الحالي مع تطبيق عزل الشركات
        $query = Notification::where('user_id', $user->id)
            ->where(function ($query) {
                $query->where('type', 'like', 'maintenance%')
                    ->orWhere('type', 'warranty_expiring')
                    ->orWhere('type', 'contract_expiring')
                    ->orWhere('type', 'daily_maintenance');
            });

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        if ($companyId) {
            $query->where('company_id', $companyId);
        }

        $notifications = $query->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('maintenance.notifications.index', [
            'notifications' => $notifications
        ]);
    }

    /**
     * عرض تفاصيل إشعار الصيانة.
     *
     * @param  \App\Models\Notification  $notification
     * @return \Illuminate\View\View
     */
    public function show(Notification $notification)
    {
        $user = Auth::user();
        $companyId = $user->company_id;

        // التحقق من أن الإشعار ينتمي للمستخدم الحالي وشركته
        if ($notification->user_id !== Auth::id() || ($companyId && $notification->company_id !== $companyId)) {
            abort(403, 'غير مصرح لك بعرض هذا الإشعار');
        }

        // تحديث حالة الإشعار إلى مقروء
        if (!$notification->is_read) {
            $notification->is_read = true;
            $notification->read_at = now();
            $notification->save();
        }

        // الحصول على الكائن المرتبط بالإشعار
        $notifiableObject = null;
        if ($notification->notifiable_type && $notification->notifiable_id) {
            $notifiableType = $notification->notifiable_type;
            $notifiableObject = $notifiableType::find($notification->notifiable_id);
        }

        return view('maintenance.notifications.show', [
            'notification' => $notification,
            'notifiableObject' => $notifiableObject
        ]);
    }

    /**
     * تحديث حالة إشعار الصيانة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Notification  $notification
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateStatus(Request $request, Notification $notification)
    {
        $user = Auth::user();
        $companyId = $user->company_id;

        // التحقق من أن الإشعار ينتمي للمستخدم الحالي وشركته
        if ($notification->user_id !== Auth::id() || ($companyId && $notification->company_id !== $companyId)) {
            abort(403, 'غير مصرح لك بتحديث هذا الإشعار');
        }

        $request->validate([
            'status' => 'required|in:read,unread',
        ]);

        if ($request->status === 'read') {
            $notification->is_read = true;
            $notification->read_at = now();
        } else {
            $notification->is_read = false;
            $notification->read_at = null;
        }

        $notification->save();

        return redirect()->route('maintenance-notifications.index')
            ->with('success', 'تم تحديث حالة الإشعار بنجاح');
    }

    /**
     * إرسال إشعارات الصيانة المعلقة.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function sendPending()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('send_maintenance_notifications')) {
            abort(403, 'غير مصرح لك بإرسال الإشعارات');
        }

        $count = $this->notificationService->sendPendingNotifications();

        return redirect()->route('maintenance-notifications.index')
            ->with('success', 'تم إرسال ' . $count . ' إشعار معلق بنجاح');
    }

    /**
     * حذف إشعار الصيانة.
     *
     * @param  \App\Models\Notification  $notification
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Notification $notification)
    {
        $user = Auth::user();
        $companyId = $user->company_id;

        // التحقق من أن الإشعار ينتمي للمستخدم الحالي وشركته
        if ($notification->user_id !== Auth::id() || ($companyId && $notification->company_id !== $companyId)) {
            abort(403, 'غير مصرح لك بحذف هذا الإشعار');
        }

        $notification->delete();

        return redirect()->route('maintenance-notifications.index')
            ->with('success', 'تم حذف الإشعار بنجاح');
    }
}
