<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\Invoice;
use App\Models\MaintenanceContract;
use App\Models\MaintenanceLog;
use App\Models\MaintenanceSchedule;
use App\Models\User;
use App\Services\MaintenanceFinancialService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;

class MaintenanceFinancialController extends Controller
{
    /**
     * خدمة التقارير المالية للصيانة.
     *
     * @var \App\Services\MaintenanceFinancialService
     */
    protected $maintenanceFinancialService;

    /**
     * إنشاء مثيل جديد للتحكم.
     *
     * @param  \App\Services\MaintenanceFinancialService  $maintenanceFinancialService
     * @return void
     */
    public function __construct(MaintenanceFinancialService $maintenanceFinancialService)
    {
        $this->maintenanceFinancialService = $maintenanceFinancialService;
    }

    /**
     * عرض صفحة التقارير المالية للصيانة.
     *
     * @return \Illuminate\View\View
     */
    public function showReports()
    {
        // الحصول على الفترة الزمنية الافتراضية (الشهر الحالي)
        $startDate = Carbon::now()->startOfMonth();
        $endDate = Carbon::now()->endOfMonth();

        // الحصول على إحصائيات افتراضية
        $companyId = auth()->user()->company_id;
        $report = $this->maintenanceFinancialService->getMaintenanceFinancialReport(
            $companyId,
            $startDate,
            $endDate
        );

        // الحصول على قائمة العملاء للتصفية
        $customers = DB::table('customers')
            ->where('company_id', $companyId)
            ->select('id', 'name')
            ->get();

        return view('maintenance.financial.reports', [
            'report' => $report,
            'startDate' => $startDate->format('Y-m-d'),
            'endDate' => $endDate->format('Y-m-d'),
            'customers' => $customers
        ]);
    }

    /**
     * معالجة طلب التقارير المالية للصيانة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\View\View
     */
    public function processReports(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'customer_id' => 'nullable|exists:customers,id',
        ]);

        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        $companyId = auth()->user()->company_id;
        $customerId = $request->customer_id;

        $report = $this->maintenanceFinancialService->getMaintenanceFinancialReport(
            $companyId,
            $startDate,
            $endDate,
            $customerId
        );

        // الحصول على قائمة العملاء للتصفية
        $customers = DB::table('customers')
            ->where('company_id', $companyId)
            ->select('id', 'name')
            ->get();

        return view('maintenance.financial.reports', [
            'report' => $report,
            'startDate' => $startDate->format('Y-m-d'),
            'endDate' => $endDate->format('Y-m-d'),
            'customers' => $customers,
            'selectedCustomerId' => $customerId
        ]);
    }

    /**
     * عرض صفحة تقارير أداء الفنيين.
     *
     * @return \Illuminate\View\View
     */
    public function showTechnicianPerformance()
    {
        // الحصول على الفترة الزمنية الافتراضية (الشهر الحالي)
        $startDate = Carbon::now()->startOfMonth();
        $endDate = Carbon::now()->endOfMonth();

        // الحصول على إحصائيات افتراضية
        $companyId = auth()->user()->company_id;
        $report = $this->maintenanceFinancialService->getTechnicianPerformanceReport(
            $companyId,
            $startDate,
            $endDate
        );

        // الحصول على قائمة الفنيين للتصفية
        $technicians = User::whereHas('roles', function ($query) {
            $query->where('name', 'Technician');
        })
            ->where('company_id', $companyId)
            ->select('id', 'name')
            ->get();

        return view('maintenance.financial.technician-performance', [
            'report' => $report,
            'startDate' => $startDate->format('Y-m-d'),
            'endDate' => $endDate->format('Y-m-d'),
            'technicians' => $technicians
        ]);
    }

    /**
     * معالجة طلب تقارير أداء الفنيين.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\View\View
     */
    public function processTechnicianPerformance(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'technician_id' => 'nullable|exists:users,id',
        ]);

        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        $companyId = auth()->user()->company_id;
        $technicianId = $request->technician_id;

        $report = $this->maintenanceFinancialService->getTechnicianPerformanceReport(
            $companyId,
            $startDate,
            $endDate,
            $technicianId
        );

        // الحصول على قائمة الفنيين للتصفية
        $technicians = User::whereHas('roles', function ($query) {
            $query->where('name', 'Technician');
        })
            ->where('company_id', $companyId)
            ->select('id', 'name')
            ->get();

        return view('maintenance.financial.technician-performance', [
            'report' => $report,
            'startDate' => $startDate->format('Y-m-d'),
            'endDate' => $endDate->format('Y-m-d'),
            'technicians' => $technicians,
            'selectedTechnicianId' => $technicianId
        ]);
    }

    /**
     * عرض صفحة إنشاء فواتير الصيانة.
     *
     * @return \Illuminate\View\View
     */
    public function showGenerateInvoices()
    {
        // الحصول على الفترة الزمنية الافتراضية (الأسبوع الماضي)
        $startDate = Carbon::now()->subDays(7);
        $endDate = Carbon::now();

        // الحصول على عدد سجلات الصيانة المكتملة التي لم يتم إنشاء فواتير لها
        $companyId = auth()->user()->company_id;
        $completedMaintenanceCount = MaintenanceLog::where('company_id', $companyId)
            ->where('status', 'completed')
            ->whereBetween('maintenance_date', [$startDate, $endDate])
            ->count();

        return view('maintenance.financial.generate-invoices', [
            'startDate' => $startDate->format('Y-m-d'),
            'endDate' => $endDate->format('Y-m-d'),
            'completedMaintenanceCount' => $completedMaintenanceCount
        ]);
    }

    /**
     * معالجة طلب إنشاء فواتير الصيانة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function processGenerateInvoices(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);

        // إنشاء الفواتير
        $stats = $this->maintenanceFinancialService->generateInvoicesFromCompletedMaintenance(
            $startDate,
            $endDate
        );

        return redirect()->route('maintenance-financial.generate-invoices')
            ->with('success', 'تم إنشاء ' . $stats['success'] . ' فاتورة بنجاح من سجلات الصيانة المكتملة.')
            ->with('stats', $stats);
    }

    /**
     * تصدير التقارير المالية.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportReports(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'customer_id' => 'nullable|exists:customers,id',
            'export_type' => 'required|in:excel,pdf',
        ]);

        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        $companyId = auth()->user()->company_id;
        $customerId = $request->customer_id;
        $exportType = $request->export_type;

        // الحصول على التقرير
        $report = $this->maintenanceFinancialService->getMaintenanceFinancialReport(
            $companyId,
            $startDate,
            $endDate,
            $customerId
        );

        // تصدير التقرير
        if ($exportType === 'excel') {
            return $this->maintenanceFinancialService->exportReportToExcel($report, $startDate, $endDate);
        } else {
            return $this->maintenanceFinancialService->exportReportToPdf($report, $startDate, $endDate);
        }
    }
}
