<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\MaintenanceContract;
use App\Models\ContractElevator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class MaintenanceElevatorsReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * تقرير كل مصاعد الصيانة
     */
    public function index(Request $request)
    {
        $companyId = Auth::user()->company_id;

        // بناء الاستعلام الأساسي
        $query = Elevator::where('elevators.company_id', $companyId)
            ->join('customers', 'elevators.customer_id', '=', 'customers.id')
            ->join('locations', 'elevators.location_id', '=', 'locations.id')
            ->join('contract_elevator', 'elevators.id', '=', 'contract_elevator.elevator_id')
            ->join('maintenance_contracts', 'contract_elevator.maintenance_contract_id', '=', 'maintenance_contracts.id')
            ->where('maintenance_contracts.status', 'active')
            ->select(
                'elevators.id',
                'elevators.serial_number',
                'elevators.status as elevator_status',
                'locations.name as location_name',
                'locations.latitude',
                'locations.longitude',
                'customers.name as customer_name',
                'customers.mobile as customer_mobile',
                'maintenance_contracts.end_date as contract_end_date'
            );

        // تطبيق البحث إذا تم تقديمه
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('customers.name', 'like', "%{$searchTerm}%")
                    ->orWhere('elevators.serial_number', 'like', "%{$searchTerm}%")
                    ->orWhere('customers.mobile', 'like', "%{$searchTerm}%");
            });
        }

        // تطبيق الفلتر حسب حالة المصعد إذا تم تقديمه
        if ($request->has('status') && !empty($request->status)) {
            $query->where('elevators.status', $request->status);
        }

        $elevators = $query->orderBy('maintenance_contracts.end_date', 'asc')->get();

        // تحويل حالة المصعد إلى أسماء مقروءة
        $elevators->map(function ($elevator) {
            $statusNames = [
                'active' => 'نشط',
                'inactive' => 'غير نشط',
                'maintenance' => 'قيد الصيانة',
                'out_of_service' => 'خارج الخدمة',
                'under_repair' => 'تحت الإصلاح'
            ];

            $elevator->elevator_status_name = $statusNames[$elevator->elevator_status] ?? $elevator->elevator_status;

            // إنشاء رابط الموقع إذا كانت الإحداثيات متوفرة
            if ($elevator->latitude && $elevator->longitude) {
                $elevator->location_url = "https://www.google.com/maps?q={$elevator->latitude},{$elevator->longitude}";
            } else {
                $elevator->location_url = null;
            }

            return $elevator;
        });

        // الحصول على قائمة حالات المصاعد للفلتر
        $elevatorStatuses = [
            'active' => 'نشط',
            'inactive' => 'غير نشط',
            'maintenance' => 'قيد الصيانة',
            'out_of_service' => 'خارج الخدمة',
            'under_repair' => 'تحت الإصلاح'
        ];

        return view('elevator-reports.maintenance-elevators', compact('elevators', 'elevatorStatuses'));
    }
}
