<?php

namespace App\Http\Controllers;

use App\Models\MaintenanceContract;
use App\Models\Customer;
use App\Models\Elevator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MaintenanceContractController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $query = MaintenanceContract::with(['customer', 'elevators']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $contracts = $query->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('maintenance_contracts.index', compact('contracts'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $customerQuery = Customer::query();
        $elevatorQuery = Elevator::query();

        if (!$user->isSystemAdmin()) {
            $customerQuery->where('company_id', $user->company_id);
            $elevatorQuery->where('company_id', $user->company_id);
        }

        $customers = $customerQuery->get();
        $elevators = $elevatorQuery->get();

        return view('maintenance_contracts.create', compact('customers', 'elevators'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'contract_number' => [
                'required',
                'string',
                'max:255',
                function ($attribute, $value, $fail) use ($user) {
                    // التحقق من فريدية رقم العقد ضمن نطاق الشركة الواحدة
                    $query = MaintenanceContract::where('contract_number', $value);
                    
                    if (!$user->isSystemAdmin()) {
                        $query->where('company_id', $user->company_id);
                    }
                    
                    if ($query->exists()) {
                        $fail('قيمة contract number مُستخدمة من قبل في نفس الشركة.');
                    }
                }
            ],
            'title' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'total_amount' => 'required|numeric|min:0',
            'billing_cycle' => 'required|in:monthly,quarterly,semi_annually,annually',
            'visits_per_year' => 'required|integer|min:1',
            'elevator_ids' => 'required|array',
            'elevator_ids.*' => 'exists:elevators,id',
        ]);

        $customer = Customer::findOrFail($request->customer_id);

        $contract = new MaintenanceContract();
        $contract->company_id = $user->company_id ?? $customer->company_id;
        $contract->branch_id = $user->branch_id ?? $customer->branch_id;
        $contract->customer_id = $request->customer_id;
        $contract->contract_number = $request->contract_number;
        $contract->title = $request->title;
        $contract->start_date = $request->start_date;
        $contract->end_date = $request->end_date;
        $contract->total_amount = $request->total_amount;
        $contract->billing_cycle = $request->billing_cycle;
        $contract->status = 'active';
        $contract->visits_per_year = $request->visits_per_year;
        $contract->terms_conditions = $request->terms_conditions;
        $contract->notes = $request->notes;
        $contract->save();

        $contract->elevators()->attach($request->elevator_ids);

        return redirect()->route('maintenance-contracts.index')
            ->with('success', 'تم إنشاء عقد الصيانة بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $query = MaintenanceContract::with(['customer', 'elevators', 'invoices']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $contract = $query->findOrFail($id);

        return view('maintenance_contracts.show', compact('contract'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $query = MaintenanceContract::with('elevators');

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $contract = $query->findOrFail($id);

        $customerQuery = Customer::query();
        $elevatorQuery = Elevator::query();

        if (!$user->isSystemAdmin()) {
            $customerQuery->where('company_id', $user->company_id);
            $elevatorQuery->where('company_id', $user->company_id);
        }

        $customers = $customerQuery->get();
        $elevators = $elevatorQuery->get();

        return view('maintenance_contracts.edit', compact('contract', 'customers', 'elevators'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = MaintenanceContract::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $contract = $query->findOrFail($id);

        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'contract_number' => [
                'required',
                'string',
                'max:255',
                function ($attribute, $value, $fail) use ($user, $id) {
                    // التحقق من فريدية رقم العقد ضمن نطاق الشركة الواحدة مع استثناء العقد الحالي
                    $query = MaintenanceContract::where('contract_number', $value)
                        ->where('id', '!=', $id);
                    
                    if (!$user->isSystemAdmin()) {
                        $query->where('company_id', $user->company_id);
                    }
                    
                    if ($query->exists()) {
                        $fail('قيمة contract number مُستخدمة من قبل في نفس الشركة.');
                    }
                }
            ],
            'title' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'total_amount' => 'required|numeric|min:0',
            'billing_cycle' => 'required|in:monthly,quarterly,semi_annually,annually',
            'visits_per_year' => 'required|integer|min:1',
            'elevator_ids' => 'required|array',
            'elevator_ids.*' => 'exists:elevators,id',
        ]);

        $contract->customer_id = $request->customer_id;
        $contract->contract_number = $request->contract_number;
        $contract->title = $request->title;
        $contract->start_date = $request->start_date;
        $contract->end_date = $request->end_date;
        $contract->total_amount = $request->total_amount;
        $contract->billing_cycle = $request->billing_cycle;
        $contract->status = $request->status ?? $contract->status;
        $contract->visits_per_year = $request->visits_per_year;
        $contract->terms_conditions = $request->terms_conditions;
        $contract->notes = $request->notes;
        $contract->save();

        $contract->elevators()->sync($request->elevator_ids);

        return redirect()->route('maintenance-contracts.index')
            ->with('success', 'تم تحديث عقد الصيانة بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $query = MaintenanceContract::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $contract = $query->findOrFail($id);

        $contract->delete();

        return redirect()->route('maintenance-contracts.index')
            ->with('success', 'تم حذف عقد الصيانة بنجاح');
    }
}
