<?php

namespace App\Http\Controllers;

use App\Models\MaintenanceSchedule;
use App\Models\MaintenanceRequest;
use App\Models\MaintenanceTeam;
use Illuminate\Http\Request;

class MaintenanceCalendarController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $teams = MaintenanceTeam::all();

        return view('maintenance.calendar.index', compact('teams'));
    }

    /**
     * Get calendar events for AJAX request
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getEvents(Request $request)
    {
        $start = $request->start;
        $end = $request->end;
        $teamId = $request->team_id;

        $events = [];

        // Get maintenance schedules
        $query = MaintenanceSchedule::with(['elevator', 'team'])
            ->whereBetween('scheduled_date', [$start, $end]);

        if ($teamId) {
            $query->where('team_id', $teamId);
        }

        $schedules = $query->get();

        foreach ($schedules as $schedule) {
            $events[] = [
                'id' => 'schedule_' . $schedule->id,
                'title' => 'صيانة دورية: ' . $schedule->elevator->name,
                'start' => $schedule->scheduled_date,
                'end' => $schedule->scheduled_date,
                'color' => '#3788d8',
                'url' => route('maintenance-schedules.show', $schedule->id),
                'extendedProps' => [
                    'type' => 'schedule',
                    'status' => $schedule->status,
                    'team' => $schedule->team ? $schedule->team->name : 'غير محدد',
                    'elevator' => $schedule->elevator->name,
                    'location' => $schedule->elevator->location ? $schedule->elevator->location->name : 'غير محدد'
                ]
            ];
        }

        // Get maintenance requests
        $query = MaintenanceRequest::with(['elevator', 'assignedTo'])
            ->whereBetween('requested_date', [$start, $end]);

        if ($teamId) {
            $query->whereHas('assignedTo', function ($q) use ($teamId) {
                $q->where('team_id', $teamId);
            });
        }

        $requests = $query->get();

        foreach ($requests as $request) {
            $events[] = [
                'id' => 'request_' . $request->id,
                'title' => 'طلب صيانة: ' . $request->elevator->name,
                'start' => $request->requested_date,
                'end' => $request->requested_date,
                'color' => '#e74c3c',
                'url' => route('maintenance-requests.show', $request->id),
                'extendedProps' => [
                    'type' => 'request',
                    'status' => $request->status,
                    'priority' => $request->priority,
                    'assignedTo' => $request->assignedTo ? $request->assignedTo->name : 'غير محدد',
                    'elevator' => $request->elevator->name,
                    'location' => $request->elevator->location ? $request->elevator->location->name : 'غير محدد'
                ]
            ];
        }

        return response()->json($events);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return redirect()->route('maintenance-schedules.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        return redirect()->route('maintenance-schedules.index');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        return redirect()->route('maintenance-schedules.show', $id);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        return redirect()->route('maintenance-schedules.edit', $id);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        return redirect()->route('maintenance-schedules.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        return redirect()->route('maintenance-schedules.index');
    }
}
