<?php

namespace App\Http\Controllers;

use App\Models\LoginHistory;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LoginHistoryController extends Controller
{
    /**
     * إنشاء مثيل جديد من وحدة التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_login_history');
    }

    /**
     * عرض قائمة سجلات تسجيل الدخول.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = LoginHistory::query();

        // تطبيق الفلاتر
        if ($request->has('user_id') && $request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('login_type') && $request->login_type) {
            $query->where('login_type', $request->login_type);
        }

        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('login_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('login_at', '<=', $request->date_to);
        }

        if ($request->has('ip_address') && $request->ip_address) {
            $query->where('ip_address', 'like', '%' . $request->ip_address . '%');
        }

        // الترتيب
        $query->orderBy('login_at', 'desc');

        // الحصول على البيانات
        $loginHistory = $query->with('user')->paginate(20);

        // الحصول على قائمة المستخدمين للفلتر
        $users = User::orderBy('name')->get();

        // الحصول على قائمة الحالات للفلتر
        $statuses = LoginHistory::distinct()->pluck('status');

        // الحصول على قائمة أنواع تسجيل الدخول للفلتر
        $loginTypes = LoginHistory::distinct()->pluck('login_type');

        return view('admin.login-history.index', compact(
            'loginHistory',
            'users',
            'statuses',
            'loginTypes'
        ));
    }

    /**
     * عرض تفاصيل سجل تسجيل الدخول.
     *
     * @param \App\Models\LoginHistory $loginHistory
     * @return \Illuminate\View\View
     */
    public function show(LoginHistory $loginHistory)
    {
        return view('admin.login-history.show', compact('loginHistory'));
    }

    /**
     * حذف سجل تسجيل الدخول.
     *
     * @param \App\Models\LoginHistory $loginHistory
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(LoginHistory $loginHistory)
    {
        // التحقق من الصلاحية
        if (!Auth::user()->hasPermissionTo('delete_login_history')) {
            return back()->withErrors([
                'error' => 'ليس لديك صلاحية لحذف سجلات تسجيل الدخول.',
            ]);
        }

        $loginHistory->delete();

        return redirect()->route('login-history.index')
            ->with('success', 'تم حذف سجل تسجيل الدخول بنجاح.');
    }

    /**
     * حذف سجلات تسجيل الدخول القديمة.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deleteOld(Request $request)
    {
        // التحقق من الصلاحية
        if (!Auth::user()->hasPermissionTo('delete_login_history')) {
            return back()->withErrors([
                'error' => 'ليس لديك صلاحية لحذف سجلات تسجيل الدخول.',
            ]);
        }

        $request->validate([
            'days' => 'required|integer|min:1',
        ]);

        $date = now()->subDays($request->days);

        $count = LoginHistory::where('login_at', '<', $date)->delete();

        return redirect()->route('login-history.index')
            ->with('success', "تم حذف {$count} من سجلات تسجيل الدخول القديمة بنجاح.");
    }

    /**
     * تصدير سجلات تسجيل الدخول.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function export(Request $request)
    {
        // التحقق من الصلاحية
        if (!Auth::user()->hasPermissionTo('export_login_history')) {
            return back()->withErrors([
                'error' => 'ليس لديك صلاحية لتصدير سجلات تسجيل الدخول.',
            ]);
        }

        $query = LoginHistory::query();

        // تطبيق الفلاتر
        if ($request->has('user_id') && $request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('login_type') && $request->login_type) {
            $query->where('login_type', $request->login_type);
        }

        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('login_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('login_at', '<=', $request->date_to);
        }

        // الترتيب
        $query->orderBy('login_at', 'desc');

        // الحصول على البيانات
        $loginHistory = $query->with('user')->get();

        // تصدير البيانات (يمكن استخدام مكتبة مثل Maatwebsite/Laravel-Excel)
        // return Excel::download(new LoginHistoryExport($loginHistory), 'login_history.xlsx');

        // بديل مؤقت: تحويل البيانات إلى CSV
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="login_history.csv"',
        ];

        $columns = ['ID', 'المستخدم', 'الحالة', 'نوع تسجيل الدخول', 'عنوان IP', 'المتصفح', 'نظام التشغيل', 'الجهاز', 'التاريخ'];

        $callback = function () use ($loginHistory, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($loginHistory as $login) {
                fputcsv($file, [
                    $login->id,
                    $login->user ? $login->user->name : 'غير مسجل',
                    $login->status,
                    $login->login_type,
                    $login->ip_address,
                    $login->browser,
                    $login->operating_system,
                    $login->device,
                    $login->login_at->format('Y-m-d H:i:s'),
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
