<?php

namespace App\Http\Controllers;

use App\Models\Location;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\LocationsImport;

class LocationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $locations = Location::with(['customer'])
            ->where('company_id', $company_id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('locations.index', compact('locations'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $customers = Customer::where('company_id', $company_id)->get();

        $locationTypes = [
            'residential' => 'سكني',
            'commercial' => 'تجاري',
            'industrial' => 'صناعي',
            'government' => 'حكومي',
            'educational' => 'تعليمي',
            'healthcare' => 'رعاية صحية',
            'other' => 'أخرى'
        ];

        return view('locations.create', compact('customers', 'locationTypes'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'name' => 'required|string|max:100',
            'address' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'required|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'contact_person' => 'nullable|string|max:100',
            'contact_phone' => 'nullable|string|max:20',
            'contact_email' => 'nullable|email|max:100',
            'location_type' => 'required|string|in:residential,commercial,industrial,government,educational,healthcare,other',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();

        $location = new Location();
        $location->company_id = $user->company_id;
        $location->customer_id = $request->customer_id;
        $location->name = $request->name;
        $location->address = $request->address;
        $location->city = $request->city;
        $location->state = $request->state;
        $location->country = $request->country;
        $location->postal_code = $request->postal_code;
        $location->contact_person = $request->contact_person;
        $location->contact_phone = $request->contact_phone;
        $location->contact_email = $request->contact_email;
        $location->location_type = $request->location_type;
        $location->notes = $request->notes;
        $location->save();

        return redirect()->route('locations.index')
            ->with('success', 'تم إضافة الموقع بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $location = Location::with(['customer', 'elevators'])
            ->where('company_id', $company_id)
            ->findOrFail($id);

        $locationTypes = [
            'residential' => 'سكني',
            'commercial' => 'تجاري',
            'industrial' => 'صناعي',
            'government' => 'حكومي',
            'educational' => 'تعليمي',
            'healthcare' => 'رعاية صحية',
            'other' => 'أخرى'
        ];

        return view('locations.show', compact('location', 'locationTypes'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $location = Location::where('company_id', $company_id)->findOrFail($id);
        $customers = Customer::where('company_id', $company_id)->get();

        $locationTypes = [
            'residential' => 'سكني',
            'commercial' => 'تجاري',
            'industrial' => 'صناعي',
            'government' => 'حكومي',
            'educational' => 'تعليمي',
            'healthcare' => 'رعاية صحية',
            'other' => 'أخرى'
        ];

        return view('locations.edit', compact('location', 'customers', 'locationTypes'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'name' => 'required|string|max:100',
            'address' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'required|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'contact_person' => 'nullable|string|max:100',
            'contact_phone' => 'nullable|string|max:20',
            'contact_email' => 'nullable|email|max:100',
            'location_type' => 'required|string|in:residential,commercial,industrial,government,educational,healthcare,other',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company_id = $user->company_id;

        $location = Location::where('company_id', $company_id)->findOrFail($id);
        $location->customer_id = $request->customer_id;
        $location->name = $request->name;
        $location->address = $request->address;
        $location->city = $request->city;
        $location->state = $request->state;
        $location->country = $request->country;
        $location->postal_code = $request->postal_code;
        $location->contact_person = $request->contact_person;
        $location->contact_phone = $request->contact_phone;
        $location->contact_email = $request->contact_email;
        $location->location_type = $request->location_type;
        $location->notes = $request->notes;
        $location->save();

        return redirect()->route('locations.index')
            ->with('success', 'تم تحديث بيانات الموقع بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company_id = $user->company_id;

        $location = Location::where('company_id', $company_id)->findOrFail($id);

        // Check if location has elevators
        if ($location->elevators()->count() > 0) {
            return redirect()->route('locations.index')
                ->with('error', 'لا يمكن حذف الموقع لأنه مرتبط بمصاعد');
        }

        $location->delete();

        return redirect()->route('locations.index')
            ->with('success', 'تم حذف الموقع بنجاح');
    }

    /**
     * Show the form for importing locations from Excel.
     *
     * @return \Illuminate\Http\Response
     */
    public function importForm()
    {
        return view('locations.import');
    }

    /**
     * Import locations from Excel file.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls,csv|max:10240',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $import = new LocationsImport;
            Excel::import($import, $request->file('file'));

            $message = "تم استيراد {$import->importedCount} موقع بنجاح.";
            if ($import->duplicateCount > 0) {
                $message .= " تم تجاهل {$import->duplicateCount} موقع مكرر.";
            }
            if ($import->skippedCount > 0) {
                $message .= " تم تخطي {$import->skippedCount} صف لعدم اكتمال البيانات أو عدم العثور على العميل.";
            }

            return redirect()->route('locations.index')
                ->with('success', $message);
        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errorMsgs = [];
            foreach ($failures as $failure) {
                $errorMsgs[] = 'Row ' . $failure->row() . ': ' . implode(', ', $failure->errors());
            }
            return redirect()->back()
                ->with('error', 'فشل التحقق من صحة بعض الصفوف في الملف.')
                ->withErrors($errorMsgs)
                ->withInput();
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء استيراد المواقع: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Download location import template in Excel format.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set headers
        $headers = [
            'name',
            'customer_id',
            'customer_name',
            'branch_id',
            'address',
            'city',
            'state',
            'country',
            'postal_code',
            'contact_person',
            'contact_phone',
            'contact_email',
            'location_type',
            'google_maps_link',
            'notes'
        ];

        // Add headers to the first row
        foreach ($headers as $index => $header) {
            $sheet->setCellValueByColumnAndRow($index + 1, 1, $header);
        }

        // Add sample data
        $sampleData = [
            ['Main Office', 1, 'Sample Customer', 1, '123 Street Name', 'Riyadh', 'Riyadh Region', 'Saudi Arabia', '12345', 'John Doe', '0501234567', 'john@example.com', 'commercial', 'https://maps.app.goo.gl/sample1', 'Sample notes'],
            ['Branch A', null, 'Sample Customer', 1, '456 Avenue', 'Jeddah', 'Makkah Region', 'Saudi Arabia', '54321', 'Jane Smith', '0507654321', 'jane@example.com', 'residential', 'https://maps.app.goo.gl/sample2', 'Other notes']
        ];

        // Add sample data starting from row 2
        foreach ($sampleData as $rowIndex => $rowData) {
            foreach ($rowData as $columnIndex => $value) {
                $sheet->setCellValueByColumnAndRow($columnIndex + 1, $rowIndex + 2, $value);
            }
        }

        // Create a temporary file
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $tempFile = tempnam(sys_get_temp_dir(), 'locations_import_template');
        $writer->save($tempFile);

        // Return the file as a download
        return response()->download(
            $tempFile,
            'locations_import_template.xlsx',
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => 'attachment; filename="locations_import_template.xlsx"',
            ]
        )->deleteFileAfterSend(true);
    }
}
