<?php

namespace App\Http\Controllers;

use App\Models\JournalTemplate;
use App\Models\JournalTemplateItem;
use App\Models\ChartOfAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class JournalTemplateController extends Controller
{
    /**
     * عرض قائمة قوالب القيود المحاسبية
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $templates = JournalTemplate::where('company_id', Auth::user()->company_id)
            ->with('creator')
            ->orderBy('name')
            ->paginate(15);

        return view('accounting.journal_templates.index', compact('templates'));
    }

    /**
     * عرض نموذج إنشاء قالب قيد محاسبي جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $accounts = ChartOfAccount::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('account_code')
            ->get();

        return view('accounting.journal_templates.create', compact('accounts'));
    }

    /**
     * حفظ قالب قيد محاسبي جديد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:journal_templates,code',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'items' => 'required|array|min:2',
            'items.*.account_id' => 'required|exists:chart_of_accounts,id',
            'items.*.type' => 'required|in:debit,credit',
            'items.*.percentage' => 'required_without:items.*.fixed_amount|nullable|numeric|min:0|max:100',
            'items.*.fixed_amount' => 'required_without:items.*.percentage|nullable|numeric|min:0',
            'items.*.description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من توازن النسب المئوية
        $totalDebitPercentage = 0;
        $totalCreditPercentage = 0;
        $totalDebitFixedAmount = 0;
        $totalCreditFixedAmount = 0;
        $hasPercentage = false;
        $hasFixedAmount = false;

        foreach ($request->items as $item) {
            if (isset($item['percentage']) && $item['percentage'] > 0) {
                $hasPercentage = true;
                if ($item['type'] == 'debit') {
                    $totalDebitPercentage += floatval($item['percentage']);
                } else {
                    $totalCreditPercentage += floatval($item['percentage']);
                }
            }

            if (isset($item['fixed_amount']) && $item['fixed_amount'] > 0) {
                $hasFixedAmount = true;
                if ($item['type'] == 'debit') {
                    $totalDebitFixedAmount += floatval($item['fixed_amount']);
                } else {
                    $totalCreditFixedAmount += floatval($item['fixed_amount']);
                }
            }
        }

        if ($hasPercentage && abs($totalDebitPercentage - $totalCreditPercentage) > 0.01) {
            return redirect()->back()
                ->with('error', 'مجموع النسب المئوية للبنود المدينة والدائنة يجب أن يكون متساوياً. مدين: ' . $totalDebitPercentage . '%, دائن: ' . $totalCreditPercentage . '%')
                ->withInput();
        }

        if ($hasFixedAmount && abs($totalDebitFixedAmount - $totalCreditFixedAmount) > 0.01) {
            return redirect()->back()
                ->with('error', 'مجموع المبالغ الثابتة للبنود المدينة والدائنة يجب أن يكون متساوياً. مدين: ' . $totalDebitFixedAmount . ', دائن: ' . $totalCreditFixedAmount)
                ->withInput();
        }

        DB::beginTransaction();

        try {
            // إنشاء قالب القيد المحاسبي
            $template = new JournalTemplate();
            $template->company_id = Auth::user()->company_id;
            $template->name = $request->name;
            $template->code = $request->code;
            $template->description = $request->description;
            $template->is_active = filter_var($request->input('is_active', false), FILTER_VALIDATE_BOOLEAN);
            $template->created_by = Auth::id();
            $template->save();

            // إنشاء بنود القالب
            foreach ($request->items as $itemData) {
                $item = new JournalTemplateItem();
                $item->journal_template_id = $template->id;
                $item->account_id = $itemData['account_id'];
                $item->type = $itemData['type'];
                $item->percentage = isset($itemData['percentage']) ? floatval($itemData['percentage']) : null;
                $item->fixed_amount = isset($itemData['fixed_amount']) ? floatval($itemData['fixed_amount']) : null;
                $item->description = $itemData['description'] ?? null;
                $item->save();
            }

            DB::commit();

            return redirect()->route('journal-templates.show', $template->id)
                ->with('success', 'تم إنشاء قالب القيد المحاسبي بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء قالب القيد المحاسبي: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * عرض تفاصيل قالب قيد محاسبي محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $template = JournalTemplate::where('company_id', Auth::user()->company_id)
            ->with(['items.account', 'creator'])
            ->findOrFail($id);

        return view('accounting.journal_templates.show', compact('template'));
    }

    /**
     * عرض نموذج تعديل قالب قيد محاسبي
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $template = JournalTemplate::where('company_id', Auth::user()->company_id)
            ->with('items.account')
            ->findOrFail($id);

        $accounts = ChartOfAccount::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('account_code')
            ->get();

        return view('accounting.journal_templates.edit', compact('template', 'accounts'));
    }

    /**
     * تحديث قالب قيد محاسبي محدد
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $template = JournalTemplate::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:journal_templates,code,' . $id,
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'items' => 'required|array|min:2',
            'items.*.account_id' => 'required|exists:chart_of_accounts,id',
            'items.*.type' => 'required|in:debit,credit',
            'items.*.percentage' => 'required_without:items.*.fixed_amount|nullable|numeric|min:0|max:100',
            'items.*.fixed_amount' => 'required_without:items.*.percentage|nullable|numeric|min:0',
            'items.*.description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من توازن النسب المئوية
        $totalDebitPercentage = 0;
        $totalCreditPercentage = 0;
        $totalDebitFixedAmount = 0;
        $totalCreditFixedAmount = 0;
        $hasPercentage = false;
        $hasFixedAmount = false;

        foreach ($request->items as $item) {
            if (isset($item['percentage']) && $item['percentage'] > 0) {
                $hasPercentage = true;
                if ($item['type'] == 'debit') {
                    $totalDebitPercentage += floatval($item['percentage']);
                } else {
                    $totalCreditPercentage += floatval($item['percentage']);
                }
            }

            if (isset($item['fixed_amount']) && $item['fixed_amount'] > 0) {
                $hasFixedAmount = true;
                if ($item['type'] == 'debit') {
                    $totalDebitFixedAmount += floatval($item['fixed_amount']);
                } else {
                    $totalCreditFixedAmount += floatval($item['fixed_amount']);
                }
            }
        }

        if ($hasPercentage && abs($totalDebitPercentage - $totalCreditPercentage) > 0.01) {
            return redirect()->back()
                ->with('error', 'مجموع النسب المئوية للبنود المدينة والدائنة يجب أن يكون متساوياً. مدين: ' . $totalDebitPercentage . '%, دائن: ' . $totalCreditPercentage . '%')
                ->withInput();
        }

        if ($hasFixedAmount && abs($totalDebitFixedAmount - $totalCreditFixedAmount) > 0.01) {
            return redirect()->back()
                ->with('error', 'مجموع المبالغ الثابتة للبنود المدينة والدائنة يجب أن يكون متساوياً. مدين: ' . $totalDebitFixedAmount . ', دائن: ' . $totalCreditFixedAmount)
                ->withInput();
        }

        DB::beginTransaction();

        try {
            // تحديث قالب القيد المحاسبي
            $template->name = $request->name;
            $template->code = $request->code;
            $template->description = $request->description;
            $template->is_active = filter_var($request->input('is_active', false), FILTER_VALIDATE_BOOLEAN);
            $template->save();

            // حذف البنود الحالية
            $template->items()->delete();

            // إنشاء البنود الجديدة
            foreach ($request->items as $itemData) {
                $item = new JournalTemplateItem();
                $item->journal_template_id = $template->id;
                $item->account_id = $itemData['account_id'];
                $item->type = $itemData['type'];
                $item->percentage = isset($itemData['percentage']) ? floatval($itemData['percentage']) : null;
                $item->fixed_amount = isset($itemData['fixed_amount']) ? floatval($itemData['fixed_amount']) : null;
                $item->description = $itemData['description'] ?? null;
                $item->save();
            }

            DB::commit();

            return redirect()->route('journal-templates.show', $template->id)
                ->with('success', 'تم تحديث قالب القيد المحاسبي بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث قالب القيد المحاسبي: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * حذف قالب قيد محاسبي محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $template = JournalTemplate::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        DB::beginTransaction();

        try {
            // حذف البنود
            $template->items()->delete();

            // حذف القالب
            $template->delete();

            DB::commit();

            return redirect()->route('journal-templates.index')
                ->with('success', 'تم حذف قالب القيد المحاسبي بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف قالب القيد المحاسبي: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء قيد محاسبي من قالب
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function createEntry($id)
    {
        $template = JournalTemplate::where('company_id', Auth::user()->company_id)
            ->with('items.account')
            ->findOrFail($id);

        if (!$template->is_active) {
            return redirect()->route('journal-templates.show', $template->id)
                ->with('error', 'لا يمكن إنشاء قيد محاسبي من قالب غير نشط');
        }

        if (!$template->isValid()) {
            return redirect()->route('journal-templates.show', $template->id)
                ->with('error', 'القالب غير صالح لإنشاء قيد محاسبي');
        }

        return view('accounting.journal_templates.create_entry', compact('template'));
    }

    /**
     * حفظ قيد محاسبي من قالب
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function storeEntry(Request $request, $id)
    {
        $template = JournalTemplate::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'entry_date' => 'required|date',
            'description' => 'nullable|string',
            'reference_type' => 'nullable|string|max:255',
            'reference_id' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $entry = JournalEntry::createFromTemplate($template->id, [
                'company_id' => Auth::user()->company_id,
                'amount' => $request->amount,
                'entry_date' => $request->entry_date,
                'description' => $request->description,
                'reference_type' => $request->reference_type,
                'reference_id' => $request->reference_id,
                'created_by' => Auth::id()
            ]);

            return redirect()->route('journal-entries.show', $entry->id)
                ->with('success', 'تم إنشاء القيد المحاسبي من القالب بنجاح');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء القيد المحاسبي: ' . $e->getMessage())
                ->withInput();
        }
    }
}
