<?php

namespace App\Http\Controllers;

use App\Models\InvoiceTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class InvoiceTemplateController extends Controller
{
    /**
     * إنشاء مثيل جديد من وحدة التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:invoice_templates.view')->only(['index', 'show', 'preview']);
        $this->middleware('permission:invoice_templates.create')->only(['create', 'store']);
        $this->middleware('permission:invoice_templates.edit')->only(['edit', 'update']);
        $this->middleware('permission:invoice_templates.delete')->only('destroy');
    }
    /**
     * عرض قائمة قوالب الفواتير.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $companyId = auth()->user()->company_id;
        $templates = InvoiceTemplate::where('company_id', $companyId)->paginate(10);

        return view('invoice_templates.index', compact('templates'));
    }

    /**
     * عرض نموذج إنشاء قالب فاتورة جديد.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('invoice_templates.create');
    }

    /**
     * تخزين قالب فاتورة جديد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'language' => 'required|string|in:ar,en',
            'header_html' => 'nullable|string',
            'footer_html' => 'nullable|string',
            'css_styles' => 'nullable|string',
            'is_default' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;

        // إذا كان القالب الجديد هو الافتراضي، نقوم بإلغاء تعيين القوالب الافتراضية الأخرى
        if ($request->input('is_default')) {
            InvoiceTemplate::where('company_id', $companyId)
                ->where('language', $request->input('language'))
                ->update(['is_default' => false]);
        }

        $template = InvoiceTemplate::create([
            'company_id' => $companyId,
            'name' => $request->input('name'),
            'language' => $request->input('language'),
            'header_html' => $request->input('header_html'),
            'footer_html' => $request->input('footer_html'),
            'css_styles' => $request->input('css_styles'),
            'is_default' => $request->input('is_default', false),
            'is_active' => true,
        ]);

        return redirect()->route('invoice-templates.index')
            ->with('success', 'تم إنشاء قالب الفاتورة بنجاح.');
    }

    /**
     * عرض قالب فاتورة محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $companyId = auth()->user()->company_id;
        $template = InvoiceTemplate::where('company_id', $companyId)
            ->findOrFail($id);

        return view('invoice_templates.show', compact('template'));
    }

    /**
     * عرض نموذج تعديل قالب فاتورة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = auth()->user()->company_id;
        $template = InvoiceTemplate::where('company_id', $companyId)
            ->findOrFail($id);

        return view('invoice_templates.edit', compact('template'));
    }

    /**
     * تحديث قالب فاتورة محدد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'language' => 'required|string|in:ar,en',
            'header_html' => 'nullable|string',
            'footer_html' => 'nullable|string',
            'css_styles' => 'nullable|string',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $template = InvoiceTemplate::where('company_id', $companyId)
            ->findOrFail($id);

        // إذا كان القالب المحدث هو الافتراضي، نقوم بإلغاء تعيين القوالب الافتراضية الأخرى
        if ($request->input('is_default') && !$template->is_default) {
            InvoiceTemplate::where('company_id', $companyId)
                ->where('language', $request->input('language'))
                ->where('id', '!=', $id)
                ->update(['is_default' => false]);
        }

        $template->update([
            'name' => $request->input('name'),
            'language' => $request->input('language'),
            'header_html' => $request->input('header_html'),
            'footer_html' => $request->input('footer_html'),
            'css_styles' => $request->input('css_styles'),
            'is_default' => $request->input('is_default', false),
            'is_active' => $request->input('is_active', true),
        ]);

        return redirect()->route('invoice-templates.index')
            ->with('success', 'تم تحديث قالب الفاتورة بنجاح.');
    }

    /**
     * حذف قالب فاتورة محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = auth()->user()->company_id;
        $template = InvoiceTemplate::where('company_id', $companyId)
            ->findOrFail($id);

        // التحقق من عدم استخدام القالب في أي فواتير
        $invoicesCount = $template->invoices()->count();

        if ($invoicesCount > 0) {
            return redirect()->route('invoice-templates.index')
                ->with('error', 'لا يمكن حذف القالب لأنه مستخدم في ' . $invoicesCount . ' فاتورة.');
        }

        $template->delete();

        return redirect()->route('invoice-templates.index')
            ->with('success', 'تم حذف قالب الفاتورة بنجاح.');
    }

    /**
     * معاينة قالب فاتورة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function preview($id)
    {
        $companyId = auth()->user()->company_id;
        $template = InvoiceTemplate::where('company_id', $companyId)
            ->findOrFail($id);

        // إنشاء بيانات وهمية للمعاينة
        $dummyData = $this->getDummyDataForPreview();

        // إنشاء HTML للمعاينة
        $html = $this->generatePreviewHtml($template, $dummyData);

        return response($html);
    }

    /**
     * الحصول على بيانات وهمية لمعاينة قالب الفاتورة.
     *
     * @return array
     */
    protected function getDummyDataForPreview()
    {
        $company = auth()->user()->company;

        return [
            'company' => [
                'name' => $company->name,
                'address' => $company->address ?? 'عنوان الشركة',
                'phone' => $company->phone ?? '0123456789',
                'email' => $company->email ?? 'info@company.com',
            ],
            'customer' => [
                'name' => 'عميل نموذجي',
                'address' => 'عنوان العميل',
                'phone' => '0123456789',
                'email' => 'customer@example.com',
            ],
            'invoice' => [
                'invoice_number' => 'INV-12345',
                'invoice_date' => date('Y-m-d'),
                'due_date' => date('Y-m-d', strtotime('+30 days')),
                'status' => 'draft',
                'subtotal' => 1000.00,
                'tax_rate' => 15.00,
                'tax_amount' => 150.00,
                'discount_percentage' => 0,
                'discount_amount' => 0,
                'total_amount' => 1150.00,
                'paid_amount' => 0,
                'due_amount' => 1150.00,
                'notes' => 'ملاحظات الفاتورة',
                'terms_conditions' => 'الشروط والأحكام',
            ],
            'items' => [
                [
                    'description' => 'خدمة صيانة المصاعد',
                    'quantity' => 1,
                    'unit_price' => 1000.00,
                    'discount_percentage' => 0,
                    'discount_amount' => 0,
                    'tax_percentage' => 15.00,
                    'tax_amount' => 150.00,
                    'subtotal' => 1000.00,
                    'total' => 1150.00,
                ],
            ],
        ];
    }

    /**
     * إنشاء HTML لمعاينة قالب الفاتورة.
     *
     * @param  InvoiceTemplate  $template
     * @param  array  $data
     * @return string
     */
    protected function generatePreviewHtml($template, $data)
    {
        $headerHtml = $template->header_html;
        $footerHtml = $template->footer_html;
        $cssStyles = $template->css_styles;

        // بناء محتوى الفاتورة
        $itemsHtml = '';
        foreach ($data['items'] as $item) {
            $itemsHtml .= '<tr>
                <td>' . $item['description'] . '</td>
                <td>' . $item['quantity'] . '</td>
                <td>' . number_format($item['unit_price'], 2) . '</td>
                <td>' . number_format($item['discount_amount'], 2) . '</td>
                <td>' . number_format($item['tax_amount'], 2) . '</td>
                <td>' . number_format($item['total'], 2) . '</td>
            </tr>';
        }

        $html = '<!DOCTYPE html>
        <html lang="' . $template->language . '" dir="' . ($template->language === 'ar' ? 'rtl' : 'ltr') . '">
        <head>
            <meta charset="UTF-8">
            <title>معاينة قالب الفاتورة</title>
            <style>' . $cssStyles . '</style>
        </head>
        <body>
            ' . $headerHtml . '
            
            <div class="invoice-info">
                <div class="company-info">
                    <h2>' . $data['company']['name'] . '</h2>
                    <p>' . $data['company']['address'] . '</p>
                    <p>' . $data['company']['phone'] . '</p>
                    <p>' . $data['company']['email'] . '</p>
                </div>
                
                <div class="customer-info">
                    <h3>العميل</h3>
                    <p>' . $data['customer']['name'] . '</p>
                    <p>' . $data['customer']['address'] . '</p>
                    <p>' . $data['customer']['phone'] . '</p>
                    <p>' . $data['customer']['email'] . '</p>
                </div>
                
                <div class="invoice-details">
                    <p><strong>رقم الفاتورة:</strong> ' . $data['invoice']['invoice_number'] . '</p>
                    <p><strong>تاريخ الفاتورة:</strong> ' . $data['invoice']['invoice_date'] . '</p>
                    <p><strong>تاريخ الاستحقاق:</strong> ' . $data['invoice']['due_date'] . '</p>
                    <p><strong>حالة الفاتورة:</strong> ' . $data['invoice']['status'] . '</p>
                </div>
            </div>
            
            <div class="invoice-items">
                <table border="1" cellspacing="0" cellpadding="5" width="100%">
                    <thead>
                        <tr>
                            <th>الوصف</th>
                            <th>الكمية</th>
                            <th>سعر الوحدة</th>
                            <th>الخصم</th>
                            <th>الضريبة</th>
                            <th>الإجمالي</th>
                        </tr>
                    </thead>
                    <tbody>
                        ' . $itemsHtml . '
                    </tbody>
                    <tfoot>
                        <tr>
                            <td colspan="5" align="right"><strong>المجموع الفرعي:</strong></td>
                            <td>' . number_format($data['invoice']['subtotal'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" align="right"><strong>الخصم:</strong></td>
                            <td>' . number_format($data['invoice']['discount_amount'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" align="right"><strong>ضريبة القيمة المضافة (' . $data['invoice']['tax_rate'] . '%):</strong></td>
                            <td>' . number_format($data['invoice']['tax_amount'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" align="right"><strong>الإجمالي:</strong></td>
                            <td>' . number_format($data['invoice']['total_amount'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" align="right"><strong>المدفوع:</strong></td>
                            <td>' . number_format($data['invoice']['paid_amount'], 2) . '</td>
                        </tr>
                        <tr>
                            <td colspan="5" align="right"><strong>المتبقي:</strong></td>
                            <td>' . number_format($data['invoice']['due_amount'], 2) . '</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
            
            <div class="invoice-notes">
                <h3>ملاحظات</h3>
                <p>' . $data['invoice']['notes'] . '</p>
            </div>
            
            <div class="invoice-terms">
                <h3>الشروط والأحكام</h3>
                <p>' . $data['invoice']['terms_conditions'] . '</p>
            </div>
            
            ' . $footerHtml . '
        </body>
        </html>';

        return $html;
    }
}
