<?php

namespace App\Http\Controllers;

use App\Models\InvoiceInstallment;
use App\Models\Invoice;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class InvoiceInstallmentController extends Controller
{
    /**
     * عرض قائمة أقساط الفواتير.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $companyId = auth()->user()->company_id;

        $installments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->paginate(15);

        return view('invoice_installments.index', compact('installments'));
    }

    /**
     * عرض نموذج إنشاء قسط فاتورة جديد.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $companyId = auth()->user()->company_id;
        $invoices = Invoice::where('company_id', $companyId)
            ->whereIn('status', ['draft', 'sent', 'partially_paid'])
            ->with('customer')
            ->get();

        return view('invoice_installments.create', compact('invoices'));
    }

    /**
     * تخزين قسط فاتورة جديد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_id' => 'required|exists:invoices,id',
            'amount' => 'required|numeric|min:0',
            'due_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $invoice = Invoice::where('company_id', $companyId)
            ->findOrFail($request->input('invoice_id'));

        $installment = InvoiceInstallment::create([
            'invoice_id' => $invoice->id,
            'amount' => $request->input('amount'),
            'due_date' => $request->input('due_date'),
            'status' => 'pending',
            'notes' => $request->input('notes'),
        ]);

        return redirect()->route('invoice-installments.index')
            ->with('success', 'تم إنشاء قسط الفاتورة بنجاح.');
    }

    /**
     * عرض قسط فاتورة محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $companyId = auth()->user()->company_id;
        $installment = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->with(['invoice', 'invoice.customer', 'payments', 'payments.paymentMethod', 'payments.createdBy'])
            ->findOrFail($id);

        // الحصول على طرق الدفع المتاحة
        $paymentMethods = \App\Models\PaymentMethod::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('invoice_installments.show', compact('installment', 'paymentMethods'));
    }

    /**
     * عرض نموذج تعديل قسط فاتورة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = auth()->user()->company_id;
        $installment = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->findOrFail($id);

        $invoices = Invoice::where('company_id', $companyId)
            ->whereIn('status', ['draft', 'sent', 'partially_paid'])
            ->with('customer')
            ->get();

        return view('invoice_installments.edit', compact('installment', 'invoices'));
    }

    /**
     * تحديث قسط فاتورة محدد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'invoice_id' => 'required|exists:invoices,id',
            'amount' => 'required|numeric|min:0',
            'due_date' => 'required|date',
            'status' => 'required|in:pending,paid,overdue,cancelled',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $installment = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->findOrFail($id);

        $installment->update([
            'invoice_id' => $request->input('invoice_id'),
            'amount' => $request->input('amount'),
            'due_date' => $request->input('due_date'),
            'status' => $request->input('status'),
            'notes' => $request->input('notes'),
        ]);

        return redirect()->route('invoice-installments.index')
            ->with('success', 'تم تحديث قسط الفاتورة بنجاح.');
    }

    /**
     * حذف قسط فاتورة محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = auth()->user()->company_id;
        $installment = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->findOrFail($id);

        // التحقق من عدم وجود مدفوعات مرتبطة بالقسط
        $paymentsCount = $installment->payments()->count();

        if ($paymentsCount > 0) {
            return redirect()->route('invoice-installments.index')
                ->with('error', 'لا يمكن حذف القسط لأنه مرتبط بـ ' . $paymentsCount . ' مدفوعات.');
        }

        $installment->delete();

        return redirect()->route('invoice-installments.index')
            ->with('success', 'تم حذف قسط الفاتورة بنجاح.');
    }

    /**
     * عرض لوحة معلومات الأقساط.
     *
     * @return \Illuminate\Http\Response
     */
    public function dashboard()
    {
        $companyId = auth()->user()->company_id;

        // إحصائيات الأقساط
        $stats = [
            'total' => 0,
            'paid' => 0,
            'pending' => 0,
            'overdue' => 0
        ];

        // الحصول على جميع الأقساط للشركة الحالية
        $allInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })->get();

        $stats['total'] = $allInstallments->count();
        $stats['paid'] = $allInstallments->filter(function ($installment) {
            return $installment->isPaid();
        })->count();
        $stats['pending'] = $allInstallments->filter(function ($installment) {
            return !$installment->isPaid() && !$installment->isOverdue();
        })->count();
        $stats['overdue'] = $allInstallments->filter(function ($installment) {
            return $installment->isOverdue();
        })->count();

        // الأقساط المستحقة خلال الأسبوع القادم
        $nextWeek = Carbon::now()->addDays(7);
        $upcomingInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where('status', 'pending')
            ->where('due_date', '<=', $nextWeek)
            ->where('due_date', '>=', Carbon::now())
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->limit(10)
            ->get();

        // الأقساط المتأخرة
        $overdueInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where(function ($query) {
                $query->where('status', 'overdue')
                    ->orWhere(function ($q) {
                        $q->where('status', 'pending')
                            ->where('due_date', '<', Carbon::now());
                    });
            })
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->limit(10)
            ->get();

        // بيانات الرسم البياني للأقساط حسب الشهر
        $sixMonthsAgo = Carbon::now()->subMonths(5)->startOfMonth();
        $installmentsByMonth = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where('due_date', '>=', $sixMonthsAgo)
            ->select(DB::raw('YEAR(due_date) as year'), DB::raw('MONTH(due_date) as month'), DB::raw('SUM(amount) as total_amount'))
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        $chartData = [
            'months' => [],
            'amounts' => []
        ];

        // إنشاء مصفوفة للأشهر الستة الماضية
        for ($i = 0; $i < 6; $i++) {
            $month = Carbon::now()->subMonths(5 - $i);
            $chartData['months'][] = $month->format('M Y');

            // البحث عن المبلغ لهذا الشهر
            $monthData = $installmentsByMonth->first(function ($item) use ($month) {
                return $item->year == $month->year && $item->month == $month->month;
            });

            $chartData['amounts'][] = $monthData ? $monthData->total_amount : 0;
        }

        return view('invoice_installments.dashboard', compact('stats', 'upcomingInstallments', 'overdueInstallments', 'chartData'));
    }

    /**
     * تسجيل دفعة لقسط فاتورة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function recordPayment(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_method_id' => 'required|exists:payment_methods,id',
            'reference_number' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $installment = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->with('invoice')
            ->findOrFail($id);

        // التحقق من أن المبلغ لا يتجاوز المبلغ المتبقي للقسط
        $remainingAmount = $installment->amount - $installment->paid_amount;

        if ($request->input('amount') > $remainingAmount) {
            return redirect()->back()
                ->with('error', 'مبلغ الدفعة يتجاوز المبلغ المتبقي للقسط (' . number_format($remainingAmount, 2) . ').')
                ->withInput();
        }

        DB::beginTransaction();

        try {
            // إنشاء سجل الدفعة
            $payment = new Payment([
                'company_id' => $companyId,
                'invoice_id' => $installment->invoice_id,
                'payment_method_id' => $request->input('payment_method_id'),
                'amount' => $request->input('amount'),
                'payment_date' => $request->input('payment_date'),
                'reference_number' => $request->input('reference_number'),
                'notes' => $request->input('notes') ?: 'دفعة للقسط رقم ' . $installment->id,
                'status' => 'completed',
                'created_by' => auth()->id(),
            ]);

            // ربط الدفعة بالقسط
            $installment->payments()->save($payment);

            // تحديث المبلغ المدفوع للقسط
            $installment->paid_amount += $request->input('amount');

            // تحديث حالة القسط
            if ($installment->paid_amount >= $installment->amount) {
                $installment->status = 'paid';
            } elseif ($installment->status === 'overdue') {
                $installment->status = 'partially_paid';
            }

            $installment->save();

            // تحديث المبالغ المدفوعة والمستحقة للفاتورة
            $invoice = $installment->invoice;
            $invoice->paid_amount += $request->input('amount');
            $invoice->due_amount -= $request->input('amount');

            // تحديث حالة الفاتورة
            if ($invoice->due_amount <= 0) {
                $invoice->status = 'paid';
            } else {
                $invoice->status = 'partially_paid';
            }

            $invoice->save();

            DB::commit();

            return redirect()->route('invoice-installments.show', $installment->id)
                ->with('success', 'تم تسجيل الدفعة بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تسجيل الدفعة: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * إرسال تذكير بالقسط.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function sendReminder(Request $request, $id)
    {
        $companyId = auth()->user()->company_id;
        $installment = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->with(['invoice', 'invoice.customer'])
            ->findOrFail($id);

        try {
            // إرسال تذكير عبر البريد الإلكتروني
            // يمكن استخدام مكتبة البريد الإلكتروني في لارافيل
            // Mail::to($installment->invoice->customer->email)->send(new InstallmentReminderMail($installment));

            // تسجيل التذكير في سجل الإشعارات
            \App\Models\NotificationLog::create([
                'user_id' => auth()->id(),
                'notification_type' => 'installment_reminder',
                'title' => 'تذكير بقسط مستحق',
                'content' => 'تم إرسال تذكير للعميل ' . $installment->invoice->customer->name . ' بخصوص القسط المستحق بتاريخ ' . $installment->due_date->format('Y-m-d'),
                'related_id' => $installment->id,
                'related_type' => 'App\\Models\\InvoiceInstallment',
                'status' => 'sent',
            ]);

            return redirect()->back()->with('success', 'تم إرسال التذكير بنجاح.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'حدث خطأ أثناء إرسال التذكير: ' . $e->getMessage());
        }
    }

    /**
     * تقرير الأقساط المتأخرة (النسخة القديمة).
     *
     * @return \Illuminate\Http\Response
     */
    public function overdueReportOld()
    {
        $companyId = auth()->user()->company_id;

        $overdueInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where(function ($query) {
                $query->where('status', 'overdue')
                    ->orWhere(function ($q) {
                        $q->where('status', 'pending')
                            ->where('due_date', '<', Carbon::now());
                    });
            })
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->paginate(20);

        // إحصائيات الأقساط المتأخرة
        $totalOverdueAmount = $overdueInstallments->sum(function ($installment) {
            return $installment->amount - $installment->paid_amount;
        });

        $stats = [
            'count' => $overdueInstallments->total(),
            'total_amount' => $totalOverdueAmount,
        ];

        return view('invoice_installments.overdue_report', compact('overdueInstallments', 'stats'));
    }

    /**
     * تقرير الأقساط المستحقة قريباً (النسخة القديمة).
     *
     * @return \Illuminate\Http\Response
     */
    public function upcomingReportOld()
    {
        $companyId = auth()->user()->company_id;

        $upcomingInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where('status', 'pending')
            ->where('due_date', '>=', Carbon::now())
            ->where('due_date', '<=', Carbon::now()->addDays(30))
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->paginate(20);

        // إحصائيات الأقساط المستحقة قريباً
        $totalUpcomingAmount = $upcomingInstallments->sum('amount');

        $stats = [
            'count' => $upcomingInstallments->total(),
            'total_amount' => $totalUpcomingAmount,
        ];

        return view('invoice_installments.upcoming_report', compact('upcomingInstallments', 'stats'));
    }

    /**
     * تحديث حالة الأقساط المتأخرة يدوياً.
     *
     * @return \Illuminate\Http\Response
     */
    public function updateOverdue()
    {
        $companyId = auth()->user()->company_id;

        try {
            DB::beginTransaction();

            // الحصول على الأقساط المتأخرة التي لم يتم تحديثها بعد
            $overdueInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
                ->where('status', 'pending')
                ->where('due_date', '<', Carbon::now()->startOfDay())
                ->where('paid_amount', '<', DB::raw('amount'))
                ->get();

            $count = $overdueInstallments->count();

            foreach ($overdueInstallments as $installment) {
                // تحديث حالة القسط إلى متأخر
                $installment->status = 'overdue';
                $installment->save();

                // تحديث حالة الفاتورة إذا لزم الأمر
                $invoice = $installment->invoice;
                if ($invoice && $invoice->status !== 'overdue' && $invoice->due_amount > 0) {
                    $invoice->status = 'overdue';
                    $invoice->save();
                }
            }

            DB::commit();

            return redirect()->back()->with('success', "تم تحديث {$count} قسط متأخر بنجاح.");
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث الأقساط المتأخرة: ' . $e->getMessage());
        }
    }

    /**
     * عرض صفحة تسجيل دفعات الأقساط.
     *
     * @return \Illuminate\Http\Response
     */
    public function payments()
    {
        $companyId = auth()->user()->company_id;

        $pendingInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->whereIn('status', ['pending', 'overdue', 'partially_paid'])
            ->where('paid_amount', '<', DB::raw('amount'))
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->paginate(15);

        // الحصول على طرق الدفع المتاحة
        $paymentMethods = \App\Models\PaymentMethod::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('invoice_installments.payments', compact('pendingInstallments', 'paymentMethods'));
    }

    /**
     * عرض صفحة تقارير الأقساط (النسخة القديمة).
     *
     * @return \Illuminate\Http\Response
     */
    public function reportsOld()
    {
        $companyId = auth()->user()->company_id;

        // إحصائيات الأقساط
        $stats = [
            'total_count' => 0,
            'total_amount' => 0,
            'paid_count' => 0,
            'paid_amount' => 0,
            'pending_count' => 0,
            'pending_amount' => 0,
            'overdue_count' => 0,
            'overdue_amount' => 0,
        ];

        // الحصول على جميع الأقساط للشركة الحالية
        $allInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })->get();

        $stats['total_count'] = $allInstallments->count();
        $stats['total_amount'] = $allInstallments->sum('amount');

        $paidInstallments = $allInstallments->filter(function ($installment) {
            return $installment->isPaid();
        });
        $stats['paid_count'] = $paidInstallments->count();
        $stats['paid_amount'] = $paidInstallments->sum('amount');

        $pendingInstallments = $allInstallments->filter(function ($installment) {
            return !$installment->isPaid() && !$installment->isOverdue();
        });
        $stats['pending_count'] = $pendingInstallments->count();
        $stats['pending_amount'] = $pendingInstallments->sum('amount');

        $overdueInstallments = $allInstallments->filter(function ($installment) {
            return $installment->isOverdue();
        });
        $stats['overdue_count'] = $overdueInstallments->count();
        $stats['overdue_amount'] = $overdueInstallments->sum('amount');

        // بيانات الرسم البياني للأقساط حسب الشهر
        $sixMonthsAgo = Carbon::now()->subMonths(5)->startOfMonth();
        $installmentsByMonth = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where('due_date', '>=', $sixMonthsAgo)
            ->select(DB::raw('YEAR(due_date) as year'), DB::raw('MONTH(due_date) as month'), DB::raw('SUM(amount) as total_amount'))
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        $chartData = [
            'months' => [],
            'amounts' => []
        ];

        // إنشاء مصفوفة للأشهر الستة الماضية
        for ($i = 0; $i < 6; $i++) {
            $month = Carbon::now()->subMonths(5 - $i);
            $chartData['months'][] = $month->format('M Y');

            // البحث عن المبلغ لهذا الشهر
            $monthData = $installmentsByMonth->first(function ($item) use ($month) {
                return $item->year == $month->year && $item->month == $month->month;
            });

            $chartData['amounts'][] = $monthData ? $monthData->total_amount : 0;
        }

        // بيانات الرسم البياني للأقساط حسب العميل
        $installmentsByCustomer = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->with(['invoice', 'invoice.customer'])
            ->get()
            ->groupBy(function ($installment) {
                return $installment->invoice->customer->name;
            })
            ->map(function ($group) {
                return [
                    'total' => $group->sum('amount'),
                    'paid' => $group->sum('paid_amount'),
                    'due' => $group->sum('amount') - $group->sum('paid_amount')
                ];
            })
            ->sortByDesc(function ($value) {
                return $value['total'];
            })
            ->take(10);

        $customerChartData = [
            'customers' => $installmentsByCustomer->keys()->toArray(),
            'total' => $installmentsByCustomer->map(function ($value) {
                return $value['total'];
            })->values()->toArray(),
            'paid' => $installmentsByCustomer->map(function ($value) {
                return $value['paid'];
            })->values()->toArray(),
            'due' => $installmentsByCustomer->map(function ($value) {
                return $value['due'];
            })->values()->toArray(),
        ];

        return view('invoice_installments.reports', compact('stats', 'chartData', 'customerChartData'));
    }

    /**
     * عرض صفحة إرسال تذكيرات الأقساط.
     *
     * @return \Illuminate\Http\Response
     */
    public function sendReminders()
    {
        $companyId = auth()->user()->company_id;

        // الأقساط المستحقة خلال 7 أيام
        $upcomingInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where('status', 'pending')
            ->where('due_date', '>=', Carbon::now()->startOfDay())
            ->where('due_date', '<=', Carbon::now()->addDays(7)->endOfDay())
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->get();

        // الأقساط المتأخرة
        $overdueInstallments = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->where(function ($query) {
                $query->where('status', 'overdue')
                    ->orWhere(function ($q) {
                        $q->where('status', 'pending')
                            ->where('due_date', '<', Carbon::now()->startOfDay());
                    });
            })
            ->where('paid_amount', '<', DB::raw('amount'))
            ->with(['invoice', 'invoice.customer'])
            ->orderBy('due_date')
            ->get();

        return view('invoice_installments.send_reminders', compact('upcomingInstallments', 'overdueInstallments'));
    }

    /**
     * إرسال تذكيرات للأقساط المستحقة قريباً.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function sendUpcomingReminders(Request $request)
    {
        $companyId = auth()->user()->company_id;

        try {
            // استخدام خدمة إشعارات الأقساط
            $notificationService = app(\App\Services\Notifications\InstallmentNotificationService::class);
            $result = $notificationService->sendUpcomingInstallmentsReminders($companyId);

            if ($result['success']) {
                return redirect()->back()->with('success', $result['message']);
            } else {
                return redirect()->back()->with('error', $result['message']);
            }
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'حدث خطأ أثناء إرسال التذكيرات: ' . $e->getMessage());
        }
    }

    /**
     * إرسال تذكيرات للأقساط المتأخرة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function sendOverdueReminders(Request $request)
    {
        $companyId = auth()->user()->company_id;

        try {
            // استخدام خدمة إشعارات الأقساط
            $notificationService = app(\App\Services\Notifications\InstallmentNotificationService::class);
            $result = $notificationService->sendOverdueInstallmentsReminders($companyId);

            if ($result['success']) {
                return redirect()->back()->with('success', $result['message']);
            } else {
                return redirect()->back()->with('error', $result['message']);
            }
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'حدث خطأ أثناء إرسال التذكيرات: ' . $e->getMessage());
        }
    }

    // تم حذف الدالة المكررة sendUpcomingReminders

    // تم حذف الدالة المكررة sendReminder

    /**
     * عرض تقرير الأقساط المتأخرة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function overdueReport(Request $request)
    {
        $companyId = auth()->user()->company_id;

        // استعلام الأقساط المتأخرة
        $query = InvoiceInstallment::where(function ($query) {
            $query->where('status', 'overdue')
                ->orWhere(function ($q) {
                    $q->where('status', 'pending')
                        ->where('due_date', '<', Carbon::now()->startOfDay());
                });
        })
            ->where('paid_amount', '<', DB::raw('amount'))
            ->whereHas('invoice', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->with(['invoice', 'invoice.customer']);

        // تطبيق الفلاتر
        if ($request->filled('customer_id')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->where('customer_id', $request->customer_id);
            });
        }

        if ($request->filled('min_amount')) {
            $query->where('amount', '>=', $request->min_amount);
        }

        if ($request->filled('max_amount')) {
            $query->where('amount', '<=', $request->max_amount);
        }

        if ($request->filled('min_days_overdue')) {
            $date = Carbon::now()->subDays($request->min_days_overdue)->endOfDay();
            $query->where('due_date', '<=', $date);
        }

        if ($request->filled('max_days_overdue')) {
            $date = Carbon::now()->subDays($request->max_days_overdue)->startOfDay();
            $query->where('due_date', '>=', $date);
        }

        // ترتيب النتائج
        $sortBy = $request->input('sort_by', 'due_date');
        $sortDirection = $request->input('sort_direction', 'asc');

        if ($sortBy === 'days_overdue') {
            $query->orderBy('due_date', $sortDirection === 'asc' ? 'desc' : 'asc');
        } else {
            $query->orderBy($sortBy, $sortDirection);
        }

        // الحصول على النتائج مع التقسيم إلى صفحات
        $overdueInstallments = $query->paginate(15);

        // الحصول على قائمة العملاء للفلتر
        $customers = \App\Models\Customer::where('company_id', $companyId)
            ->orderBy('name')
            ->get();

        // الحصول على طرق الدفع المتاحة
        $paymentMethods = \App\Models\PaymentMethod::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('invoice_installments.overdue_report', compact('overdueInstallments', 'customers', 'paymentMethods'));
    }

    /**
     * عرض تقرير الأقساط المستحقة قريباً.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function upcomingReport(Request $request)
    {
        $companyId = auth()->user()->company_id;

        // استعلام الأقساط المستحقة قريباً
        $query = InvoiceInstallment::where('status', 'pending')
            ->where('due_date', '>=', Carbon::now()->startOfDay())
            ->where('due_date', '<=', Carbon::now()->addDays(30)->endOfDay())
            ->whereHas('invoice', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->with(['invoice', 'invoice.customer']);

        // تطبيق الفلاتر
        if ($request->filled('customer_id')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->where('customer_id', $request->customer_id);
            });
        }

        if ($request->filled('min_amount')) {
            $query->where('amount', '>=', $request->min_amount);
        }

        if ($request->filled('max_amount')) {
            $query->where('amount', '<=', $request->max_amount);
        }

        if ($request->filled('min_days_remaining')) {
            $date = Carbon::now()->addDays($request->min_days_remaining)->startOfDay();
            $query->where('due_date', '>=', $date);
        }

        if ($request->filled('max_days_remaining')) {
            $date = Carbon::now()->addDays($request->max_days_remaining)->endOfDay();
            $query->where('due_date', '<=', $date);
        }

        // ترتيب النتائج
        $sortBy = $request->input('sort_by', 'due_date');
        $sortDirection = $request->input('sort_direction', 'asc');

        $query->orderBy($sortBy, $sortDirection);

        // الحصول على النتائج مع التقسيم إلى صفحات
        $upcomingInstallments = $query->paginate(15);

        // الحصول على قائمة العملاء للفلتر
        $customers = \App\Models\Customer::where('company_id', $companyId)
            ->orderBy('name')
            ->get();

        // الحصول على طرق الدفع المتاحة
        $paymentMethods = \App\Models\PaymentMethod::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('invoice_installments.upcoming_report', compact('upcomingInstallments', 'customers', 'paymentMethods'));
    }

    /**
     * تصدير تقرير الأقساط المتأخرة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function exportOverdueReport(Request $request)
    {
        $companyId = auth()->user()->company_id;

        // استعلام الأقساط المتأخرة (نفس الاستعلام المستخدم في طريقة overdueReport)
        $query = InvoiceInstallment::where(function ($query) {
            $query->where('status', 'overdue')
                ->orWhere(function ($q) {
                    $q->where('status', 'pending')
                        ->where('due_date', '<', Carbon::now()->startOfDay());
                });
        })
            ->where('paid_amount', '<', DB::raw('amount'))
            ->whereHas('invoice', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->with(['invoice', 'invoice.customer']);

        // تطبيق نفس الفلاتر المستخدمة في الطريقة overdueReport
        if ($request->filled('customer_id')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->where('customer_id', $request->customer_id);
            });
        }

        if ($request->filled('min_amount')) {
            $query->where('amount', '>=', $request->min_amount);
        }

        if ($request->filled('max_amount')) {
            $query->where('amount', '<=', $request->max_amount);
        }

        if ($request->filled('min_days_overdue')) {
            $date = Carbon::now()->subDays($request->min_days_overdue)->endOfDay();
            $query->where('due_date', '<=', $date);
        }

        if ($request->filled('max_days_overdue')) {
            $date = Carbon::now()->subDays($request->max_days_overdue)->startOfDay();
            $query->where('due_date', '>=', $date);
        }

        // ترتيب النتائج
        $sortBy = $request->input('sort_by', 'due_date');
        $sortDirection = $request->input('sort_direction', 'asc');

        if ($sortBy === 'days_overdue') {
            $query->orderBy('due_date', $sortDirection === 'asc' ? 'desc' : 'asc');
        } else {
            $query->orderBy($sortBy, $sortDirection);
        }

        // الحصول على النتائج
        $overdueInstallments = $query->get();

        // تحديد صيغة التصدير
        $format = $request->input('format', 'excel');

        // تحضير البيانات للتصدير
        $exportData = [];

        foreach ($overdueInstallments as $installment) {
            $exportData[] = [
                'رقم القسط' => $installment->id,
                'رقم الفاتورة' => $installment->invoice->invoice_number,
                'العميل' => $installment->invoice->customer->name,
                'تاريخ الاستحقاق' => $installment->due_date->format('Y-m-d'),
                'أيام التأخير' => now()->diffInDays($installment->due_date),
                'المبلغ الكلي' => $installment->amount,
                'المبلغ المدفوع' => $installment->paid_amount,
                'المبلغ المتبقي' => $installment->amount - $installment->paid_amount,
                'الحالة' => $installment->status === 'overdue' ? 'متأخر' : 'قيد الانتظار',
            ];
        }

        // تصدير البيانات حسب الصيغة المطلوبة
        $fileName = 'تقرير_الأقساط_المتأخرة_' . date('Y-m-d');

        if ($format === 'excel') {
            return (new \App\Exports\ArrayExport($exportData))
                ->download($fileName . '.xlsx');
        } elseif ($format === 'csv') {
            return (new \App\Exports\ArrayExport($exportData))
                ->download($fileName . '.csv');
        } elseif ($format === 'pdf') {
            $company = \App\Models\Company::find($companyId);
            $data = [
                'title' => 'تقرير الأقساط المتأخرة',
                'company' => $company,
                'date' => date('Y-m-d'),
                'installments' => $overdueInstallments,
                'total_amount' => $overdueInstallments->sum('amount'),
                'total_paid' => $overdueInstallments->sum('paid_amount'),
                'total_remaining' => $overdueInstallments->sum(function ($item) {
                    return $item->amount - $item->paid_amount;
                }),
            ];

            $pdf = \PDF::loadView('exports.overdue_installments_pdf', $data);
            return $pdf->download($fileName . '.pdf');
        }

        return redirect()->back()->with('error', 'صيغة التصدير غير مدعومة');
    }

    /**
     * تصدير تقرير الأقساط المستحقة قريباً.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function exportUpcomingReport(Request $request)
    {
        $companyId = auth()->user()->company_id;

        // استعلام الأقساط المستحقة قريباً (نفس الاستعلام المستخدم في طريقة upcomingReport)
        $query = InvoiceInstallment::where('status', 'pending')
            ->where('due_date', '>=', Carbon::now()->startOfDay())
            ->where('due_date', '<=', Carbon::now()->addDays(30)->endOfDay())
            ->whereHas('invoice', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->with(['invoice', 'invoice.customer']);

        // تطبيق نفس الفلاتر المستخدمة في الطريقة upcomingReport
        if ($request->filled('customer_id')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->where('customer_id', $request->customer_id);
            });
        }

        if ($request->filled('min_amount')) {
            $query->where('amount', '>=', $request->min_amount);
        }

        if ($request->filled('max_amount')) {
            $query->where('amount', '<=', $request->max_amount);
        }

        if ($request->filled('min_days_remaining')) {
            $date = Carbon::now()->addDays($request->min_days_remaining)->startOfDay();
            $query->where('due_date', '>=', $date);
        }

        if ($request->filled('max_days_remaining')) {
            $date = Carbon::now()->addDays($request->max_days_remaining)->endOfDay();
            $query->where('due_date', '<=', $date);
        }

        // ترتيب النتائج
        $sortBy = $request->input('sort_by', 'due_date');
        $sortDirection = $request->input('sort_direction', 'asc');

        $query->orderBy($sortBy, $sortDirection);

        // الحصول على النتائج
        $upcomingInstallments = $query->get();

        // تحديد صيغة التصدير
        $format = $request->input('format', 'excel');

        // تحضير البيانات للتصدير
        $exportData = [];

        foreach ($upcomingInstallments as $installment) {
            $exportData[] = [
                'رقم القسط' => $installment->id,
                'رقم الفاتورة' => $installment->invoice->invoice_number,
                'العميل' => $installment->invoice->customer->name,
                'تاريخ الاستحقاق' => $installment->due_date->format('Y-m-d'),
                'الأيام المتبقية' => $installment->due_date->diffInDays(now()),
                'المبلغ' => $installment->amount,
                'الحالة' => 'قيد الانتظار',
            ];
        }

        // تصدير البيانات حسب الصيغة المطلوبة
        $fileName = 'تقرير_الأقساط_المستحقة_قريباً_' . date('Y-m-d');

        if ($format === 'excel') {
            return (new \App\Exports\ArrayExport($exportData))
                ->download($fileName . '.xlsx');
        } elseif ($format === 'csv') {
            return (new \App\Exports\ArrayExport($exportData))
                ->download($fileName . '.csv');
        } elseif ($format === 'pdf') {
            $company = \App\Models\Company::find($companyId);
            $data = [
                'title' => 'تقرير الأقساط المستحقة قريباً',
                'company' => $company,
                'date' => date('Y-m-d'),
                'installments' => $upcomingInstallments,
                'total_amount' => $upcomingInstallments->sum('amount'),
                'total_count' => $upcomingInstallments->count(),
                'next_week_count' => $upcomingInstallments->where('due_date', '<=', Carbon::now()->addDays(7))->count(),
                'next_week_amount' => $upcomingInstallments->where('due_date', '<=', Carbon::now()->addDays(7))->sum('amount'),
            ];

            $pdf = \PDF::loadView('exports.upcoming_installments_pdf', $data);
            return $pdf->download($fileName . '.pdf');
        }

        return redirect()->back()->with('error', 'صيغة التصدير غير مدعومة');
    }

    /**
     * عرض صفحة تقارير الأقساط الرئيسية.
     *
     * @return \Illuminate\Http\Response
     */
    public function reports()
    {
        $companyId = auth()->user()->company_id;

        // إحصائيات الأقساط المتأخرة
        $overdueCount = InvoiceInstallment::where(function ($query) {
            $query->where('status', 'overdue')
                ->orWhere(function ($q) {
                    $q->where('status', 'pending')
                        ->where('due_date', '<', Carbon::now()->startOfDay());
                });
        })
            ->where('paid_amount', '<', DB::raw('amount'))
            ->whereHas('invoice', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->count();

        // إحصائيات الأقساط المستحقة قريباً
        $upcomingCount = InvoiceInstallment::where('status', 'pending')
            ->where('due_date', '>=', Carbon::now()->startOfDay())
            ->where('due_date', '<=', Carbon::now()->addDays(30)->endOfDay())
            ->whereHas('invoice', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->count();

        // إحصائيات الأقساط المدفوعة
        $paidCount = InvoiceInstallment::where('status', 'paid')
            ->whereHas('invoice', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->count();

        // عدد العملاء الذين لديهم أقساط
        $customersCount = \App\Models\Customer::whereHas('invoices.installments', function ($query) {
            $query->where('paid_amount', '<', DB::raw('amount'));
        })
            ->where('company_id', $companyId)
            ->count();

        // عدد مدفوعات الأقساط
        $paymentsCount = \App\Models\InstallmentPayment::whereHas('installment.invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->count();

        // إجمالي عدد الأقساط
        $totalCount = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->count();

        // إجمالي مبالغ الأقساط
        $totalAmount = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->sum('amount');

        // إجمالي المبالغ المدفوعة
        $paidAmount = InvoiceInstallment::whereHas('invoice', function ($query) use ($companyId) {
            $query->where('company_id', $companyId);
        })
            ->sum('paid_amount');

        return view('invoice_installments.reports', compact(
            'overdueCount',
            'upcomingCount',
            'paidCount',
            'customersCount',
            'paymentsCount',
            'totalCount',
            'totalAmount',
            'paidAmount'
        ));
    }
}
