<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Customer;
use App\Models\MaintenanceContract;
use App\Models\InvoiceTemplate;
use App\Models\Company;
use App\Services\InvoiceService;
use App\Services\InvoiceTemplateService;
use App\Services\InstallmentService;
use App\Services\ZatcaService;
use App\Services\JournalEntryService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class InvoiceController extends Controller
{
    protected $invoiceService;
    protected $invoiceTemplateService;
    protected $installmentService;
    protected $zatcaService;
    protected $journalEntryService;

    /**
     * Create a new controller instance.
     *
     * @param  InvoiceService  $invoiceService
     * @param  InvoiceTemplateService  $invoiceTemplateService
     * @param  InstallmentService  $installmentService
     * @param  ZatcaService  $zatcaService
     * @param  JournalEntryService  $journalEntryService
     * @return void
     */
    public function __construct(
        InvoiceService $invoiceService,
        InvoiceTemplateService $invoiceTemplateService,
        InstallmentService $installmentService,
        ZatcaService $zatcaService,
        JournalEntryService $journalEntryService
    ) {
        $this->middleware('auth');
        $this->invoiceService = $invoiceService;
        $this->invoiceTemplateService = $invoiceTemplateService;
        $this->installmentService = $installmentService;
        $this->zatcaService = $zatcaService;
        $this->journalEntryService = $journalEntryService;
    }

    /**
     * عرض قائمة الفواتير.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        $query = Invoice::with(['customer', 'maintenance_contract', 'company']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        } elseif ($request->has('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        $invoices = $query->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('invoices.index', compact('invoices'));
    }

    /**
     * عرض قائمة الفواتير المتكررة.
     *
     * @return \Illuminate\Http\Response
     */
    public function recurring(Request $request)
    {
        $user = auth()->user();
        $query = Invoice::where('is_recurring', true)
            ->with(['customer', 'maintenance_contract', 'company']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        } elseif ($request->has('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        $invoices = $query->orderBy('next_recurring_date')
            ->paginate(15);

        return view('invoices.recurring', compact('invoices'));
    }

    /**
     * عرض نموذج إنشاء فاتورة جديدة.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $user = auth()->user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;

        if ($user->isSystemAdmin() && !$companyId) {
            $companies = Company::all();
            return view('invoices.create', compact('companies'));
        }

        $customers = Customer::where('company_id', $companyId)->get();
        $contracts = MaintenanceContract::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();
        $templates = InvoiceTemplate::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        return view('invoices.create', compact('customers', 'contracts', 'templates', 'companyId'));
    }

    /**
     * تخزين فاتورة جديدة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = auth()->user();
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'maintenance_contract_id' => 'nullable|exists:maintenance_contracts,id',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.discount_percentage' => 'nullable|numeric|min:0|max:100',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'notes' => 'nullable|string',
            'terms_conditions' => 'nullable|string',
            'template_id' => 'nullable|exists:invoice_templates,id',
            'language' => 'required|in:ar,en',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Get customer to derive company_id
        $customer = Customer::findOrFail($request->customer_id);
        $companyId = $customer->company_id;
        $company = Company::findOrFail($companyId);

        // إنشاء رقم الفاتورة
        $invoiceNumber = 'INV-' . $companyId . '-' . date('Ymd') . '-' . rand(1000, 9999);

        // حساب المبالغ
        $subtotal = 0;
        $totalDiscount = 0;

        foreach ($request->input('items') as $item) {
            $itemSubtotal = $item['quantity'] * $item['unit_price'];
            $itemDiscountPercentage = isset($item['discount_percentage']) ? $item['discount_percentage'] : 0;
            $itemDiscountAmount = $itemSubtotal * ($itemDiscountPercentage / 100);

            $subtotal += $itemSubtotal;
            $totalDiscount += $itemDiscountAmount;
        }

        $taxRate = $request->input('tax_rate');
        $taxAmount = ($subtotal - $totalDiscount) * ($taxRate / 100);
        $totalAmount = $subtotal - $totalDiscount + $taxAmount;

        try {
            DB::beginTransaction();

            // إنشاء الفاتورة
            $invoice = Invoice::create([
                'company_id' => $companyId,
                'customer_id' => $request->input('customer_id'),
                'maintenance_contract_id' => $request->input('maintenance_contract_id'),
                'branch_id' => $company->branches->first()->id ?? null,
                'created_by' => auth()->id(),
                'invoice_number' => $invoiceNumber,
                'invoice_date' => $request->input('invoice_date'),
                'due_date' => $request->input('due_date'),
                'subtotal' => $subtotal,
                'tax_rate' => $taxRate,
                'tax_amount' => $taxAmount,
                'discount_percentage' => $subtotal > 0 ? ($totalDiscount / $subtotal) * 100 : 0,
                'discount_amount' => $totalDiscount,
                'total_amount' => $totalAmount,
                'paid_amount' => 0,
                'due_amount' => $totalAmount,
                'status' => 'draft',
                'notes' => $request->input('notes'),
                'terms_conditions' => $request->input('terms_conditions'),
                'template_id' => $request->input('template_id'),
                'language' => $request->input('language'),
            ]);

            // إضافة بنود الفاتورة
            foreach ($request->input('items') as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemDiscountPercentage = isset($item['discount_percentage']) ? $item['discount_percentage'] : 0;
                $itemDiscountAmount = $itemSubtotal * ($itemDiscountPercentage / 100);
                $itemTaxableAmount = $itemSubtotal - $itemDiscountAmount;
                $itemTaxAmount = $itemTaxableAmount * ($taxRate / 100);
                $itemTotal = $itemTaxableAmount + $itemTaxAmount;

                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'discount_percentage' => $itemDiscountPercentage,
                    'discount_amount' => $itemDiscountAmount,
                    'tax_percentage' => $taxRate,
                    'tax_amount' => $itemTaxAmount,
                    'subtotal' => $itemSubtotal,
                    'total' => $itemTotal,
                ]);
            }

            // إعداد الفاتورة للفوترة الإلكترونية
            $invoice->zatca_invoice_type = 'standard';
            $invoice->save();

            // إنشاء رمز الاستجابة السريعة
            try {
                $qrCode = $this->zatcaService->generateQrCode($invoice);
                $qrImage = $this->generateQrImage($qrCode);
                $invoice->zatca_qr_code = $qrImage;
                $invoice->save();
            } catch (\Exception $e) {
                // تسجيل الخطأ ولكن لا نوقف العملية
                Log::error('فشل في إنشاء رمز الاستجابة السريعة: ' . $e->getMessage());
            }

            // إنشاء قيد محاسبي للفاتورة
            try {
                $journalEntry = $this->journalEntryService->createInvoiceJournalEntry($invoice);
                if ($journalEntry) {
                    Log::info('تم إنشاء قيد محاسبي للفاتورة بنجاح', [
                        'invoice_id' => $invoice->id,
                        'journal_entry_id' => $journalEntry->id
                    ]);
                }
            } catch (\Exception $e) {
                // تسجيل الخطأ ولكن لا نوقف العملية
                Log::error('فشل في إنشاء قيد محاسبي للفاتورة: ' . $e->getMessage(), [
                    'invoice_id' => $invoice->id,
                    'error' => $e->getMessage()
                ]);
            }

            DB::commit();

            return redirect()->route('invoices.show', $invoice->id)
                ->with('success', 'تم إنشاء الفاتورة بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء الفاتورة: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * عرض فاتورة محددة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = auth()->user();
        $query = Invoice::with(['customer', 'maintenance_contract', 'items', 'payments', 'installments', 'template']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);
        $companyId = $invoice->company_id;

        // Get available templates for this invoice language
        $templates = InvoiceTemplate::where('company_id', $companyId)
            ->where('language', $invoice->language)
            ->where('is_active', true)
            ->get();

        // التحقق من حالة الفوترة الإلكترونية
        $zatcaStatus = null;
        if ($invoice->zatca_reporting_status) {
            try {
                $zatcaStatus = $this->zatcaService->checkInvoiceStatus($invoice);
            } catch (\Exception $e) {
                \Log::error('فشل في التحقق من حالة الفاتورة: ' . $e->getMessage());
            }
        }

        return view('invoices.show', compact('invoice', 'templates', 'zatcaStatus'));
    }

    /**
     * إعادة إنشاء رمز QR للزكاة والدخل للفاتورة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function regenerateQrCode($id)
    {
        try {
            $user = auth()->user();
            $query = Invoice::with(['company']); // تحميل بيانات الشركة

            if (!$user->isSystemAdmin()) {
                $query->where('company_id', $user->company_id);
            }

            $invoice = $query->findOrFail($id);
            $companyId = $invoice->company_id;

            // التحقق من وجود بيانات الشركة
            if (!$invoice->company) {
                throw new \Exception('بيانات الشركة غير متوفرة للفاتورة');
            }

            // التحقق من وجود الرقم الضريبي للشركة
            if (empty($invoice->company->vat_number)) {
                // إذا لم يكن هناك رقم ضريبي، نستخدم رمز QR بسيط
                $dummyData = [
                    'seller_name' => $invoice->company->name ?? 'شركة المصاعد',
                    'vat_number' => '123456789012345', // رقم ضريبي افتراضي
                    'timestamp' => date('Y-m-d\TH:i:s\Z'),
                    'invoice_total' => number_format($invoice->total_amount, 2, '.', ''),
                    'vat_total' => number_format($invoice->tax_amount, 2, '.', '')
                ];

                // تحويل البيانات إلى سلسلة TLV بسيطة
                $tlvString = '';
                $tags = [
                    'seller_name' => 1,
                    'vat_number' => 2,
                    'timestamp' => 3,
                    'invoice_total' => 4,
                    'vat_total' => 5
                ];

                foreach ($dummyData as $key => $value) {
                    if (isset($tags[$key])) {
                        $tag = $tags[$key];
                        $length = strlen($value);
                        $tlvString .= chr($tag) . chr($length) . $value;
                    }
                }

                // تشفير البيانات بـ Base64
                $tlvBase64 = base64_encode($tlvString);
            } else {
                // استخدام خدمة الزكاة لإنشاء رمز QR
                $tlvBase64 = $this->zatcaService->generateQrCode($invoice);
            }

            // إنشاء صورة QR من البيانات
            $qrImage = $this->generateQrImage($tlvBase64);

            // حفظ صورة QR في الفاتورة
            $invoice->zatca_qr_code = $qrImage;
            $invoice->save();

            return redirect()->route('invoices.show', $invoice->id)
                ->with('success', 'تم إعادة إنشاء رمز الاستجابة السريعة (QR) بنجاح.');
        } catch (\Exception $e) {
            \Log::error('خطأ في إنشاء رمز QR: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إعادة إنشاء رمز الاستجابة السريعة: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء صورة QR من البيانات
     *
     * @param string $data
     * @return string
     */
    private function generateQrImage($data)
    {
        // استخدام مكتبة Simple QR Code لإنشاء رمز QR
        try {
            // استخدام صيغة 'svg' دائمًا لأنها لا تتطلب Imagick
            $qrcode = \SimpleSoftwareIO\QrCode\Facades\QrCode::format('svg')
                ->size(150)
                ->margin(1)
                ->errorCorrection('H')
                ->generate($data);

            // تشفير الصورة بـ Base64
            return base64_encode($qrcode);
        } catch (\Exception $e) {
            \Log::error('خطأ في إنشاء رمز QR باستخدام المكتبة: ' . $e->getMessage());

            // إذا فشلت المكتبة، نستخدم الطريقة البديلة مع GD
            $size = 150; // حجم الصورة
            $padding = 10; // الهامش

            // إنشاء صورة جديدة
            $image = imagecreatetruecolor($size, $size);

            // تعيين الألوان
            $white = imagecolorallocate($image, 255, 255, 255);
            $black = imagecolorallocate($image, 0, 0, 0);

            // ملء الخلفية باللون الأبيض
            imagefill($image, 0, 0, $white);

            // رسم نص في وسط الصورة (بديل مؤقت لرمز QR)
            $text = "QR Code";
            $font = 5; // حجم الخط
            $textWidth = imagefontwidth($font) * strlen($text);
            $textHeight = imagefontheight($font);
            $x = ($size - $textWidth) / 2;
            $y = ($size - $textHeight) / 2;

            // رسم النص
            imagestring($image, $font, $x, $y, $text, $black);

            // رسم إطار
            imagerectangle($image, $padding, $padding, $size - $padding - 1, $size - $padding - 1, $black);

            // تحويل الصورة إلى سلسلة Base64
            ob_start();
            imagepng($image);
            $imageData = ob_get_clean();

            // تحرير الذاكرة
            imagedestroy($image);

            // تشفير الصورة بـ Base64
            return base64_encode($imageData);
        }
    }

    /**
     * عرض معاينة الفاتورة بتنسيق HTML.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function previewHtml($id)
    {
        $user = auth()->user();
        $query = Invoice::with(['customer', 'maintenance_contract', 'items', 'payments', 'company', 'template']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);
        $companyId = $invoice->company_id;

        $templateId = request('template_id');
        $template = null;

        if ($templateId) {
            $template = InvoiceTemplate::where('company_id', $companyId)
                ->where('id', $templateId)
                ->first();
        }

        $html = $this->invoiceTemplateService->generateInvoiceHtml($invoice, $template);

        return response($html);
    }

    /**
     * تنزيل الفاتورة بتنسيق PDF.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function downloadPdf($id)
    {
        $user = auth()->user();
        $query = Invoice::with(['customer', 'maintenance_contract', 'items', 'payments', 'company', 'template']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);
        $companyId = $invoice->company_id;

        $templateId = request('template_id');
        $template = null;

        if ($templateId) {
            $template = InvoiceTemplate::where('company_id', $companyId)
                ->where('id', $templateId)
                ->first();
        }

        $pdf = $this->invoiceTemplateService->generateInvoicePdf($invoice, $template);

        return $pdf->download('invoice_' . $invoice->invoice_number . '.pdf');
    }

    /**
     * تغيير قالب الفاتورة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function changeTemplate(Request $request, $id)
    {
        $request->validate([
            'template_id' => 'required|exists:invoice_templates,id',
        ]);

        $user = auth()->user();
        $query = Invoice::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);
        $companyId = $invoice->company_id;

        $template = InvoiceTemplate::where('company_id', $companyId)
            ->where('id', $request->template_id)
            ->first();

        if (!$template) {
            return redirect()->back()->with('error', 'القالب غير موجود.');
        }

        $invoice->template_id = $template->id;
        $invoice->save();

        return redirect()->route('invoices.show', $invoice->id)
            ->with('success', 'تم تغيير قالب الفاتورة بنجاح.');
    }

    /**
     * عرض نموذج تعديل فاتورة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = auth()->user();
        $query = Invoice::with(['customer', 'maintenance_contract', 'items']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);
        $companyId = $invoice->company_id;

        // التحقق من أن الفاتورة في حالة مسودة
        if ($invoice->status !== 'draft') {
            return redirect()->route('invoices.show', $invoice->id)
                ->with('error', 'لا يمكن تعديل الفاتورة لأنها ليست في حالة مسودة.');
        }

        $customers = Customer::where('company_id', $companyId)->get();
        $contracts = MaintenanceContract::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();
        $templates = InvoiceTemplate::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        return view('invoices.edit', compact('invoice', 'customers', 'contracts', 'templates'));
    }

    /**
     * تحديث فاتورة محددة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = auth()->user();
        $query = Invoice::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);
        $companyId = $invoice->company_id;

        // التحقق من أن الفاتورة في حالة مسودة
        if ($invoice->status !== 'draft') {
            return redirect()->route('invoices.show', $invoice->id)
                ->with('error', 'لا يمكن تعديل الفاتورة لأنها ليست في حالة مسودة.');
        }

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'maintenance_contract_id' => 'nullable|exists:maintenance_contracts,id',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.discount_percentage' => 'nullable|numeric|min:0|max:100',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'notes' => 'nullable|string',
            'terms_conditions' => 'nullable|string',
            'template_id' => 'nullable|exists:invoice_templates,id',
            'language' => 'required|in:ar,en',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Get customer to ensure it belongs to the same company
        $customer = Customer::findOrFail($request->customer_id);
        if ($customer->company_id != $companyId) {
             return redirect()->back()
                ->with('error', 'العميل لا ينتمي لنفس الشركة.')
                ->withInput();
        }

        // حساب المبالغ
        $subtotal = 0;
        $totalDiscount = 0;

        foreach ($request->input('items') as $item) {
            $itemSubtotal = $item['quantity'] * $item['unit_price'];
            $itemDiscountPercentage = isset($item['discount_percentage']) ? $item['discount_percentage'] : 0;
            $itemDiscountAmount = $itemSubtotal * ($itemDiscountPercentage / 100);

            $subtotal += $itemSubtotal;
            $totalDiscount += $itemDiscountAmount;
        }

        $taxRate = $request->input('tax_rate');
        $taxAmount = ($subtotal - $totalDiscount) * ($taxRate / 100);
        $totalAmount = $subtotal - $totalDiscount + $taxAmount;

        try {
            DB::beginTransaction();

            // تحديث الفاتورة
            $invoice->update([
                'customer_id' => $request->input('customer_id'),
                'maintenance_contract_id' => $request->input('maintenance_contract_id'),
                'invoice_date' => $request->input('invoice_date'),
                'due_date' => $request->input('due_date'),
                'subtotal' => $subtotal,
                'tax_rate' => $taxRate,
                'tax_amount' => $taxAmount,
                'discount_percentage' => $subtotal > 0 ? ($totalDiscount / $subtotal) * 100 : 0,
                'discount_amount' => $totalDiscount,
                'total_amount' => $totalAmount,
                'due_amount' => $totalAmount,
                'notes' => $request->input('notes'),
                'terms_conditions' => $request->input('terms_conditions'),
                'template_id' => $request->input('template_id'),
                'language' => $request->input('language'),
            ]);

            // حذف بنود الفاتورة الحالية
            $invoice->items()->delete();

            // إضافة بنود الفاتورة الجديدة
            foreach ($request->input('items') as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemDiscountPercentage = isset($item['discount_percentage']) ? $item['discount_percentage'] : 0;
                $itemDiscountAmount = $itemSubtotal * ($itemDiscountPercentage / 100);
                $itemTaxableAmount = $itemSubtotal - $itemDiscountAmount;
                $itemTaxAmount = $itemTaxableAmount * ($taxRate / 100);
                $itemTotal = $itemTaxableAmount + $itemTaxAmount;

                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'discount_percentage' => $itemDiscountPercentage,
                    'discount_amount' => $itemDiscountAmount,
                    'tax_percentage' => $taxRate,
                    'tax_amount' => $itemTaxAmount,
                    'subtotal' => $itemSubtotal,
                    'total' => $itemTotal,
                ]);
            }

            // إعادة إنشاء رمز الاستجابة السريعة بعد التحديث
            try {
                $qrCode = $this->zatcaService->generateQrCode($invoice);
                $qrImage = $this->generateQrImage($qrCode);
                $invoice->zatca_qr_code = $qrImage;
                $invoice->save();
            } catch (\Exception $e) {
                // تسجيل الخطأ ولكن لا نوقف العملية
                Log::warning('فشل في تحديث رمز الاستجابة السريعة بعد تعديل الفاتورة: ' . $e->getMessage());
            }

            // تحديث القيد المحاسبي للفاتورة
            try {
                $journalEntry = $this->journalEntryService->updateInvoiceJournalEntry($invoice);
                if ($journalEntry) {
                    Log::info('تم تحديث القيد المحاسبي للفاتورة بنجاح', [
                        'invoice_id' => $invoice->id,
                        'journal_entry_id' => $journalEntry->id
                    ]);
                }
            } catch (\Exception $e) {
                // تسجيل الخطأ ولكن لا نوقف العملية
                Log::error('فشل في تحديث القيد المحاسبي للفاتورة: ' . $e->getMessage(), [
                    'invoice_id' => $invoice->id,
                    'error' => $e->getMessage()
                ]);
            }

            DB::commit();

            return redirect()->route('invoices.show', $invoice->id)
                ->with('success', 'تم تحديث الفاتورة بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث الفاتورة: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * حذف فاتورة محددة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = auth()->user();
        $query = Invoice::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);

        // التحقق من أن الفاتورة في حالة مسودة
        if ($invoice->status !== 'draft') {
            return redirect()->route('invoices.index')
                ->with('error', 'لا يمكن حذف الفاتورة لأنها ليست في حالة مسودة.');
        }

        // التحقق من عدم وجود مدفوعات مرتبطة بالفاتورة
        if ($invoice->payments()->count() > 0) {
            return redirect()->route('invoices.index')
                ->with('error', 'لا يمكن حذف الفاتورة لأنها مرتبطة بمدفوعات.');
        }

        try {
            DB::beginTransaction();

            // حذف بنود الفاتورة
            $invoice->items()->delete();

            // حذف أقساط الفاتورة
            $invoice->installments()->delete();

            // حذف الفاتورة
            $invoice->delete();

            DB::commit();

            return redirect()->route('invoices.index')
                ->with('success', 'تم حذف الفاتورة بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('invoices.index')
                ->with('error', 'حدث خطأ أثناء حذف الفاتورة: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء فاتورة من عقد صيانة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $contractId
     * @return \Illuminate\Http\Response
     */
    public function generateFromContract(Request $request, $contractId)
    {
        $user = auth()->user();
        $query = MaintenanceContract::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $contract = $query->findOrFail($contractId);

        try {
            $invoice = Invoice::generateFromContract($contract);

            return redirect()->route('invoices.show', $invoice->id)
                ->with('success', 'تم إنشاء الفاتورة من العقد بنجاح.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء الفاتورة: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء الفواتير المتكررة المستحقة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function generateRecurringInvoices(Request $request)
    {
        try {
            $results = $this->invoiceService->generateRecurringInvoices();

            if ($results['success'] > 0) {
                return redirect()->route('invoices.recurring')
                    ->with('success', 'تم إنشاء ' . $results['success'] . ' فاتورة متكررة بنجاح.');
            } else {
                return redirect()->route('invoices.recurring')
                    ->with('info', 'لا توجد فواتير متكررة مستحقة للإنشاء.');
            }
        } catch (\Exception $e) {
            return redirect()->route('invoices.recurring')
                ->with('error', 'حدث خطأ أثناء إنشاء الفواتير المتكررة: ' . $e->getMessage());
        }
    }

    /**
     * تحديث حالة الفواتير.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function updateInvoiceStatuses(Request $request)
    {
        try {
            $results = $this->invoiceService->updateInvoiceStatuses();

            return redirect()->route('invoices.index')
                ->with('success', 'تم تحديث حالة ' . $results['updated'] . ' فاتورة بنجاح.');
        } catch (\Exception $e) {
            return redirect()->route('invoices.index')
                ->with('error', 'حدث خطأ أثناء تحديث حالة الفواتير: ' . $e->getMessage());
        }
    }

    /**
     * تصدير الفاتورة إلى PDF.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function exportPdf($id)
    {
        $user = auth()->user();
        $query = Invoice::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);

        $content = $invoice->exportToPdf();

        return response($content)
            ->header('Content-Type', 'text/html');
    }

    /**
     * طباعة الفاتورة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function print($id)
    {
        $user = auth()->user();
        $query = Invoice::with(['customer', 'maintenance_contract', 'items', 'payments', 'company', 'template']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);

        return view('invoices.print', compact('invoice'));
    }

    /**
     * تصدير الفاتورة إلى Excel.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function exportExcel($id)
    {
        $user = auth()->user();
        $query = Invoice::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);

        $content = $invoice->exportToExcel();

        return redirect()->route('invoices.show', $invoice->id)
            ->with('info', $content);
    }

    /**
     * جعل الفاتورة متكررة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function makeRecurring(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'recurring_cycle' => 'required|in:monthly,quarterly,semi_annually,annually',
            'next_recurring_date' => 'required|date|after:today',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = auth()->user();
        $query = Invoice::query();

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);

        try {
            $invoice->makeRecurring(
                $request->input('recurring_cycle'),
                $request->input('next_recurring_date')
            );

            return redirect()->route('invoices.show', $invoice->id)
                ->with('success', 'تم تحويل الفاتورة إلى فاتورة متكررة بنجاح.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحويل الفاتورة إلى فاتورة متكررة: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * إنشاء أقساط للفاتورة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function createInstallments(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'number_of_installments' => 'required|integer|min:2|max:24',
            'frequency' => 'required|in:monthly,quarterly,custom',
            'custom_dates' => 'required_if:frequency,custom|array',
            'custom_dates.*' => 'required_if:frequency,custom|date',
            'custom_amounts' => 'required_if:frequency,custom|array',
            'custom_amounts.*' => 'required_if:frequency,custom|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = auth()->user();
        $query = Invoice::with('customer');

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($id);

        try {
            $numberOfInstallments = $request->input('number_of_installments');
            $frequency = $request->input('frequency');
            $customDates = $request->input('custom_dates', []);
            $customAmounts = $request->input('custom_amounts', []);

            $installments = $this->installmentService->createInstallments(
                $invoice,
                $numberOfInstallments,
                $frequency,
                $customDates,
                $customAmounts
            );

            return redirect()->route('invoices.show', $invoice->id)
                ->with('success', 'تم إنشاء ' . count($installments) . ' أقساط للفاتورة بنجاح.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء أقساط الفاتورة: ' . $e->getMessage())
                ->withInput();
        }
    }
}
