<?php

namespace App\Http\Controllers;

use App\Models\InventoryUnit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class InventoryUnitController extends Controller
{
    /**
     * عرض قائمة الوحدات
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $units = InventoryUnit::where('company_id', Auth::user()->company_id)
            ->orderBy('is_base', 'desc')
            ->orderBy('name')
            ->paginate(15);

        return view('inventory.units.index', compact('units'));
    }

    /**
     * عرض نموذج إنشاء وحدة جديدة
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('inventory.units.create');
    }

    /**
     * تخزين وحدة جديدة
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'code' => 'required|string|max:10|unique:inventory_units,code,NULL,id,company_id,' . Auth::user()->company_id,
            'description' => 'nullable|string',
            'is_base' => 'boolean',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $unit = new InventoryUnit();
        $unit->company_id = Auth::user()->company_id;
        $unit->name = $request->name;
        $unit->name_en = $request->name_en;
        $unit->code = $request->code;
        $unit->description = $request->description;
        $unit->is_base = $request->has('is_base') ? true : false;
        $unit->is_active = $request->has('is_active') ? true : false;
        $unit->save();

        return redirect()->route('inventory.units.index')
            ->with('success', 'تم إضافة الوحدة بنجاح');
    }

    /**
     * عرض تفاصيل وحدة محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $unit = InventoryUnit::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // الحصول على الأصناف المرتبطة بهذه الوحدة
        $items = $unit->inventory_items()
            ->with('inventory_category')
            ->paginate(10);

        // الحصول على الأصناف التي تستخدم هذه الوحدة كوحدة فرعية
        $itemUnits = $unit->inventory_item_units()
            ->with('inventory_item')
            ->paginate(10);

        return view('inventory.units.show', compact('unit', 'items', 'itemUnits'));
    }

    /**
     * عرض نموذج تعديل وحدة محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $unit = InventoryUnit::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        return view('inventory.units.edit', compact('unit'));
    }

    /**
     * تحديث وحدة محددة
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $unit = InventoryUnit::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'code' => 'required|string|max:10|unique:inventory_units,code,' . $id . ',id,company_id,' . Auth::user()->company_id,
            'description' => 'nullable|string',
            'is_base' => 'boolean',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $unit->name = $request->name;
        $unit->name_en = $request->name_en;
        $unit->code = $request->code;
        $unit->description = $request->description;
        $unit->is_base = $request->has('is_base') ? true : false;
        $unit->is_active = $request->has('is_active') ? true : false;
        $unit->save();

        return redirect()->route('inventory.units.index')
            ->with('success', 'تم تحديث الوحدة بنجاح');
    }

    /**
     * حذف وحدة محددة
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $unit = InventoryUnit::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // التحقق من عدم استخدام الوحدة في أي صنف
        if ($unit->inventory_items()->count() > 0) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف الوحدة لأنها مستخدمة كوحدة أساسية في بعض الأصناف');
        }

        // التحقق من عدم استخدام الوحدة كوحدة فرعية لأي صنف
        if ($unit->inventory_item_units()->count() > 0) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف الوحدة لأنها مستخدمة كوحدة فرعية في بعض الأصناف');
        }

        $unit->delete();

        return redirect()->route('inventory.units.index')
            ->with('success', 'تم حذف الوحدة بنجاح');
    }
}
