<?php

namespace App\Http\Controllers;

use App\Models\InventoryTransaction;
use App\Models\InventoryItem;
use App\Models\Warehouse;
use App\Models\InventoryCategory;
use App\Models\Supplier;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class InventoryTransactionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $items = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $query = InventoryTransaction::where('company_id', $company->id)
            ->with(['item.category', 'warehouse', 'toWarehouse', 'creator']);

        // Filter by warehouse
        if ($request->has('warehouse_id') && $request->warehouse_id) {
            $query->where(function ($q) use ($request) {
                $q->where('warehouse_id', $request->warehouse_id)
                    ->orWhere('to_warehouse_id', $request->warehouse_id);
            });
        }

        // Filter by item
        if ($request->has('item_id') && $request->item_id) {
            $query->where('item_id', $request->item_id);
        }

        // Filter by transaction type
        if ($request->has('transaction_type') && $request->transaction_type) {
            $query->where('transaction_type', $request->transaction_type);
        }

        // Filter by date range
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $transactions = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('inventory.transactions.index', compact('transactions', 'warehouses', 'items'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $transaction = InventoryTransaction::where('company_id', $company->id)
            ->where('id', $id)
            ->with(['item.category', 'warehouse', 'toWarehouse', 'batch', 'serial', 'creator'])
            ->firstOrFail();

        return view('inventory.transactions.show', compact('transaction'));
    }

    /**
     * Display stock value report.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function stockValueReport(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $categories = InventoryCategory::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $selectedWarehouse = null;
        if ($request->has('warehouse_id') && $request->warehouse_id) {
            $selectedWarehouse = Warehouse::where('company_id', $company->id)
                ->where('id', $request->warehouse_id)
                ->first();
        }

        $selectedCategory = null;
        if ($request->has('category_id') && $request->category_id) {
            $selectedCategory = InventoryCategory::where('company_id', $company->id)
                ->where('id', $request->category_id)
                ->first();
        }

        $query = InventoryItem::where('company_id', $company->id)
            ->with(['category', 'stocks.warehouse']);

        if ($selectedCategory) {
            $query->where('category_id', $selectedCategory->id);
        }

        $items = $query->orderBy('name')->get();

        // Filter items by warehouse if selected
        if ($selectedWarehouse) {
            $items = $items->filter(function ($item) use ($selectedWarehouse) {
                return $item->stocks->contains('warehouse_id', $selectedWarehouse->id);
            });
        }

        // Calculate totals
        $totalValue = 0;
        $totalItems = 0;

        foreach ($items as $item) {
            if ($selectedWarehouse) {
                $stock = $item->stocks->where('warehouse_id', $selectedWarehouse->id)->first();
                if ($stock) {
                    $totalValue += $stock->quantity * $item->purchase_price;
                    $totalItems += $stock->quantity > 0 ? 1 : 0;
                }
            } else {
                $totalValue += $item->total_stock_value;
                $totalItems += $item->total_quantity > 0 ? 1 : 0;
            }
        }

        return view('inventory.reports.stock_value', compact('items', 'warehouses', 'categories', 'selectedWarehouse', 'selectedCategory', 'totalValue', 'totalItems'));
    }

    /**
     * Display stock movement report.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function stockMovementReport(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $items = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $selectedItem = null;
        if ($request->has('item_id') && $request->item_id) {
            $selectedItem = InventoryItem::where('company_id', $company->id)
                ->where('id', $request->item_id)
                ->first();
        }

        $selectedWarehouse = null;
        if ($request->has('warehouse_id') && $request->warehouse_id) {
            $selectedWarehouse = Warehouse::where('company_id', $company->id)
                ->where('id', $request->warehouse_id)
                ->first();
        }

        $dateFrom = $request->input('date_from', Carbon::now()->subDays(30)->format('Y-m-d'));
        $dateTo = $request->input('date_to', Carbon::now()->format('Y-m-d'));

        $query = InventoryTransaction::where('company_id', $company->id)
            ->with(['item.category', 'warehouse', 'toWarehouse', 'creator'])
            ->whereDate('created_at', '>=', $dateFrom)
            ->whereDate('created_at', '<=', $dateTo);

        if ($selectedItem) {
            $query->where('item_id', $selectedItem->id);
        }

        if ($selectedWarehouse) {
            $query->where(function ($q) use ($selectedWarehouse) {
                $q->where('warehouse_id', $selectedWarehouse->id)
                    ->orWhere('to_warehouse_id', $selectedWarehouse->id);
            });
        }

        $transactions = $query->orderBy('created_at', 'desc')->get();

        // Calculate summary
        $summary = [
            'purchases' => 0,
            'sales' => 0,
            'transfers' => 0,
            'adjustments' => 0,
            'returns' => 0,
            'maintenance' => 0,
            'write_offs' => 0,
        ];

        foreach ($transactions as $transaction) {
            switch ($transaction->transaction_type) {
                case 'purchase':
                    $summary['purchases'] += $transaction->quantity;
                    break;
                case 'sale':
                    $summary['sales'] += abs($transaction->quantity);
                    break;
                case 'transfer':
                    $summary['transfers'] += $transaction->quantity;
                    break;
                case 'adjustment':
                    $summary['adjustments'] += $transaction->quantity;
                    break;
                case 'return_to_supplier':
                    $summary['returns'] += abs($transaction->quantity);
                    break;
                case 'return_from_customer':
                    $summary['returns'] += $transaction->quantity;
                    break;
                case 'maintenance_usage':
                    $summary['maintenance'] += abs($transaction->quantity);
                    break;
                case 'write_off':
                    $summary['write_offs'] += abs($transaction->quantity);
                    break;
            }
        }

        return view('inventory.reports.stock_movement', compact('transactions', 'warehouses', 'items', 'selectedItem', 'selectedWarehouse', 'dateFrom', 'dateTo', 'summary'));
    }

    /**
     * Display low stock report.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function lowStockReport(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $categories = InventoryCategory::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $selectedWarehouse = null;
        if ($request->has('warehouse_id') && $request->warehouse_id) {
            $selectedWarehouse = Warehouse::where('company_id', $company->id)
                ->where('id', $request->warehouse_id)
                ->first();
        }

        $selectedCategory = null;
        if ($request->has('category_id') && $request->category_id) {
            $selectedCategory = InventoryCategory::where('company_id', $company->id)
                ->where('id', $request->category_id)
                ->first();
        }

        $query = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->with(['category', 'stocks.warehouse']);

        if ($selectedCategory) {
            $query->where('category_id', $selectedCategory->id);
        }

        $items = $query->get();

        // Filter low stock items
        $lowStockItems = $items->filter(function ($item) use ($selectedWarehouse) {
            if ($selectedWarehouse) {
                $stock = $item->stocks->where('warehouse_id', $selectedWarehouse->id)->first();
                return $stock && $stock->quantity <= $item->min_stock_level;
            } else {
                return $item->is_low_stock;
            }
        });

        return view('inventory.reports.low_stock', compact('lowStockItems', 'warehouses', 'categories', 'selectedWarehouse', 'selectedCategory'));
    }

    /**
     * Display expiring items report.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function expiringItemsReport(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $selectedWarehouse = null;
        if ($request->has('warehouse_id') && $request->warehouse_id) {
            $selectedWarehouse = Warehouse::where('company_id', $company->id)
                ->where('id', $request->warehouse_id)
                ->first();
        }

        $daysToExpiry = $request->input('days_to_expiry', 30);
        $expiryDate = Carbon::now()->addDays($daysToExpiry);

        $query = \App\Models\InventoryBatch::where('company_id', $company->id)
            ->where('quantity', '>', 0)
            ->where('expiry_date', '<=', $expiryDate)
            ->with(['item.category', 'warehouse']);

        if ($selectedWarehouse) {
            $query->where('warehouse_id', $selectedWarehouse->id);
        }

        $batches = $query->orderBy('expiry_date')->get();

        return view('inventory.reports.expiring_items', compact('batches', 'warehouses', 'selectedWarehouse', 'daysToExpiry'));
    }

    /**
     * Display purchase orders report.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function purchaseOrdersReport(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $suppliers = Supplier::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $selectedSupplier = null;
        if ($request->has('supplier_id') && $request->supplier_id) {
            $selectedSupplier = Supplier::where('company_id', $company->id)
                ->where('id', $request->supplier_id)
                ->first();
        }

        $selectedWarehouse = null;
        if ($request->has('warehouse_id') && $request->warehouse_id) {
            $selectedWarehouse = Warehouse::where('company_id', $company->id)
                ->where('id', $request->warehouse_id)
                ->first();
        }

        $status = $request->input('status');
        $dateFrom = $request->input('date_from', Carbon::now()->subDays(30)->format('Y-m-d'));
        $dateTo = $request->input('date_to', Carbon::now()->format('Y-m-d'));

        $query = \App\Models\PurchaseOrder::where('company_id', $company->id)
            ->with(['supplier', 'warehouse', 'items.item', 'creator', 'approver'])
            ->whereDate('order_date', '>=', $dateFrom)
            ->whereDate('order_date', '<=', $dateTo);

        if ($selectedSupplier) {
            $query->where('supplier_id', $selectedSupplier->id);
        }

        if ($selectedWarehouse) {
            $query->where('warehouse_id', $selectedWarehouse->id);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $purchaseOrders = $query->orderBy('order_date', 'desc')->get();

        // Calculate summary
        $summary = [
            'total_orders' => $purchaseOrders->count(),
            'total_amount' => $purchaseOrders->sum('total_amount'),
            'pending_amount' => $purchaseOrders->where('status', 'pending')->sum('total_amount'),
            'approved_amount' => $purchaseOrders->where('status', 'approved')->sum('total_amount'),
            'received_amount' => $purchaseOrders->where('status', 'completed')->sum('total_amount'),
            'cancelled_amount' => $purchaseOrders->where('status', 'cancelled')->sum('total_amount'),
        ];

        return view('inventory.reports.purchase_orders', compact('purchaseOrders', 'suppliers', 'warehouses', 'selectedSupplier', 'selectedWarehouse', 'status', 'dateFrom', 'dateTo', 'summary'));
    }

    /**
     * Display supplier performance report.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function supplierPerformanceReport(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $suppliers = Supplier::where('company_id', $company->id)
            ->where('is_active', true)
            ->with(['evaluations' => function ($query) {
                $query->orderBy('evaluation_date', 'desc');
            }])
            ->get();

        $selectedSupplier = null;
        if ($request->has('supplier_id') && $request->supplier_id) {
            $selectedSupplier = Supplier::where('company_id', $company->id)
                ->where('id', $request->supplier_id)
                ->with(['evaluations' => function ($query) {
                    $query->orderBy('evaluation_date', 'desc');
                }])
                ->first();

            // Get purchase orders for selected supplier
            $purchaseOrders = \App\Models\PurchaseOrder::where('company_id', $company->id)
                ->where('supplier_id', $selectedSupplier->id)
                ->with(['items.item'])
                ->orderBy('order_date', 'desc')
                ->get();

            // Calculate on-time delivery percentage
            $completedOrders = $purchaseOrders->where('status', 'completed');
            $onTimeDelivery = 0;

            if ($completedOrders->count() > 0) {
                $onTimeCount = $completedOrders->filter(function ($order) {
                    return $order->delivery_date && $order->expected_delivery_date &&
                        $order->delivery_date->lte($order->expected_delivery_date);
                })->count();

                $onTimeDelivery = round(($onTimeCount / $completedOrders->count()) * 100, 2);
            }

            // Calculate average lead time
            $leadTimes = [];
            foreach ($completedOrders as $order) {
                if ($order->order_date && $order->delivery_date) {
                    $leadTimes[] = $order->order_date->diffInDays($order->delivery_date);
                }
            }

            $avgLeadTime = count($leadTimes) > 0 ? round(array_sum($leadTimes) / count($leadTimes), 1) : 0;

            return view('inventory.reports.supplier_performance_detail', compact('selectedSupplier', 'suppliers', 'purchaseOrders', 'onTimeDelivery', 'avgLeadTime'));
        }

        return view('inventory.reports.supplier_performance', compact('suppliers'));
    }
}
