<?php

namespace App\Http\Controllers;

use App\Models\InventoryItem;
use App\Models\InventoryTransaction;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class InventoryItemMovementController extends Controller
{
    /**
     * عرض تقرير حركة صنف محدد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function itemMovementReport(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        // الحصول على قائمة المخازن
        $warehouses = Warehouse::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // المخزن المحدد
        $selectedWarehouse = null;
        if ($request->has('warehouse_id') && $request->warehouse_id) {
            $selectedWarehouse = Warehouse::where('company_id', $company->id)
                ->where('id', $request->warehouse_id)
                ->first();
        }

        // الحصول على قائمة الأصناف
        $query = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true);

        // تصفية الأصناف حسب المخزن إذا تم تحديده
        if ($selectedWarehouse) {
            $warehouseId = $selectedWarehouse->id;
            $query->whereHas('stocks', function ($q) use ($warehouseId) {
                $q->where('warehouse_id', $warehouseId);
            });
        }

        $items = $query->orderBy('name')->get();

        // الصنف المحدد
        $selectedItem = null;
        if ($request->has('item_id') && $request->item_id) {
            $selectedItem = InventoryItem::where('company_id', $company->id)
                ->where('id', $request->item_id)
                ->first();
        }

        // تواريخ البحث
        $dateFrom = $request->input('date_from', Carbon::now()->subDays(30)->format('Y-m-d'));
        $dateTo = $request->input('date_to', Carbon::now()->format('Y-m-d'));

        // البيانات التفصيلية للتقرير
        $movements = collect();
        $openingBalance = 0;

        // إذا تم تحديد صنف، نقوم بجلب حركاته
        if ($selectedItem) {
            // حساب الرصيد الافتتاحي (مجموع الحركات قبل تاريخ البداية)
            $openingTransactions = InventoryTransaction::where('company_id', $company->id)
                ->where('item_id', $selectedItem->id)
                ->whereDate('created_at', '<', $dateFrom);

            // تصفية حسب المخزن إذا تم تحديده
            if ($selectedWarehouse) {
                $openingTransactions->where(function ($q) use ($selectedWarehouse) {
                    $q->where('warehouse_id', $selectedWarehouse->id)
                        ->orWhere('to_warehouse_id', $selectedWarehouse->id);
                });
            }

            $openingTransactions = $openingTransactions->get();

            foreach ($openingTransactions as $transaction) {
                // إذا كان هناك مخزن محدد، نحسب فقط الحركات المتعلقة بهذا المخزن
                if ($selectedWarehouse) {
                    if ($transaction->warehouse_id == $selectedWarehouse->id) {
                        // إذا كان المخزن هو مصدر الحركة
                        if (in_array($transaction->transaction_type, ['sale', 'transfer', 'return_to_supplier', 'maintenance_usage', 'write_off'])) {
                            $openingBalance -= abs($transaction->quantity);
                        } else {
                            $openingBalance += $transaction->quantity;
                        }
                    } elseif ($transaction->to_warehouse_id == $selectedWarehouse->id) {
                        // إذا كان المخزن هو وجهة الحركة (في حالة التحويل)
                        $openingBalance += $transaction->quantity;
                    }
                } else {
                    // إذا لم يتم تحديد مخزن، نحسب جميع الحركات
                    if (in_array($transaction->transaction_type, ['sale', 'transfer', 'return_to_supplier', 'maintenance_usage', 'write_off'])) {
                        $openingBalance -= abs($transaction->quantity);
                    } else {
                        $openingBalance += $transaction->quantity;
                    }
                }
            }

            // جلب الحركات خلال الفترة المحددة
            $query = InventoryTransaction::where('company_id', $company->id)
                ->where('item_id', $selectedItem->id)
                ->whereDate('created_at', '>=', $dateFrom)
                ->whereDate('created_at', '<=', $dateTo)
                ->with(['warehouse', 'toWarehouse', 'creator']);

            // تصفية حسب المخزن إذا تم تحديده
            if ($selectedWarehouse) {
                $query->where(function ($q) use ($selectedWarehouse) {
                    $q->where('warehouse_id', $selectedWarehouse->id)
                        ->orWhere('to_warehouse_id', $selectedWarehouse->id);
                });
            }

            $transactions = $query->orderBy('created_at')->get();

            // إعداد بيانات الحركات مع حساب الرصيد المتراكم
            $runningBalance = $openingBalance;

            foreach ($transactions as $transaction) {
                $inQty = 0;
                $outQty = 0;
                $affectsBalance = true;

                // تحديد نوع الحركة (وارد أو منصرف) وتأثيرها على الرصيد
                if ($selectedWarehouse) {
                    // إذا كان هناك مخزن محدد
                    if ($transaction->warehouse_id == $selectedWarehouse->id) {
                        // إذا كان المخزن هو مصدر الحركة
                        if (in_array($transaction->transaction_type, ['sale', 'return_to_supplier', 'maintenance_usage', 'write_off'])) {
                            $outQty = abs($transaction->quantity);
                            $runningBalance -= $outQty;
                        } elseif ($transaction->transaction_type == 'transfer') {
                            $outQty = abs($transaction->quantity);
                            $runningBalance -= $outQty;
                        } else {
                            $inQty = $transaction->quantity;
                            $runningBalance += $inQty;
                        }
                    } elseif ($transaction->to_warehouse_id == $selectedWarehouse->id) {
                        // إذا كان المخزن هو وجهة الحركة (في حالة التحويل)
                        $inQty = $transaction->quantity;
                        $runningBalance += $inQty;
                    } else {
                        $affectsBalance = false;
                    }
                } else {
                    // إذا لم يتم تحديد مخزن، نحسب جميع الحركات
                    if (in_array($transaction->transaction_type, ['sale', 'return_to_supplier', 'maintenance_usage', 'write_off'])) {
                        $outQty = abs($transaction->quantity);
                        $runningBalance -= $outQty;
                    } elseif ($transaction->transaction_type == 'transfer') {
                        // التحويل لا يؤثر على الرصيد الإجمالي إذا لم يتم تحديد مخزن
                        $affectsBalance = false;
                    } else {
                        $inQty = $transaction->quantity;
                        $runningBalance += $inQty;
                    }
                }

                if ($affectsBalance) {
                    // إضافة الحركة إلى قائمة الحركات
                    $movements->push([
                        'date' => $transaction->created_at,
                        'transaction_type' => $transaction->transaction_type,
                        'reference_type' => $transaction->reference_type,
                        'reference_number' => $transaction->reference_number,
                        'warehouse' => $transaction->warehouse ? $transaction->warehouse->name : '',
                        'to_warehouse' => $transaction->toWarehouse ? $transaction->toWarehouse->name : '',
                        'in_qty' => $inQty,
                        'out_qty' => $outQty,
                        'balance' => $runningBalance,
                        'notes' => $transaction->notes,
                        'created_by' => $transaction->creator ? $transaction->creator->name : ''
                    ]);
                }
            }
        }

        return view('inventory.reports.item_movement', compact(
            'warehouses',
            'items',
            'selectedWarehouse',
            'selectedItem',
            'dateFrom',
            'dateTo',
            'movements',
            'openingBalance'
        ));
    }
}
