<?php

namespace App\Http\Controllers;

use App\Models\InventoryItem;
use App\Models\InventoryCategory;
use App\Models\Warehouse;
use App\Models\InventoryStock;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class InventoryItemController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $company = $user->company;

        $items = InventoryItem::where('company_id', $company->id)
            ->with(['category', 'inventory_stocks'])
            ->orderBy('name')
            ->get();

        return view('inventory.items.index', compact('items'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $company = $user->company;

        $categories = InventoryCategory::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $suppliers = \App\Models\Supplier::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // Get inventory units for the dropdown
        $units = \App\Models\InventoryUnit::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('is_base', 'desc')
            ->orderBy('name')
            ->get();

        return view('inventory.items.create', compact('categories', 'suppliers', 'units'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'sku' => [
                'required',
                'string',
                'max:50',
                function ($attribute, $value, $fail) use ($company) {
                    $exists = InventoryItem::where('company_id', $company->id)
                        ->where('sku', $value)
                        ->exists();

                    if ($exists) {
                        $fail('رقم العنصر (SKU) مستخدم بالفعل في هذه الشركة.');
                    }
                },
            ],
            'barcode' => [
                'nullable',
                'string',
                'max:50',
                function ($attribute, $value, $fail) use ($company) {
                    if (!$value) return; // Skip validation if barcode is null

                    $exists = InventoryItem::where('company_id', $company->id)
                        ->where('barcode', $value)
                        ->exists();

                    if ($exists) {
                        $fail('الباركود مستخدم بالفعل في هذه الشركة.');
                    }
                },
            ],
            'category_id' => 'nullable|exists:inventory_categories,id',
            'description' => 'nullable|string',
            'base_unit_id' => 'required|exists:inventory_units,id',
            'quantity' => 'required|numeric|min:0',
            'min_quantity' => 'nullable|numeric|min:0',
            'unit_price' => 'required|numeric|min:0',
            'currency' => 'required|string|max:50',
            'location' => 'nullable|string|max:255',
            'is_active' => 'boolean',
            'image' => 'nullable|image|max:2048',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $item = new InventoryItem();
            $item->company_id = $company->id;
            $item->name = $request->name;
            $item->sku = $request->sku;
            $item->barcode = $request->barcode;
            $item->category_id = $request->category_id;
            $item->description = $request->description;
            $item->base_unit_id = $request->base_unit_id;
            $item->purchase_price = $request->unit_price;
            $item->selling_price = $request->unit_price; // يمكن تعديله لاحقاً
            $item->min_stock_level = $request->min_quantity ?? 0;
            $item->max_stock_level = $request->min_quantity * 3 ?? 10; // قيمة افتراضية
            $item->track_serial_numbers = false; // يمكن تعديله لاحقاً
            $item->track_batches = false; // يمكن تعديله لاحقاً
            $item->is_active = $request->has('is_active');
            $item->technical_specs = null; // يمكن تعديله لاحقاً

            // Handle image upload
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $imageName = time() . '_' . $image->getClientOriginalName();
                $image->storeAs('public/inventory/items', $imageName);
                $item->image = 'inventory/items/' . $imageName;
            }

            $item->save();

            try {
                // Create stock records for all warehouses
                $warehouses = Warehouse::where('company_id', $company->id)
                    ->where('is_active', true)
                    ->get();

                // إذا لم تكن هناك مستودعات، قم بإنشاء مستودع افتراضي أولاً
                if ($warehouses->isEmpty()) {
                    // إنشاء مستودع افتراضي
                    $defaultWarehouse = new Warehouse();
                    $defaultWarehouse->company_id = $company->id;
                    $defaultWarehouse->name = 'المستودع الرئيسي';
                    $defaultWarehouse->code = 'MAIN-' . $company->id;
                    $defaultWarehouse->address = 'العنوان الافتراضي';
                    $defaultWarehouse->is_active = true;
                    $defaultWarehouse->is_default = true;
                    $defaultWarehouse->save();

                    // إنشاء سجل مخزون للمستودع الافتراضي
                    InventoryStock::create([
                        'company_id' => $company->id,
                        'warehouse_id' => $defaultWarehouse->id,
                        'item_id' => $item->id,
                        'quantity' => $request->quantity ?? 0,
                        'reserved_quantity' => 0,
                        'location_code' => $request->location ?? null,
                    ]);
                } else {
                    // تحديد ما إذا كان هناك مستودع افتراضي
                    $defaultWarehouse = $warehouses->where('is_default', true)->first();

                    // إذا لم يكن هناك مستودع افتراضي، استخدم أول مستودع
                    if (!$defaultWarehouse) {
                        $defaultWarehouse = $warehouses->first();
                    }

                    // إنشاء سجلات المخزون لجميع المستودعات
                    foreach ($warehouses as $warehouse) {
                        InventoryStock::create([
                            'company_id' => $company->id,
                            'warehouse_id' => $warehouse->id,
                            'item_id' => $item->id,
                            'quantity' => ($warehouse->id == $defaultWarehouse->id) ? ($request->quantity ?? 0) : 0,
                            'reserved_quantity' => 0,
                            'location_code' => ($warehouse->id == $defaultWarehouse->id) ? ($request->location ?? null) : null,
                        ]);
                    }
                }
            } catch (\Exception $e) {
                // تسجيل الخطأ المحدد
                \Log::error('خطأ في إنشاء سجلات المخزون: ' . $e->getMessage());

                // حذف العنصر إذا تم إنشاؤه بالفعل
                if (isset($item) && $item->id) {
                    $item->delete();
                }

                throw $e; // إعادة رمي الاستثناء ليتم التقاطه في الـ catch الخارجي
            }

            return redirect()->route('inventory.items.index')
                ->with('success', 'تم إنشاء العنصر بنجاح');
        } catch (\Exception $e) {
            // تسجيل الخطأ للمطورين
            \Log::error('خطأ في إنشاء عنصر المخزون: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حفظ العنصر. الرجاء المحاولة مرة أخرى أو الاتصال بالدعم الفني.')
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $item = InventoryItem::where('company_id', $company->id)
            ->where('id', $id)
            ->with(['category', 'inventory_stocks.warehouse'])
            ->firstOrFail();

        // Generate QR code
        $qrCode = QrCode::size(200)->generate(route('inventory.items.show', $item->id));

        return view('inventory.items.show', compact('item', 'qrCode'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $item = InventoryItem::where('company_id', $company->id)
            ->where('id', $id)
            ->firstOrFail();

        $categories = InventoryCategory::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $suppliers = \App\Models\Supplier::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // Get inventory units for the dropdown
        $units = \App\Models\InventoryUnit::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('is_base', 'desc')
            ->orderBy('name')
            ->get();

        return view('inventory.items.edit', compact('item', 'categories', 'suppliers', 'units'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $company = $user->company;

        $item = InventoryItem::where('company_id', $company->id)
            ->where('id', $id)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'sku' => [
                'required',
                'string',
                'max:50',
                function ($attribute, $value, $fail) use ($company, $item) {
                    $exists = InventoryItem::where('company_id', $company->id)
                        ->where('sku', $value)
                        ->where('id', '!=', $item->id)
                        ->exists();

                    if ($exists) {
                        $fail('رقم العنصر (SKU) مستخدم بالفعل في هذه الشركة.');
                    }
                },
            ],
            'barcode' => [
                'nullable',
                'string',
                'max:50',
                function ($attribute, $value, $fail) use ($company, $item) {
                    if (!$value) return; // Skip validation if barcode is null

                    $exists = InventoryItem::where('company_id', $company->id)
                        ->where('barcode', $value)
                        ->where('id', '!=', $item->id)
                        ->exists();

                    if ($exists) {
                        $fail('الباركود مستخدم بالفعل في هذه الشركة.');
                    }
                },
            ],
            'category_id' => 'nullable|exists:inventory_categories,id',
            'description' => 'nullable|string',
            'base_unit_id' => 'required|exists:inventory_units,id',
            'purchase_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'min_stock_level' => 'required|numeric|min:0',
            'max_stock_level' => 'required|numeric|min:0',
            'track_serial_numbers' => 'boolean',
            'track_batches' => 'boolean',
            'is_active' => 'boolean',
            'image' => 'nullable|image|max:2048',
            'technical_specs' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $item->name = $request->name;
        $item->sku = $request->sku;
        $item->barcode = $request->barcode;
        $item->category_id = $request->category_id;
        $item->description = $request->description;
        $item->base_unit_id = $request->base_unit_id;
        $item->purchase_price = $request->purchase_price;
        $item->selling_price = $request->selling_price;
        $item->min_stock_level = $request->min_stock_level;
        $item->max_stock_level = $request->max_stock_level;
        $item->track_serial_numbers = $request->has('track_serial_numbers');
        $item->track_batches = $request->has('track_batches');
        $item->is_active = $request->has('is_active');
        $item->technical_specs = $request->technical_specs;

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($item->image) {
                Storage::delete('public/' . $item->image);
            }

            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->storeAs('public/inventory/items', $imageName);
            $item->image = 'inventory/items/' . $imageName;
        }

        $item->save();

        return redirect()->route('inventory.items.index')
            ->with('success', 'تم تحديث العنصر بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $item = InventoryItem::where('company_id', $company->id)
            ->where('id', $id)
            ->firstOrFail();

        // Check if item has stock
        $hasStock = false;
        foreach ($item->inventory_stocks as $stock) {
            if ($stock->quantity > 0) {
                $hasStock = true;
                break;
            }
        }

        if ($hasStock) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف العنصر لأنه يحتوي على مخزون');
        }

        // Check if item has transactions
        $hasTransactions = $item->inventory_transactions()->count() > 0;

        if ($hasTransactions) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف العنصر لأنه مرتبط بمعاملات');
        }

        // Delete image if exists
        if ($item->image) {
            Storage::delete('public/' . $item->image);
        }

        // Delete stock records
        $item->inventory_stocks()->delete();

        $item->delete();

        return redirect()->route('inventory.items.index')
            ->with('success', 'تم حذف العنصر بنجاح');
    }

    /**
     * Generate QR code for the item.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function qrcode($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $item = InventoryItem::where('company_id', $company->id)
            ->where('id', $id)
            ->firstOrFail();

        $qrCode = QrCode::size(300)->generate(route('inventory.items.show', $item->id));

        return response($qrCode)->header('Content-Type', 'image/svg+xml');
    }

    /**
     * Display low stock items.
     *
     * @return \Illuminate\Http\Response
     */
    public function lowStock()
    {
        $user = Auth::user();
        $company = $user->company;

        $items = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->with(['category', 'stocks.warehouse'])
            ->get()
            ->filter(function ($item) {
                return $item->is_low_stock;
            });

        return view('inventory.items.low_stock', compact('items'));
    }
}
