<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\InventoryIssue;
use App\Models\InventoryIssueItem;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\InventoryTransaction;
use App\Models\InventoryUnit;
use App\Models\InventoryItemUnit;
use App\Models\Warehouse;
use App\Models\WorkOrder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class InventoryIssueController extends Controller
{
    /**
     * عرض قائمة أذونات الصرف
     */
    public function index()
    {
        // عرض أذونات الصرف الخاصة بالشركة الحالية فقط
        $inventoryIssues = InventoryIssue::with(['company', 'workOrder', 'warehouse', 'user'])
            ->where('company_id', Auth::user()->company_id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('inventory_issues.index', compact('inventoryIssues'));
    }

    /**
     * عرض نموذج إنشاء إذن صرف جديد
     */
    public function create()
    {
        // الحصول على الشركة الحالية للمستخدم
        $companies = Company::where('id', Auth::user()->company_id)->get();

        // الحصول على المستودعات التابعة للشركة الحالية
        $warehouses = Warehouse::where('company_id', Auth::user()->company_id)->get();

        // الحصول على أوامر الشغل المفتوحة فقط للشركة الحالية
        $workOrders = WorkOrder::where('status', 'open')
            ->where('company_id', Auth::user()->company_id)
            ->get();

        // الحصول على عناصر المخزون للشركة الحالية
        $inventoryItems = InventoryItem::where('company_id', Auth::user()->company_id)->get();

        // الحصول على وحدات الأصناف من الجدول الوسيط
        $inventoryItemUnits = InventoryItemUnit::with(['inventory_unit', 'inventory_item'])
            ->whereHas('inventory_item', function ($query) {
                $query->where('company_id', Auth::user()->company_id);
            })
            ->get();

        return view('inventory_issues.create', compact('companies', 'warehouses', 'workOrders', 'inventoryItems', 'inventoryItemUnits'));
    }

    /**
     * تخزين إذن صرف جديد
     */
    public function store(Request $request)
    {
        $request->validate([
            'work_order_id' => 'required|exists:work_orders,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'issue_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'items.*.unit_id' => 'required|exists:inventory_units,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.notes' => 'nullable|string',
            'auto_complete' => 'nullable|boolean', // إضافة حقل لاعتماد إذن الصرف تلقائيًا
        ]);

        try {
            DB::beginTransaction();

            // إنشاء رقم إذن الصرف
            $latestIssue = InventoryIssue::where('company_id', Auth::user()->company_id)
                ->orderBy('id', 'desc')
                ->first();

            $issueNumber = 'IS-' . date('Ymd') . '-';
            if ($latestIssue) {
                // استخراج الرقم من آخر إذن صرف وزيادته بواحد
                $lastNumber = intval(substr($latestIssue->issue_number, strrpos($latestIssue->issue_number, '-') + 1));
                $issueNumber .= str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
            } else {
                $issueNumber .= '0001';
            }

            // تحديد حالة إذن الصرف (معلق أو معتمد)
            $status = $request->has('auto_complete') && $request->auto_complete ? 'completed' : 'pending';
            $completedAt = $status === 'completed' ? now() : null;

            // إنشاء إذن الصرف مع تعيين الشركة تلقائياً من المستخدم الحالي
            $inventoryIssue = InventoryIssue::create([
                'company_id' => Auth::user()->company_id,
                'work_order_id' => $request->work_order_id,
                'warehouse_id' => $request->warehouse_id,
                'issue_date' => $request->issue_date,
                'status' => $status,
                'completed_at' => $completedAt,
                'notes' => $request->notes,
                'user_id' => Auth::id(),
                'issue_number' => $issueNumber, // إضافة رقم إذن الصرف
            ]);

            // إضافة بنود إذن الصرف
            foreach ($request->items as $item) {
                InventoryIssueItem::create([
                    'inventory_issue_id' => $inventoryIssue->id,
                    'inventory_item_id' => $item['inventory_item_id'],
                    'unit_id' => $item['unit_id'],
                    'quantity' => $item['quantity'],
                    'notes' => $item['notes'] ?? null,
                ]);
            }

            // إذا كان إذن الصرف معتمدًا، قم بتحديث المخزون
            if ($status === 'completed') {
                // إعادة تحميل إذن الصرف مع العناصر
                $inventoryIssue->refresh();
                $this->updateInventory($inventoryIssue);
            }

            DB::commit();

            $successMessage = $status === 'completed'
                ? 'تم إنشاء واعتماد إذن الصرف وتحديث المخزون بنجاح'
                : 'تم إنشاء إذن الصرف بنجاح';

            return redirect()->route('inventory-issues.show', $inventoryIssue->id)
                ->with('success', $successMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ أثناء إنشاء إذن الصرف: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل إذن صرف محدد
     */
    public function show(InventoryIssue $inventoryIssue)
    {
        // التحقق من أن إذن الصرف ينتمي لشركة المستخدم الحالي
        if ($inventoryIssue->company_id !== Auth::user()->company_id) {
            return redirect()->route('inventory-issues.index')
                ->with('error', 'لا يمكنك عرض إذن صرف لا ينتمي لشركتك');
        }

        $inventoryIssue->load(['company', 'workOrder', 'warehouse', 'user', 'items.inventoryItem']);

        return view('inventory_issues.show', compact('inventoryIssue'));
    }

    /**
     * عرض نموذج تعديل إذن صرف
     */
    public function edit(InventoryIssue $inventoryIssue)
    {
        // لا يمكن تعديل إذن الصرف إذا كان معتمداً
        if ($inventoryIssue->status === 'completed') {
            return redirect()->route('inventory-issues.show', $inventoryIssue->id)
                ->with('error', 'لا يمكن تعديل إذن صرف معتمد');
        }

        // التحقق من أن إذن الصرف ينتمي لشركة المستخدم الحالي
        if ($inventoryIssue->company_id !== Auth::user()->company_id) {
            return redirect()->route('inventory-issues.index')
                ->with('error', 'لا يمكنك تعديل إذن صرف لا ينتمي لشركتك');
        }

        $companies = Company::where('id', Auth::user()->company_id)->get();
        $warehouses = Warehouse::where('company_id', Auth::user()->company_id)->get();
        $workOrders = WorkOrder::where('status', 'open')
            ->where('company_id', Auth::user()->company_id)
            ->get();
        $inventoryItems = InventoryItem::where('company_id', Auth::user()->company_id)->get();
        $inventoryItemUnits = InventoryItemUnit::with(['inventory_unit', 'inventory_item'])
            ->whereHas('inventory_item', function ($query) {
                $query->where('company_id', Auth::user()->company_id);
            })
            ->get();
        $inventoryIssue->load(['items.inventoryItem', 'items.unit']);

        return view('inventory_issues.edit', compact('inventoryIssue', 'companies', 'warehouses', 'workOrders', 'inventoryItems', 'inventoryItemUnits'));
    }

    /**
     * تحديث المخزون عند تعديل إذن الصرف
     *
     * @param InventoryIssue $issue
     * @param array $oldItems
     * @return void
     */
    private function updateInventoryOnEdit(InventoryIssue $issue, array $oldItems)
    {
        $companyId = $issue->company_id;
        $warehouseId = $issue->warehouse_id;

        // تجميع العناصر القديمة والجديدة للمقارنة
        $newItems = $issue->items->keyBy('id');

        // معالجة العناصر المحدثة والجديدة
        foreach ($issue->items as $issueItem) {
            $itemId = $issueItem->inventory_item_id;
            $newQuantity = $issueItem->quantity;

            // التحقق مما إذا كان هذا عنصر موجود مسبقًا أو جديد
            if (isset($oldItems[$issueItem->id])) {
                // عنصر موجود مسبقًا - حساب الفرق في الكمية
                $oldQuantity = $oldItems[$issueItem->id]['quantity'];
                $quantityDifference = $newQuantity - $oldQuantity;

                // إذا لم يكن هناك تغيير في الكمية، تخطي هذا العنصر
                if ($quantityDifference == 0) {
                    continue;
                }

                // الحصول على سجل المخزون
                $stock = InventoryStock::firstOrNew([
                    'company_id' => $companyId,
                    'warehouse_id' => $warehouseId,
                    'item_id' => $itemId,
                ]);

                if (!$stock->exists) {
                    $stock->quantity = 0;
                    $stock->reserved_quantity = 0;
                }

                // تحديث كمية المخزون بناءً على الفرق (طرح الفرق لأنها عملية صرف)
                $stock->quantity -= $quantityDifference;
                $stock->save();

                // إنشاء معاملة مخزون للتعديل
                $transaction = new InventoryTransaction();
                $transaction->company_id = $companyId;
                $transaction->transaction_type = 'adjustment';
                $transaction->item_id = $itemId;
                $transaction->warehouse_id = $warehouseId;
                $transaction->quantity = -$quantityDifference; // قيمة سالبة لأنها عملية صرف
                $transaction->reference_number = $issue->issue_number;
                $transaction->reference_type = 'inventory_issue_edit';
                $transaction->reference_id = $issue->id;
                $transaction->created_by = auth()->id();
                $transaction->notes = 'تعديل كمية الصرف في إذن الصرف #' . $issue->issue_number;
                $transaction->save();
            } else {
                // عنصر جديد - طرح الكمية الجديدة من المخزون
                $stock = InventoryStock::firstOrNew([
                    'company_id' => $companyId,
                    'warehouse_id' => $warehouseId,
                    'item_id' => $itemId,
                ]);

                if (!$stock->exists) {
                    $stock->quantity = 0;
                    $stock->reserved_quantity = 0;
                }

                // طرح كمية المخزون
                $stock->quantity -= $newQuantity;
                $stock->save();

                // إنشاء معاملة مخزون للإضافة
                $transaction = new InventoryTransaction();
                $transaction->company_id = $companyId;
                $transaction->transaction_type = 'maintenance_usage';
                $transaction->item_id = $itemId;
                $transaction->warehouse_id = $warehouseId;
                $transaction->quantity = -$newQuantity; // قيمة سالبة لأنها عملية صرف
                $transaction->reference_number = $issue->issue_number;
                $transaction->reference_type = 'inventory_issue';
                $transaction->reference_id = $issue->id;
                $transaction->created_by = auth()->id();
                $transaction->notes = 'إضافة عنصر جديد للصرف في إذن الصرف #' . $issue->issue_number;
                $transaction->save();
            }
        }

        // معالجة العناصر المحذوفة
        foreach ($oldItems as $oldItemId => $oldItem) {
            if (!$newItems->has($oldItemId)) {
                // عنصر محذوف - إعادة الكمية للمخزون
                $itemId = $oldItem['inventory_item_id'];
                $oldQuantity = $oldItem['quantity'];

                $stock = InventoryStock::where([
                    'company_id' => $companyId,
                    'warehouse_id' => $warehouseId,
                    'item_id' => $itemId,
                ])->first();

                if ($stock) {
                    // إضافة الكمية المحذوفة للمخزون (إعادة الكمية)
                    $stock->quantity += $oldQuantity;
                    $stock->save();

                    // إنشاء معاملة مخزون للحذف
                    $transaction = new InventoryTransaction();
                    $transaction->company_id = $companyId;
                    $transaction->transaction_type = 'adjustment';
                    $transaction->item_id = $itemId;
                    $transaction->warehouse_id = $warehouseId;
                    $transaction->quantity = $oldQuantity; // قيمة موجبة لأنها إعادة للمخزون
                    $transaction->reference_number = $issue->issue_number;
                    $transaction->reference_type = 'inventory_issue_delete_item';
                    $transaction->reference_id = $issue->id;
                    $transaction->created_by = auth()->id();
                    $transaction->notes = 'حذف عنصر من إذن الصرف #' . $issue->issue_number;
                    $transaction->save();
                }
            }
        }
    }

    /**
     * تحديث إذن صرف محدد
     */
    public function update(Request $request, InventoryIssue $inventoryIssue)
    {
        // لا يمكن تعديل إذن الصرف إذا كان معتمداً
        if ($inventoryIssue->status === 'completed') {
            return redirect()->route('inventory-issues.show', $inventoryIssue->id)
                ->with('error', 'لا يمكن تعديل إذن صرف معتمد');
        }

        $request->validate([
            'work_order_id' => 'required|exists:work_orders,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'issue_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.id' => 'nullable|exists:inventory_issue_items,id',
            'items.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'items.*.unit_id' => 'required|exists:inventory_units,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            // حفظ الكميات القديمة قبل التحديث
            $oldItems = $inventoryIssue->items->keyBy('id')->toArray();

            // تحديث إذن الصرف (مع الاحتفاظ بالشركة الأصلية)
            $inventoryIssue->update([
                'work_order_id' => $request->work_order_id,
                'warehouse_id' => $request->warehouse_id,
                'issue_date' => $request->issue_date,
                'notes' => $request->notes,
            ]);

            // تتبع معرفات البنود المحدثة للتعرف على البنود المحذوفة
            $updatedItemIds = [];

            // تحديث أو إنشاء بنود إذن الصرف
            foreach ($request->items as $item) {
                if (isset($item['id'])) {
                    // تحديث بند موجود
                    $issueItem = InventoryIssueItem::find($item['id']);
                    $issueItem->update([
                        'inventory_item_id' => $item['inventory_item_id'],
                        'unit_id' => $item['unit_id'],
                        'quantity' => $item['quantity'],
                        'notes' => $item['notes'] ?? null,
                    ]);
                    $updatedItemIds[] = $issueItem->id;
                } else {
                    // إنشاء بند جديد
                    $issueItem = InventoryIssueItem::create([
                        'inventory_issue_id' => $inventoryIssue->id,
                        'inventory_item_id' => $item['inventory_item_id'],
                        'unit_id' => $item['unit_id'],
                        'quantity' => $item['quantity'],
                        'notes' => $item['notes'] ?? null,
                    ]);
                    $updatedItemIds[] = $issueItem->id;
                }
            }

            // حذف البنود التي لم يتم تحديثها
            InventoryIssueItem::where('inventory_issue_id', $inventoryIssue->id)
                ->whereNotIn('id', $updatedItemIds)
                ->delete();

            // إعادة تحميل إذن الصرف مع العناصر المحدثة
            $inventoryIssue->refresh();

            // إذا كان إذن الصرف معتمداً، قم بتحديث المخزون
            if ($inventoryIssue->status === 'completed') {
                $this->updateInventoryOnEdit($inventoryIssue, $oldItems);
            }

            DB::commit();

            return redirect()->route('inventory-issues.show', $inventoryIssue->id)
                ->with('success', 'تم تحديث إذن الصرف بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ أثناء تحديث إذن الصرف: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * استعادة المخزون عند حذف إذن صرف معتمد
     *
     * @param InventoryIssue $issue
     * @return void
     */
    private function restoreInventory(InventoryIssue $issue)
    {
        $companyId = $issue->company_id;
        $warehouseId = $issue->warehouse_id;

        foreach ($issue->items as $issueItem) {
            $itemId = $issueItem->inventory_item_id;
            $quantity = $issueItem->quantity;

            // الحصول على سجل المخزون
            $stock = InventoryStock::where([
                'company_id' => $companyId,
                'warehouse_id' => $warehouseId,
                'item_id' => $itemId,
            ])->first();

            if ($stock) {
                // إعادة الكمية المصروفة إلى المخزون
                $stock->quantity += $quantity;
                $stock->save();

                // إنشاء معاملة مخزون للإلغاء
                $transaction = new InventoryTransaction();
                $transaction->company_id = $companyId;
                $transaction->transaction_type = 'adjustment';
                $transaction->item_id = $itemId;
                $transaction->warehouse_id = $warehouseId;
                $transaction->quantity = $quantity; // قيمة موجبة لأنها إعادة للمخزون
                $transaction->reference_number = $issue->issue_number;
                $transaction->reference_type = 'inventory_issue_cancel';
                $transaction->reference_id = $issue->id;
                $transaction->created_by = auth()->id();
                $transaction->notes = 'إلغاء إذن الصرف #' . $issue->issue_number;
                $transaction->save();
            }
        }
    }

    /**
     * حذف إذن صرف محدد
     */
    public function destroy(InventoryIssue $inventoryIssue)
    {
        // لا يمكن حذف إذن الصرف إذا كان معتمداً
        if ($inventoryIssue->status === 'completed') {
            return redirect()->route('inventory-issues.index')
                ->with('error', 'لا يمكن حذف إذن صرف معتمد');
        }

        try {
            DB::beginTransaction();

            // تحميل بنود إذن الصرف قبل الحذف
            $inventoryIssue->load('items');

            // حذف بنود إذن الصرف أولاً
            InventoryIssueItem::where('inventory_issue_id', $inventoryIssue->id)->delete();

            // ثم حذف إذن الصرف نفسه
            $inventoryIssue->delete();

            DB::commit();

            return redirect()->route('inventory-issues.index')
                ->with('success', 'تم حذف إذن الصرف بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ أثناء حذف إذن الصرف: ' . $e->getMessage());
        }
    }

    /**
     * تحديث المخزون بناءً على إذن الصرف
     *
     * @param InventoryIssue $issue
     * @return void
     */
    private function updateInventory(InventoryIssue $issue)
    {
        $companyId = $issue->company_id;
        $warehouseId = $issue->warehouse_id;

        foreach ($issue->items as $issueItem) {
            $itemId = $issueItem->inventory_item_id;
            $quantity = $issueItem->quantity;

            // الحصول على سجل المخزون
            $stock = InventoryStock::firstOrNew([
                'company_id' => $companyId,
                'warehouse_id' => $warehouseId,
                'item_id' => $itemId,
            ]);

            if (!$stock->exists) {
                $stock->quantity = 0;
                $stock->reserved_quantity = 0;
            }

            // تقليل كمية المخزون (طرح الكمية المنصرفة)
            $stock->quantity -= $quantity;
            $stock->save();

            // إنشاء معاملة مخزون
            $transaction = new InventoryTransaction();
            $transaction->company_id = $companyId;
            $transaction->transaction_type = 'maintenance_usage'; // نوع المعاملة: استخدام للصيانة
            $transaction->item_id = $itemId;
            $transaction->warehouse_id = $warehouseId;
            $transaction->quantity = -$quantity; // قيمة سالبة لأنها عملية صرف
            $transaction->reference_number = $issue->issue_number;
            $transaction->reference_type = 'inventory_issue';
            $transaction->reference_id = $issue->id;
            $transaction->created_by = auth()->id();
            $transaction->notes = 'تم صرف المخزون من إذن الصرف #' . $issue->issue_number;
            $transaction->save();
        }
    }

    /**
     * تغيير حالة إذن الصرف إلى معتمد
     */
    public function complete(InventoryIssue $inventoryIssue)
    {
        // لا يمكن اعتماد إذن الصرف إذا كان معتمداً بالفعل
        if ($inventoryIssue->status === 'completed') {
            return redirect()->route('inventory-issues.show', $inventoryIssue->id)
                ->with('error', 'إذن الصرف معتمد بالفعل');
        }

        try {
            DB::beginTransaction();

            // تحديث حالة إذن الصرف
            $inventoryIssue->update([
                'status' => 'completed',
                'completed_at' => now(),
            ]);

            // تحديث المخزون بناءً على إذن الصرف
            $this->updateInventory($inventoryIssue);

            DB::commit();

            return redirect()->route('inventory-issues.show', $inventoryIssue->id)
                ->with('success', 'تم اعتماد إذن الصرف وتحديث المخزون بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ أثناء اعتماد إذن الصرف: ' . $e->getMessage());
        }
    }

    /**
     * الحصول على معلومات أمر الشغل عبر AJAX
     */
    public function getWorkOrderInfo(Request $request)
    {
        // تسجيل الطلب للتصحيح بشكل مفصل
        \Log::info('=== بدء معالجة طلب معلومات أمر الشغل ===', [
            'time' => now()->toDateTimeString(),
            'work_order_id' => $request->input('work_order_id'),
            'all_data' => $request->all(),
            'request_url' => $request->fullUrl(),
            'request_method' => $request->method(),
            'request_headers' => $request->header(),
            'request_ip' => $request->ip()
        ]);

        // إضافة تأخير مؤقت للتأكد من أن المشكلة ليست في توقيت الاستجابة
        sleep(1);

        $workOrderId = $request->input('work_order_id');
        \Log::info('معرف أمر الشغل المستلم', ['work_order_id' => $workOrderId]);

        if (!$workOrderId) {
            \Log::warning('معرف أمر الشغل مفقود في الطلب');
            return response()->json(['error' => 'معرف أمر الشغل مفقود', 'debug' => 'work_order_id parameter is missing'], 400);
        }

        try {
            // استخدام eager loading للعلاقات
            \Log::info('جاري البحث عن أمر الشغل', ['work_order_id' => $workOrderId]);
            $workOrder = WorkOrder::with(['elevator', 'projectPhase', 'elevatorStage'])->find($workOrderId);

            if (!$workOrder) {
                \Log::warning('أمر الشغل غير موجود', ['work_order_id' => $workOrderId]);
                return response()->json(['error' => 'أمر الشغل غير موجود', 'debug' => 'Work order not found in database'], 404);
            }

            \Log::info('تم العثور على أمر الشغل', ['work_order' => $workOrder->toArray()]);

            // التحقق من وجود المصعد
            \Log::info('التحقق من وجود المصعد', ['elevator_id' => $workOrder->elevator_id]);
            if (!$workOrder->elevator_id) {
                \Log::warning('معرف المصعد غير موجود في أمر الشغل', ['work_order_id' => $workOrderId]);
            }

            // التحقق من وجود مرحلة المصعد
            \Log::info('التحقق من وجود مرحلة المصعد', ['elevator_stage_id' => $workOrder->elevator_stage_id]);
            if (!$workOrder->elevator_stage_id) {
                \Log::warning('معرف مرحلة المصعد غير موجود في أمر الشغل', ['work_order_id' => $workOrderId]);
            }

            // تسجيل بيانات أمر الشغل للتصحيح
            \Log::info('بيانات أمر الشغل', [
                'id' => $workOrder->id,
                'elevator_id' => $workOrder->elevator_id,
                'elevator_relation_loaded' => $workOrder->relationLoaded('elevator'),
                'elevator' => $workOrder->elevator ? [
                    'id' => $workOrder->elevator->id,
                    'serial_number' => $workOrder->elevator->serial_number
                ] : 'العلاقة فارغة',
                'elevator_stage_id' => $workOrder->elevator_stage_id,
                'elevatorStage_relation_loaded' => $workOrder->relationLoaded('elevatorStage'),
                'elevatorStage' => $workOrder->elevatorStage ? [
                    'id' => $workOrder->elevatorStage->id,
                    'name' => $workOrder->elevatorStage->name
                ] : 'العلاقة فارغة'
            ]);

            // التحقق من وجود المصعد والمرحلة قبل محاولة الوصول إلى خصائصهما
            $elevatorNumber = null;
            $phaseName = null;

            // التحقق من وجود المصعد
            if ($workOrder->elevator) {
                \Log::info('تم العثور على المصعد', ['elevator' => $workOrder->elevator->toArray()]);
                $elevatorNumber = $workOrder->elevator->serial_number;
                \Log::info('تم استخراج رقم المصعد', ['serial_number' => $elevatorNumber]);
            } else {
                \Log::warning('المصعد غير موجود لأمر الشغل', ['work_order_id' => $workOrderId]);

                // محاولة تحميل العلاقة مرة أخرى
                \Log::info('محاولة تحميل علاقة المصعد مرة أخرى');
                $workOrder->load('elevator');

                if ($workOrder->elevator) {
                    \Log::info('تم تحميل المصعد بنجاح في المحاولة الثانية');
                    $elevatorNumber = $workOrder->elevator->serial_number;
                    \Log::info('تم استخراج رقم المصعد في المحاولة الثانية', ['serial_number' => $elevatorNumber]);
                } else {
                    \Log::error('فشل تحميل المصعد في المحاولة الثانية');
                }
            }

            // التحقق من وجود مرحلة المصعد
            if ($workOrder->elevatorStage) {
                \Log::info('تم العثور على مرحلة المصعد', ['elevatorStage' => $workOrder->elevatorStage->toArray()]);
                $phaseName = $workOrder->elevatorStage->name;
                \Log::info('تم استخراج اسم مرحلة المصعد', ['name' => $phaseName]);
            } else {
                \Log::warning('مرحلة المصعد غير موجودة لأمر الشغل', ['work_order_id' => $workOrderId]);

                // محاولة تحميل العلاقة مرة أخرى
                \Log::info('محاولة تحميل علاقة مرحلة المصعد مرة أخرى');
                $workOrder->load('elevatorStage');

                if ($workOrder->elevatorStage) {
                    \Log::info('تم تحميل مرحلة المصعد بنجاح في المحاولة الثانية');
                    $phaseName = $workOrder->elevatorStage->name;
                    \Log::info('تم استخراج اسم مرحلة المصعد في المحاولة الثانية', ['name' => $phaseName]);
                } else {
                    \Log::error('فشل تحميل مرحلة المصعد في المحاولة الثانية');
                }
            }

            // التأكد من أن القيم ليست فارغة
            if (empty($elevatorNumber)) {
                \Log::warning('رقم المصعد فارغ، محاولة الحصول عليه مباشرة من قاعدة البيانات');

                // محاولة الحصول على رقم المصعد مباشرة من قاعدة البيانات
                $elevator = \App\Models\Elevator::find($workOrder->elevator_id);
                if ($elevator) {
                    $elevatorNumber = $elevator->serial_number;
                    \Log::info('تم الحصول على رقم المصعد مباشرة من قاعدة البيانات', ['serial_number' => $elevatorNumber]);
                } else {
                    \Log::error('لم يتم العثور على المصعد في قاعدة البيانات', ['elevator_id' => $workOrder->elevator_id]);
                }
            }

            if (empty($phaseName)) {
                \Log::warning('اسم مرحلة المصعد فارغ، محاولة الحصول عليه مباشرة من قاعدة البيانات');

                // محاولة الحصول على اسم مرحلة المصعد مباشرة من قاعدة البيانات
                $elevatorStage = \App\Models\ElevatorStage::find($workOrder->elevator_stage_id);
                if ($elevatorStage) {
                    $phaseName = $elevatorStage->name;
                    \Log::info('تم الحصول على اسم مرحلة المصعد مباشرة من قاعدة البيانات', ['name' => $phaseName]);
                } else {
                    \Log::error('لم يتم العثور على مرحلة المصعد في قاعدة البيانات', ['elevator_stage_id' => $workOrder->elevator_stage_id]);
                }
            }

            // إضافة معلومات عن وجود المرحلة
            $hasPhase = !empty($workOrder->elevator_stage_id) && !empty($phaseName);

            $response = [
                'elevator_id' => $workOrder->elevator_id,
                'elevator_number' => $elevatorNumber ?: 'غير محدد',
                'phase_id' => $workOrder->elevator_stage_id,
                'phase_name' => $phaseName ?: 'غير محدد',
                'has_phase' => $hasPhase,
                'debug_info' => [
                    'timestamp' => now()->toDateTimeString(),
                    'work_order_found' => true,
                    'elevator_relation_loaded' => $workOrder->relationLoaded('elevator'),
                    'elevatorStage_relation_loaded' => $workOrder->relationLoaded('elevatorStage'),
                    'direct_query_used' => empty($workOrder->elevator) || empty($workOrder->elevatorStage),
                    'elevator_stage_id_exists' => !empty($workOrder->elevator_stage_id),
                    'phase_name_exists' => !empty($phaseName)
                ],
                // إضافة معلومات إضافية للتأكد من أن البيانات موجودة في الاستجابة
                'work_order_id' => $workOrder->id,
                'request_time' => now()->toDateTimeString(),
                'response_generated' => true
            ];

            \Log::info('إرسال الاستجابة', $response);

            // التأكد من أن الاستجابة تحتوي على البيانات المطلوبة
            if (empty($response['elevator_number']) || empty($response['phase_name'])) {
                \Log::warning('الاستجابة تحتوي على قيم فارغة', [
                    'elevator_number' => $response['elevator_number'],
                    'phase_name' => $response['phase_name']
                ]);
            }

            // إضافة رأس HTTP لمنع التخزين المؤقت
            return response()->json($response)
                ->header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0')
                ->header('Pragma', 'no-cache')
                ->header('Expires', '0');
        } catch (\Exception $e) {
            \Log::error('حدث خطأ أثناء معالجة طلب معلومات أمر الشغل', [
                'work_order_id' => $workOrderId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'error' => 'حدث خطأ أثناء معالجة الطلب',
                'debug' => $e->getMessage(),
                'work_order_id' => $workOrderId
            ], 500);
        } finally {
            \Log::info('=== انتهاء معالجة طلب معلومات أمر الشغل ===');
        }
    }
}
