<?php

namespace App\Http\Controllers;

use App\Models\InventoryItem;
use App\Models\InventoryTransaction;
use App\Models\PurchaseOrder;
use App\Models\InventoryStock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class InventoryDashboardController extends Controller
{
    /**
     * عرض لوحة معلومات المخزون
     */
    public function index()
    {
        $user = Auth::user();
        $company = $user->company;

        // إحصائيات العناصر
        $totalItems = InventoryItem::where('company_id', $company->id)->count();

        // العناصر منخفضة المخزون
        $lowStockItems = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->with(['category', 'stocks.warehouse'])
            ->get()
            ->filter(function ($item) {
                return $item->is_low_stock;
            })
            ->take(10);

        $lowStockCount = $lowStockItems->count();

        // قيمة المخزون
        $stockValue = InventoryStock::whereHas('item', function ($query) use ($company) {
            $query->where('company_id', $company->id);
        })
            ->join('inventory_items', 'inventory_stocks.item_id', '=', 'inventory_items.id')
            ->select(DB::raw('SUM(inventory_stocks.quantity * inventory_items.purchase_price) as total_value'))
            ->first()
            ->total_value ?? 0;

        // طلبات الشراء المعلقة
        $pendingPOCount = PurchaseOrder::where('company_id', $company->id)
            ->whereIn('status', ['draft', 'approved', 'partial'])
            ->count();

        // آخر المعاملات
        $recentTransactions = InventoryTransaction::whereHas('item', function ($query) use ($company) {
            $query->where('company_id', $company->id);
        })
            ->with('item')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // طلبات الشراء الأخيرة
        $recentPurchaseOrders = PurchaseOrder::where('company_id', $company->id)
            ->with(['supplier', 'warehouse'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        return view('inventory.dashboard', compact(
            'totalItems',
            'lowStockItems',
            'lowStockCount',
            'stockValue',
            'pendingPOCount',
            'recentTransactions',
            'recentPurchaseOrders'
        ));
    }
}
