<?php

namespace App\Http\Controllers;

use App\Models\InventoryCount;
use App\Models\InventoryCountItem;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\InventoryTransaction;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class InventoryCountController extends Controller
{
    /**
     * Display a listing of the inventory counts.
     */
    public function index()
    {
        $counts = InventoryCount::with(['warehouse', 'createdBy'])
            ->where('company_id', auth()->user()->company_id)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('inventory.counts.index', compact('counts'));
    }

    /**
     * Show the form for creating a new inventory count.
     */
    public function create()
    {
        $warehouses = Warehouse::where('company_id', auth()->user()->company_id)
            ->where('is_active', true)
            ->get();

        return view('inventory.counts.create', compact('warehouses'));
    }

    /**
     * Store a newly created inventory count in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'warehouse_id' => 'required|exists:warehouses,id',
            'count_date' => 'required|date',
            'notes' => 'nullable|string|max:1000',
        ]);

        // Generate a unique count number
        $countNumber = 'CNT-' . date('Ymd') . '-' . Str::upper(Str::random(5));

        DB::beginTransaction();

        try {
            // Create the inventory count
            $count = InventoryCount::create([
                'company_id' => auth()->user()->company_id,
                'warehouse_id' => $request->warehouse_id,
                'count_number' => $countNumber,
                'count_date' => $request->count_date,
                'status' => 'draft',
                'created_by' => auth()->id(),
                'notes' => $request->notes,
            ]);

            // Get all items in the warehouse
            $stocks = InventoryStock::with('item')
                ->where('warehouse_id', $request->warehouse_id)
                ->where('quantity', '>', 0)
                ->get();

            // Create count items for each stock
            foreach ($stocks as $stock) {
                InventoryCountItem::create([
                    'inventory_count_id' => $count->id,
                    'item_id' => $stock->item_id,
                    'system_quantity' => $stock->quantity,
                    'counted_quantity' => 0,
                    'is_counted' => false,
                ]);
            }

            DB::commit();

            return redirect()->route('inventory.counts.show', $count->id)
                ->with('success', 'Inventory count created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'Failed to create inventory count: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified inventory count.
     */
    public function show(string $id)
    {
        $count = InventoryCount::with(['warehouse', 'createdBy', 'approvedBy', 'items.item', 'items.counter'])
            ->findOrFail($id);

        // Check if user belongs to the same company
        if ($count->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        return view('inventory.counts.show', compact('count'));
    }

    /**
     * Show the form for counting items.
     */
    public function count(string $id)
    {
        $count = InventoryCount::with(['warehouse', 'items.item'])
            ->findOrFail($id);

        // Check if user belongs to the same company
        if ($count->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Check if count is in draft status
        if ($count->status !== 'draft') {
            return redirect()->route('inventory.counts.show', $count->id)
                ->with('error', 'This inventory count cannot be modified.');
        }

        return view('inventory.counts.count', compact('count'));
    }

    /**
     * Update the counted quantity for an item.
     */
    public function updateItem(Request $request, string $id, string $itemId)
    {
        $request->validate([
            'counted_quantity' => 'required|numeric|min:0',
            'notes' => 'nullable|string|max:500',
        ]);

        $countItem = InventoryCountItem::where('inventory_count_id', $id)
            ->where('id', $itemId)
            ->firstOrFail();

        // Check if count is in draft status
        $count = InventoryCount::findOrFail($id);
        if ($count->status !== 'draft') {
            return redirect()->route('inventory.counts.show', $count->id)
                ->with('error', 'This inventory count cannot be modified.');
        }

        $countItem->update([
            'counted_quantity' => $request->counted_quantity,
            'is_counted' => true,
            'counted_by' => auth()->id(),
            'counted_at' => now(),
            'notes' => $request->notes,
        ]);

        return redirect()->back()
            ->with('success', 'Item counted successfully.');
    }

    /**
     * Complete the inventory count.
     */
    public function complete(string $id)
    {
        $count = InventoryCount::findOrFail($id);

        // Check if user belongs to the same company
        if ($count->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Check if count is in draft status
        if ($count->status !== 'draft') {
            return redirect()->route('inventory.counts.show', $count->id)
                ->with('error', 'This inventory count cannot be completed.');
        }

        // Check if all items have been counted
        $uncountedItems = $count->items()->where('is_counted', false)->count();
        if ($uncountedItems > 0) {
            return redirect()->route('inventory.counts.show', $count->id)
                ->with('error', "There are {$uncountedItems} uncounted items. Please count all items before completing.");
        }

        $count->update([
            'status' => 'completed',
        ]);

        return redirect()->route('inventory.counts.show', $count->id)
            ->with('success', 'Inventory count completed successfully.');
    }

    /**
     * Approve the inventory count and adjust inventory.
     */
    public function approve(Request $request, string $id)
    {
        $count = InventoryCount::with('items.item')
            ->findOrFail($id);

        // Check if user belongs to the same company
        if ($count->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Check if count is in completed status
        if ($count->status !== 'completed') {
            return redirect()->route('inventory.counts.show', $count->id)
                ->with('error', 'This inventory count cannot be approved.');
        }

        DB::beginTransaction();

        try {
            // Update the count status
            $count->update([
                'status' => 'approved',
                'approved_by' => auth()->id(),
                'approved_at' => now(),
            ]);

            // Process each item with discrepancy
            foreach ($count->items as $countItem) {
                if (abs($countItem->discrepancy) > 0) {
                    // Get the stock record
                    $stock = InventoryStock::where('warehouse_id', $count->warehouse_id)
                        ->where('item_id', $countItem->item_id)
                        ->first();

                    if (!$stock) {
                        // Create a new stock record if it doesn't exist
                        $stock = InventoryStock::create([
                            'company_id' => $count->company_id,
                            'warehouse_id' => $count->warehouse_id,
                            'item_id' => $countItem->item_id,
                            'quantity' => 0,
                            'reserved_quantity' => 0,
                        ]);
                    }

                    // Update the stock quantity
                    $stock->quantity = $countItem->counted_quantity;
                    $stock->save();

                    // Create an inventory transaction
                    InventoryTransaction::create([
                        'company_id' => $count->company_id,
                        'transaction_type' => 'adjustment',
                        'item_id' => $countItem->item_id,
                        'warehouse_id' => $count->warehouse_id,
                        'quantity' => $countItem->discrepancy,
                        'reference_number' => $count->count_number,
                        'reference_type' => 'inventory_count',
                        'reference_id' => $count->id,
                        'created_by' => auth()->id(),
                        'notes' => "Inventory adjustment from count {$count->count_number}",
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('inventory.counts.show', $count->id)
                ->with('success', 'Inventory count approved and stock adjusted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'Failed to approve inventory count: ' . $e->getMessage());
        }
    }

    /**
     * Cancel the inventory count.
     */
    public function cancel(string $id)
    {
        $count = InventoryCount::findOrFail($id);

        // Check if user belongs to the same company
        if ($count->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Check if count is in draft or completed status
        if (!in_array($count->status, ['draft', 'completed'])) {
            return redirect()->route('inventory.counts.show', $count->id)
                ->with('error', 'This inventory count cannot be cancelled.');
        }

        $count->update([
            'status' => 'cancelled',
        ]);

        return redirect()->route('inventory.counts.show', $count->id)
            ->with('success', 'Inventory count cancelled successfully.');
    }

    /**
     * Print the inventory count.
     */
    public function print(string $id)
    {
        $count = InventoryCount::with(['warehouse', 'createdBy', 'approvedBy', 'items.item', 'items.counter'])
            ->findOrFail($id);

        // Check if user belongs to the same company
        if ($count->company_id !== auth()->user()->company_id) {
            abort(403, 'Unauthorized action.');
        }

        return view('inventory.counts.print', compact('count'));
    }
}
