<?php

namespace App\Http\Controllers;

use App\Models\InventoryCategory;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class InventoryCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $company = $user->company;

        $categories = InventoryCategory::where('company_id', $company->id)
            ->with('parent')
            ->orderBy('name')
            ->get();

        return view('inventory.categories.index', compact('categories'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $company = $user->company;

        $parentCategories = InventoryCategory::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('inventory.categories.create', compact('parentCategories'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:inventory_categories,code',
            'parent_id' => 'nullable|exists:inventory_categories,id',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Check for circular reference
        if ($request->parent_id) {
            $parent = InventoryCategory::find($request->parent_id);
            if ($parent && $parent->parent_id == $request->id) {
                return redirect()->back()
                    ->with('error', 'لا يمكن إنشاء تسلسل دائري في التصنيفات')
                    ->withInput();
            }
        }

        $category = new InventoryCategory();
        $category->company_id = $company->id;
        $category->name = $request->name;
        $category->code = $request->code;
        $category->parent_id = $request->parent_id;
        $category->description = $request->description;
        $category->is_active = $request->has('is_active');
        $category->save();

        return redirect()->route('inventory.categories.index')
            ->with('success', 'تم إنشاء التصنيف بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $category = InventoryCategory::where('company_id', $company->id)
            ->where('id', $id)
            ->with(['parent', 'children', 'items'])
            ->firstOrFail();

        return view('inventory.categories.show', compact('category'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $category = InventoryCategory::where('company_id', $company->id)
            ->where('id', $id)
            ->firstOrFail();

        $parentCategories = InventoryCategory::where('company_id', $company->id)
            ->where('is_active', true)
            ->where('id', '!=', $id)
            ->orderBy('name')
            ->get();

        return view('inventory.categories.edit', compact('category', 'parentCategories'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $company = $user->company;

        $category = InventoryCategory::where('company_id', $company->id)
            ->where('id', $id)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:inventory_categories,code,' . $category->id,
            'parent_id' => 'nullable|exists:inventory_categories,id',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Check for circular reference
        if ($request->parent_id) {
            if ($request->parent_id == $id) {
                return redirect()->back()
                    ->with('error', 'لا يمكن جعل التصنيف أب لنفسه')
                    ->withInput();
            }

            $parent = InventoryCategory::find($request->parent_id);
            $parentIds = [];

            // Get all parent IDs to check for circular reference
            while ($parent) {
                $parentIds[] = $parent->id;
                $parent = $parent->parent;
            }

            if (in_array($id, $parentIds)) {
                return redirect()->back()
                    ->with('error', 'لا يمكن إنشاء تسلسل دائري في التصنيفات')
                    ->withInput();
            }
        }

        $category->name = $request->name;
        $category->code = $request->code;
        $category->parent_id = $request->parent_id;
        $category->description = $request->description;
        $category->is_active = $request->has('is_active');
        $category->save();

        return redirect()->route('inventory.categories.index')
            ->with('success', 'تم تحديث التصنيف بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $category = InventoryCategory::where('company_id', $company->id)
            ->where('id', $id)
            ->firstOrFail();

        // Check if category has children
        $hasChildren = $category->children()->count() > 0;

        if ($hasChildren) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف التصنيف لأنه يحتوي على تصنيفات فرعية');
        }

        // Check if category has items
        $hasItems = $category->items()->count() > 0;

        if ($hasItems) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف التصنيف لأنه يحتوي على عناصر');
        }

        $category->delete();

        return redirect()->route('inventory.categories.index')
            ->with('success', 'تم حذف التصنيف بنجاح');
    }
}
