<?php

namespace App\Http\Controllers;

use App\Models\InventoryAlert;
use App\Models\InventoryBatch;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Mail\LowStockAlert;
use App\Mail\ExpiryAlert;

class InventoryAlertController extends Controller
{
    /**
     * Display the alert settings page.
     */
    public function settings()
    {
        $alert = InventoryAlert::where('company_id', auth()->user()->company_id)->first();

        if (!$alert) {
            // Create default alert settings if not exists
            $alert = InventoryAlert::create([
                'company_id' => auth()->user()->company_id,
                'low_stock_enabled' => true,
                'low_stock_threshold' => 10, // 10%
                'expiry_enabled' => true,
                'expiry_threshold_days' => 30,
                'notification_emails' => [],
                'notification_users' => [],
            ]);
        }

        $users = User::where('company_id', auth()->user()->company_id)
            ->where('is_active', true)
            ->get();

        return view('inventory.alerts.settings', compact('alert', 'users'));
    }

    /**
     * Update the alert settings.
     */
    public function updateSettings(Request $request)
    {
        $request->validate([
            'low_stock_enabled' => 'boolean',
            'low_stock_threshold' => 'required|numeric|min:1|max:100',
            'expiry_enabled' => 'boolean',
            'expiry_threshold_days' => 'required|integer|min:1|max:365',
            'notification_emails' => 'nullable|array',
            'notification_emails.*' => 'email',
            'notification_users' => 'nullable|array',
            'notification_users.*' => 'exists:users,id',
        ]);

        $alert = InventoryAlert::where('company_id', auth()->user()->company_id)->first();

        if (!$alert) {
            $alert = new InventoryAlert();
            $alert->company_id = auth()->user()->company_id;
        }

        $alert->low_stock_enabled = $request->has('low_stock_enabled');
        $alert->low_stock_threshold = $request->low_stock_threshold;
        $alert->expiry_enabled = $request->has('expiry_enabled');
        $alert->expiry_threshold_days = $request->expiry_threshold_days;
        $alert->notification_emails = $request->notification_emails ?? [];
        $alert->notification_users = $request->notification_users ?? [];
        $alert->save();

        return redirect()->route('inventory.alerts.settings')
            ->with('success', 'Alert settings updated successfully.');
    }

    /**
     * Display the low stock alerts.
     */
    public function lowStock()
    {
        $alert = InventoryAlert::where('company_id', auth()->user()->company_id)->first();

        if (!$alert || !$alert->low_stock_enabled) {
            return view('inventory.alerts.low-stock', [
                'items' => collect(),
                'alertsEnabled' => false,
            ]);
        }

        // Get items with low stock
        $items = InventoryItem::with(['stocks.warehouse', 'category'])
            ->where('company_id', auth()->user()->company_id)
            ->where('is_active', true)
            ->get()
            ->filter(function ($item) use ($alert) {
                // Calculate the percentage of current stock compared to minimum stock level
                if ($item->min_stock_level <= 0) {
                    return false;
                }

                $percentage = ($item->available_stock / $item->min_stock_level) * 100;
                return $percentage <= $alert->low_stock_threshold;
            });

        return view('inventory.alerts.low-stock', [
            'items' => $items,
            'alertsEnabled' => true,
            'threshold' => $alert->low_stock_threshold,
        ]);
    }

    /**
     * Display the expiry alerts.
     */
    public function expiry()
    {
        $alert = InventoryAlert::where('company_id', auth()->user()->company_id)->first();

        if (!$alert || !$alert->expiry_enabled) {
            return view('inventory.alerts.expiry', [
                'batches' => collect(),
                'alertsEnabled' => false,
            ]);
        }

        // Get batches that will expire soon
        $expiryDate = now()->addDays($alert->expiry_threshold_days);

        $batches = InventoryBatch::with(['item', 'warehouse'])
            ->where('company_id', auth()->user()->company_id)
            ->where('expiry_date', '<=', $expiryDate)
            ->where('expiry_date', '>=', now())
            ->where('quantity', '>', 0)
            ->orderBy('expiry_date')
            ->get();

        return view('inventory.alerts.expiry', [
            'batches' => $batches,
            'alertsEnabled' => true,
            'thresholdDays' => $alert->expiry_threshold_days,
        ]);
    }

    /**
     * Send low stock alerts manually.
     */
    public function sendLowStockAlerts()
    {
        $alert = InventoryAlert::where('company_id', auth()->user()->company_id)->first();

        if (!$alert || !$alert->low_stock_enabled) {
            return redirect()->route('inventory.alerts.low-stock')
                ->with('error', 'Low stock alerts are not enabled.');
        }

        // Get items with low stock
        $items = InventoryItem::with(['stocks.warehouse', 'category'])
            ->where('company_id', auth()->user()->company_id)
            ->where('is_active', true)
            ->get()
            ->filter(function ($item) use ($alert) {
                if ($item->min_stock_level <= 0) {
                    return false;
                }

                $percentage = ($item->available_stock / $item->min_stock_level) * 100;
                return $percentage <= $alert->low_stock_threshold;
            });

        if ($items->isEmpty()) {
            return redirect()->route('inventory.alerts.low-stock')
                ->with('info', 'No low stock items found.');
        }

        // Send email notifications
        if (!empty($alert->notification_emails)) {
            foreach ($alert->notification_emails as $email) {
                Mail::to($email)->send(new LowStockAlert($items, $alert));
            }
        }

        // Send notifications to users
        if (!empty($alert->notification_users)) {
            $users = User::whereIn('id', $alert->notification_users)->get();
            foreach ($users as $user) {
                $user->notify(new \App\Notifications\LowStockAlert($items, $alert));
            }
        }

        return redirect()->route('inventory.alerts.low-stock')
            ->with('success', 'Low stock alerts sent successfully.');
    }

    /**
     * Send expiry alerts manually.
     */
    public function sendExpiryAlerts()
    {
        $alert = InventoryAlert::where('company_id', auth()->user()->company_id)->first();

        if (!$alert || !$alert->expiry_enabled) {
            return redirect()->route('inventory.alerts.expiry')
                ->with('error', 'Expiry alerts are not enabled.');
        }

        // Get batches that will expire soon
        $expiryDate = now()->addDays($alert->expiry_threshold_days);

        $batches = InventoryBatch::with(['item', 'warehouse'])
            ->where('company_id', auth()->user()->company_id)
            ->where('expiry_date', '<=', $expiryDate)
            ->where('expiry_date', '>=', now())
            ->where('quantity', '>', 0)
            ->orderBy('expiry_date')
            ->get();

        if ($batches->isEmpty()) {
            return redirect()->route('inventory.alerts.expiry')
                ->with('info', 'No expiring batches found.');
        }

        // Send email notifications
        if (!empty($alert->notification_emails)) {
            foreach ($alert->notification_emails as $email) {
                Mail::to($email)->send(new ExpiryAlert($batches, $alert));
            }
        }

        // Send notifications to users
        if (!empty($alert->notification_users)) {
            $users = User::whereIn('id', $alert->notification_users)->get();
            foreach ($users as $user) {
                $user->notify(new \App\Notifications\ExpiryAlert($batches, $alert));
            }
        }

        return redirect()->route('inventory.alerts.expiry')
            ->with('success', 'Expiry alerts sent successfully.');
    }
}
