<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\InventoryTransaction;
use App\Models\InventoryValuation;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class InventoryValuationController extends Controller
{
    /**
     * عرض صفحة تقييم المخزون
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_inventory_valuation')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        $warehouses = Warehouse::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->get();

        $valuationMethods = [
            'fifo' => 'الوارد أولاً صادر أولاً (FIFO)',
            'lifo' => 'الوارد أخيراً صادر أولاً (LIFO)',
            'average' => 'متوسط التكلفة (Average Cost)',
            'weighted_average' => 'المتوسط المرجح (Weighted Average)',
        ];

        return view('inventory.valuation.index', compact('warehouses', 'valuationMethods'));
    }

    /**
     * عرض تقرير تقييم المخزون
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function report(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_inventory_valuation')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        $validated = $request->validate([
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'category_id' => 'nullable|exists:inventory_categories,id',
            'valuation_method' => 'required|in:fifo,lifo,average,weighted_average',
            'as_of_date' => 'required|date',
        ]);

        $asOfDate = $validated['as_of_date'];
        $warehouseId = $validated['warehouse_id'];
        $categoryId = $validated['category_id'];
        $valuationMethod = $validated['valuation_method'];

        // بناء الاستعلام الأساسي
        $query = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true);

        // تصفية حسب الفئة إذا تم تحديدها
        if ($categoryId) {
            $query->where('category_id', $categoryId);
        }

        // الحصول على العناصر
        $items = $query->get();

        // حساب تقييم المخزون لكل عنصر
        $valuationResults = [];
        $totalValue = 0;

        foreach ($items as $item) {
            // الحصول على مستوى المخزون الحالي
            $stockQuery = InventoryStock::where('item_id', $item->id);
            if ($warehouseId) {
                $stockQuery->where('warehouse_id', $warehouseId);
            }
            $currentStock = $stockQuery->sum('quantity');

            // تخطي العناصر التي ليس لها مخزون
            if ($currentStock <= 0) {
                continue;
            }

            // حساب تكلفة الوحدة بناءً على طريقة التقييم
            $unitCost = $this->calculateUnitCost($item->id, $warehouseId, $valuationMethod, $asOfDate);
            $totalCost = $currentStock * $unitCost;
            $totalValue += $totalCost;

            $valuationResults[] = [
                'item' => $item,
                'current_stock' => $currentStock,
                'unit_cost' => $unitCost,
                'total_cost' => $totalCost,
            ];
        }

        // الحصول على المستودع إذا تم تحديده
        $warehouse = $warehouseId ? Warehouse::find($warehouseId) : null;

        $valuationMethods = [
            'fifo' => 'الوارد أولاً صادر أولاً (FIFO)',
            'lifo' => 'الوارد أخيراً صادر أولاً (LIFO)',
            'average' => 'متوسط التكلفة (Average Cost)',
            'weighted_average' => 'المتوسط المرجح (Weighted Average)',
        ];

        return view('inventory.valuation.report', compact(
            'valuationResults',
            'totalValue',
            'warehouse',
            'valuationMethod',
            'valuationMethods',
            'asOfDate'
        ));
    }

    /**
     * عرض سجل تقييم المخزون
     *
     * @return \Illuminate\Http\Response
     */
    public function history()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_inventory_valuation_history')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        $valuations = InventoryValuation::with(['item', 'warehouse', 'creator'])
            ->where('company_id', Auth::user()->company_id)
            ->orderBy('valuation_date', 'desc')
            ->paginate(15);

        return view('inventory.valuation.history', compact('valuations'));
    }

    /**
     * إنشاء تقييم مخزون جديد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('create_inventory_valuation')) {
            return redirect()->route('inventory.valuation.index')->with('error', 'ليس لديك صلاحية لإنشاء تقييم مخزون');
        }

        $validated = $request->validate([
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'valuation_method' => 'required|in:fifo,lifo,average,weighted_average',
            'valuation_date' => 'required|date',
        ]);

        $warehouseId = $validated['warehouse_id'];
        $valuationMethod = $validated['valuation_method'];
        $valuationDate = $validated['valuation_date'];

        // بناء الاستعلام الأساسي
        $query = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true);

        // الحصول على العناصر
        $items = $query->get();

        DB::beginTransaction();
        try {
            $createdCount = 0;

            foreach ($items as $item) {
                // الحصول على مستوى المخزون الحالي
                $stockQuery = InventoryStock::where('item_id', $item->id);
                if ($warehouseId) {
                    $stockQuery->where('warehouse_id', $warehouseId);
                }

                // الحصول على جميع سجلات المخزون للعنصر
                $stocks = $stockQuery->get();

                foreach ($stocks as $stock) {
                    // تخطي العناصر التي ليس لها مخزون
                    if ($stock->quantity <= 0) {
                        continue;
                    }

                    // حساب تكلفة الوحدة بناءً على طريقة التقييم
                    $unitCost = $this->calculateUnitCost($item->id, $stock->warehouse_id, $valuationMethod, $valuationDate);
                    $totalCost = $stock->quantity * $unitCost;

                    // إنشاء سجل تقييم جديد
                    InventoryValuation::create([
                        'company_id' => Auth::user()->company_id,
                        'item_id' => $item->id,
                        'warehouse_id' => $stock->warehouse_id,
                        'valuation_date' => $valuationDate,
                        'quantity' => $stock->quantity,
                        'unit_cost' => $unitCost,
                        'total_cost' => $totalCost,
                        'valuation_method' => $valuationMethod,
                        'created_by' => Auth::id(),
                    ]);

                    $createdCount++;
                }
            }

            DB::commit();

            return redirect()->route('inventory.valuation.history')
                ->with('success', "تم إنشاء {$createdCount} سجل تقييم مخزون بنجاح");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('inventory.valuation.index')
                ->with('error', 'حدث خطأ أثناء إنشاء تقييم المخزون: ' . $e->getMessage());
        }
    }

    /**
     * حساب تكلفة الوحدة بناءً على طريقة التقييم
     *
     * @param int $itemId
     * @param int|null $warehouseId
     * @param string $method
     * @param string $asOfDate
     * @return float
     */
    private function calculateUnitCost($itemId, $warehouseId, $method, $asOfDate)
    {
        // بناء الاستعلام الأساسي للمعاملات
        $query = InventoryTransaction::where('item_id', $itemId)
            ->whereDate('created_at', '<=', $asOfDate)
            ->whereIn('transaction_type', ['purchase', 'return_from_customer', 'adjustment_in'])
            ->where('unit_price', '>', 0);

        if ($warehouseId) {
            $query->where(function ($q) use ($warehouseId) {
                $q->where('warehouse_id', $warehouseId)
                    ->orWhere('to_warehouse_id', $warehouseId);
            });
        }

        // تطبيق طريقة التقييم
        switch ($method) {
            case 'fifo':
                // الوارد أولاً صادر أولاً - استخدام أقدم المعاملات أولاً
                $transactions = $query->orderBy('created_at', 'asc')->get();
                if ($transactions->isEmpty()) {
                    return $this->getFallbackCost($itemId);
                }

                // تطبيق FIFO بشكل أكثر دقة
                $remainingQuantity = $this->getCurrentStock($itemId, $warehouseId);
                $totalValue = 0;

                foreach ($transactions as $transaction) {
                    $transactionQuantity = min($remainingQuantity, $transaction->quantity);
                    if ($transactionQuantity <= 0) break;

                    $totalValue += $transactionQuantity * $transaction->unit_price;
                    $remainingQuantity -= $transactionQuantity;
                }

                $currentStock = $this->getCurrentStock($itemId, $warehouseId);
                return $currentStock > 0 ? $totalValue / $currentStock : $transactions->first()->unit_price;

            case 'lifo':
                // الوارد أخيراً صادر أولاً - استخدام أحدث المعاملات أولاً
                $transactions = $query->orderBy('created_at', 'desc')->get();
                if ($transactions->isEmpty()) {
                    return $this->getFallbackCost($itemId);
                }

                // تطبيق LIFO بشكل أكثر دقة
                $remainingQuantity = $this->getCurrentStock($itemId, $warehouseId);
                $totalValue = 0;

                foreach ($transactions as $transaction) {
                    $transactionQuantity = min($remainingQuantity, $transaction->quantity);
                    if ($transactionQuantity <= 0) break;

                    $totalValue += $transactionQuantity * $transaction->unit_price;
                    $remainingQuantity -= $transactionQuantity;
                }

                $currentStock = $this->getCurrentStock($itemId, $warehouseId);
                return $currentStock > 0 ? $totalValue / $currentStock : $transactions->first()->unit_price;

            case 'average':
                // متوسط التكلفة - حساب متوسط تكلفة جميع المعاملات
                $transactions = $query->get();
                if ($transactions->isEmpty()) {
                    return $this->getFallbackCost($itemId);
                }

                $totalQuantity = $transactions->sum('quantity');
                $totalValue = 0;

                foreach ($transactions as $transaction) {
                    $totalValue += $transaction->quantity * $transaction->unit_price;
                }

                return $totalQuantity > 0 ? $totalValue / $totalQuantity : $this->getFallbackCost($itemId);

            case 'weighted_average':
                // المتوسط المرجح - حساب المتوسط المرجح مع مراعاة المعاملات السابقة
                return $this->calculateWeightedAverage($itemId, $warehouseId, $asOfDate);

            default:
                return $this->getFallbackCost($itemId);
        }
    }

    /**
     * حساب المتوسط المرجح للتكلفة
     *
     * @param int $itemId
     * @param int|null $warehouseId
     * @param string $asOfDate
     * @return float
     */
    private function calculateWeightedAverage($itemId, $warehouseId, $asOfDate)
    {
        // الحصول على جميع المعاملات مرتبة حسب التاريخ
        $query = InventoryTransaction::where('item_id', $itemId)
            ->whereDate('created_at', '<=', $asOfDate)
            ->orderBy('created_at', 'asc');

        if ($warehouseId) {
            $query->where(function ($q) use ($warehouseId) {
                $q->where('warehouse_id', $warehouseId)
                    ->orWhere('to_warehouse_id', $warehouseId);
            });
        }

        $transactions = $query->get();

        if ($transactions->isEmpty()) {
            return $this->getFallbackCost($itemId);
        }

        $currentQuantity = 0;
        $currentValue = 0;

        foreach ($transactions as $transaction) {
            // معالجة المعاملات الواردة
            if (in_array($transaction->transaction_type, ['purchase', 'return_from_customer', 'adjustment_in', 'transfer_in'])) {
                $transactionValue = $transaction->quantity * $transaction->unit_price;
                $currentValue += $transactionValue;
                $currentQuantity += $transaction->quantity;
            }
            // معالجة المعاملات الصادرة
            elseif (in_array($transaction->transaction_type, ['sale', 'return_to_supplier', 'adjustment_out', 'transfer_out'])) {
                // لا نغير التكلفة، فقط نقلل الكمية
                $currentQuantity -= $transaction->quantity;
            }

            // إعادة ضبط القيمة إذا أصبحت الكمية صفر أو سالبة
            if ($currentQuantity <= 0) {
                $currentQuantity = 0;
                $currentValue = 0;
            }
        }

        // حساب المتوسط المرجح
        return $currentQuantity > 0 ? $currentValue / $currentQuantity : $this->getFallbackCost($itemId);
    }

    /**
     * الحصول على الكمية الحالية في المخزون
     *
     * @param int $itemId
     * @param int|null $warehouseId
     * @return float
     */
    private function getCurrentStock($itemId, $warehouseId = null)
    {
        $query = InventoryStock::where('item_id', $itemId);

        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        return $query->sum('quantity');
    }

    /**
     * الحصول على تكلفة احتياطية إذا لم يتم العثور على معاملات
     *
     * @param int $itemId
     * @return float
     */
    private function getFallbackCost($itemId)
    {
        $item = InventoryItem::find($itemId);
        return $item->purchase_price ?? 0;
    }

    /**
     * حساب المتوسط المرجح للعنصر والمستودع المحددين (AJAX)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function calculateAverage(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_inventory_valuation')) {
            return response()->json(['success' => false, 'message' => 'ليس لديك صلاحية للوصول إلى هذه الوظيفة'], 403);
        }

        $validated = $request->validate([
            'item_id' => 'required|exists:inventory_items,id',
            'warehouse_id' => 'required|exists:warehouses,id',
        ]);

        $itemId = $validated['item_id'];
        $warehouseId = $validated['warehouse_id'];

        // الحصول على الكمية الحالية في المخزون
        $currentStock = $this->getCurrentStock($itemId, $warehouseId);

        if ($currentStock <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'لا يوجد مخزون متاح لهذا العنصر في المستودع المحدد'
            ]);
        }

        // حساب المتوسط المرجح
        $unitCost = $this->calculateWeightedAverage($itemId, $warehouseId, date('Y-m-d'));

        return response()->json([
            'success' => true,
            'quantity' => $currentStock,
            'unit_cost' => $unitCost,
            'total_cost' => $currentStock * $unitCost
        ]);
    }
}
