<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InventoryCount;
use App\Models\InventoryCountItem;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\InventoryTransaction;
use App\Models\Warehouse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class InventoryCountController extends Controller
{
    /**
     * عرض قائمة جلسات الجرد
     */
    public function index()
    {
        $user = Auth::user();
        $company = $user->company;

        $counts = InventoryCount::with(['warehouse', 'createdBy', 'completedBy', 'approvedBy'])
            ->where('company_id', $company->id)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('inventory.counts.index', compact('counts'));
    }

    /**
     * عرض نموذج إنشاء جلسة جرد جديدة
     */
    public function create()
    {
        $user = Auth::user();
        $company = $user->company;

        $warehouses = Warehouse::where('company_id', $company->id)->get();
        $items = InventoryItem::where('company_id', $company->id)
            ->where('is_active', true)
            ->get();

        return view('inventory.counts.create', compact('warehouses', 'items'));
    }

    /**
     * حفظ جلسة جرد جديدة
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'warehouse_id' => 'required|exists:warehouses,id',
            'count_date' => 'required|date',
            'notes' => 'nullable|string|max:500',
            'count_type' => 'required|in:full,partial,cycle',
            'items' => 'required_if:count_type,partial|array',
            'items.*' => 'exists:inventory_items,id',
            'category_id' => 'required_if:count_type,cycle|exists:inventory_categories,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company = $user->company;

        DB::beginTransaction();

        try {
            $count = new InventoryCount();
            $count->company_id = $company->id;
            $count->warehouse_id = $request->input('warehouse_id');
            $count->count_date = $request->input('count_date');
            $count->notes = $request->input('notes');
            $count->count_type = $request->input('count_type');
            $count->status = 'draft';
            $count->created_by = $user->id;
            $count->save();

            // إضافة العناصر للجرد
            if ($request->input('count_type') == 'full') {
                // جرد كامل - إضافة جميع العناصر في المستودع
                $stocks = InventoryStock::where('warehouse_id', $request->input('warehouse_id'))
                    ->where('company_id', $company->id)
                    ->get();

                foreach ($stocks as $stock) {
                    $countItem = new InventoryCountItem();
                    $countItem->inventory_count_id = $count->id;
                    $countItem->inventory_item_id = $stock->inventory_item_id;
                    $countItem->expected_quantity = $stock->quantity;
                    $countItem->counted_quantity = 0;
                    $countItem->status = 'pending';
                    $countItem->save();
                }
            } elseif ($request->input('count_type') == 'partial') {
                // جرد جزئي - إضافة العناصر المحددة فقط
                $items = $request->input('items');

                foreach ($items as $itemId) {
                    $stock = InventoryStock::where('warehouse_id', $request->input('warehouse_id'))
                        ->where('inventory_item_id', $itemId)
                        ->where('company_id', $company->id)
                        ->first();

                    $countItem = new InventoryCountItem();
                    $countItem->inventory_count_id = $count->id;
                    $countItem->inventory_item_id = $itemId;
                    $countItem->expected_quantity = $stock ? $stock->quantity : 0;
                    $countItem->counted_quantity = 0;
                    $countItem->status = 'pending';
                    $countItem->save();
                }
            } elseif ($request->input('count_type') == 'cycle') {
                // جرد دوري - إضافة العناصر من فئة معينة
                $categoryId = $request->input('category_id');

                $items = InventoryItem::where('category_id', $categoryId)
                    ->where('company_id', $company->id)
                    ->get();

                foreach ($items as $item) {
                    $stock = InventoryStock::where('warehouse_id', $request->input('warehouse_id'))
                        ->where('inventory_item_id', $item->id)
                        ->where('company_id', $company->id)
                        ->first();

                    $countItem = new InventoryCountItem();
                    $countItem->inventory_count_id = $count->id;
                    $countItem->inventory_item_id = $item->id;
                    $countItem->expected_quantity = $stock ? $stock->quantity : 0;
                    $countItem->counted_quantity = 0;
                    $countItem->status = 'pending';
                    $countItem->save();
                }
            }

            DB::commit();

            return redirect()->route('inventory.counts.show', $count->id)
                ->with('success', 'تم إنشاء جلسة الجرد بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء جلسة الجرد: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * عرض تفاصيل جلسة الجرد
     */
    public function show($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $count = InventoryCount::with(['warehouse', 'createdBy', 'completedBy', 'approvedBy', 'items.item'])
            ->where('company_id', $company->id)
            ->findOrFail($id);

        return view('inventory.counts.show', compact('count'));
    }

    /**
     * عرض صفحة إدخال الكميات المحسوبة
     */
    public function count($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $count = InventoryCount::with(['warehouse', 'items.item'])
            ->where('company_id', $company->id)
            ->findOrFail($id);

        if ($count->status != 'draft' && $count->status != 'in_progress') {
            return redirect()->route('inventory.counts.show', $id)
                ->with('error', 'لا يمكن تعديل جلسة الجرد في الحالة الحالية');
        }

        return view('inventory.counts.count', compact('count'));
    }

    /**
     * تحديث كمية عنصر في جلسة الجرد
     */
    public function updateItem(Request $request, $id, $itemId)
    {
        $validator = Validator::make($request->all(), [
            'counted_quantity' => 'required|numeric|min:0',
            'notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ]);
        }

        $user = Auth::user();
        $company = $user->company;

        $count = InventoryCount::where('company_id', $company->id)->findOrFail($id);

        if ($count->status != 'draft' && $count->status != 'in_progress') {
            return response()->json([
                'success' => false,
                'message' => 'لا يمكن تعديل جلسة الجرد في الحالة الحالية',
            ]);
        }

        $countItem = InventoryCountItem::where('inventory_count_id', $id)
            ->where('inventory_item_id', $itemId)
            ->firstOrFail();

        $countItem->counted_quantity = $request->input('counted_quantity');
        $countItem->notes = $request->input('notes');
        $countItem->counted_by = $user->id;
        $countItem->counted_at = Carbon::now();

        // تحديد حالة العنصر بناءً على الفرق بين الكمية المتوقعة والكمية المحسوبة
        $difference = $countItem->counted_quantity - $countItem->expected_quantity;

        if ($difference == 0) {
            $countItem->status = 'match';
        } elseif ($difference > 0) {
            $countItem->status = 'surplus';
        } else {
            $countItem->status = 'shortage';
        }

        $countItem->save();

        // تحديث حالة جلسة الجرد إلى قيد التنفيذ إذا كانت مسودة
        if ($count->status == 'draft') {
            $count->status = 'in_progress';
            $count->save();
        }

        return response()->json([
            'success' => true,
            'message' => 'تم تحديث العنصر بنجاح',
            'item' => $countItem,
        ]);
    }

    /**
     * إكمال جلسة الجرد
     */
    public function complete(Request $request, $id)
    {
        $user = Auth::user();
        $company = $user->company;

        $count = InventoryCount::where('company_id', $company->id)->findOrFail($id);

        if ($count->status != 'in_progress') {
            return redirect()->back()
                ->with('error', 'لا يمكن إكمال جلسة الجرد في الحالة الحالية');
        }

        // التحقق من أن جميع العناصر تم عدها
        $pendingItems = InventoryCountItem::where('inventory_count_id', $id)
            ->where('status', 'pending')
            ->count();

        if ($pendingItems > 0) {
            return redirect()->back()
                ->with('error', 'يجب عد جميع العناصر قبل إكمال جلسة الجرد');
        }

        $count->status = 'completed';
        $count->completed_by = $user->id;
        $count->completed_at = Carbon::now();
        $count->save();

        return redirect()->route('inventory.counts.show', $id)
            ->with('success', 'تم إكمال جلسة الجرد بنجاح');
    }

    /**
     * الموافقة على جلسة الجرد وتعديل المخزون
     */
    public function approve(Request $request, $id)
    {
        $user = Auth::user();
        $company = $user->company;

        $count = InventoryCount::with('items.item')
            ->where('company_id', $company->id)
            ->findOrFail($id);

        if ($count->status != 'completed') {
            return redirect()->back()
                ->with('error', 'لا يمكن الموافقة على جلسة الجرد في الحالة الحالية');
        }

        DB::beginTransaction();

        try {
            // تعديل المخزون بناءً على نتائج الجرد
            foreach ($count->items as $countItem) {
                $stock = InventoryStock::where('warehouse_id', $count->warehouse_id)
                    ->where('inventory_item_id', $countItem->inventory_item_id)
                    ->where('company_id', $company->id)
                    ->first();

                if (!$stock) {
                    // إنشاء مخزون جديد إذا لم يكن موجوداً
                    $stock = new InventoryStock();
                    $stock->company_id = $company->id;
                    $stock->warehouse_id = $count->warehouse_id;
                    $stock->inventory_item_id = $countItem->inventory_item_id;
                    $stock->quantity = 0;
                }

                $difference = $countItem->counted_quantity - $stock->quantity;

                // تحديث كمية المخزون
                $stock->quantity = $countItem->counted_quantity;
                $stock->save();

                // إنشاء معاملة مخزون لتسجيل التعديل
                $transaction = new InventoryTransaction();
                $transaction->company_id = $company->id;
                $transaction->warehouse_id = $count->warehouse_id;
                $transaction->inventory_item_id = $countItem->inventory_item_id;
                $transaction->transaction_type = $difference >= 0 ? 'adjustment_add' : 'adjustment_subtract';
                $transaction->quantity = abs($difference);
                $transaction->reference_type = 'inventory_count';
                $transaction->reference_id = $count->id;
                $transaction->notes = 'تعديل بناءً على جلسة الجرد #' . $count->id;
                $transaction->created_by = $user->id;
                $transaction->save();
            }

            // تحديث حالة جلسة الجرد
            $count->status = 'approved';
            $count->approved_by = $user->id;
            $count->approved_at = Carbon::now();
            $count->save();

            DB::commit();

            return redirect()->route('inventory.counts.show', $id)
                ->with('success', 'تمت الموافقة على جلسة الجرد وتعديل المخزون بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء الموافقة على جلسة الجرد: ' . $e->getMessage());
        }
    }

    /**
     * إلغاء جلسة الجرد
     */
    public function cancel(Request $request, $id)
    {
        $user = Auth::user();
        $company = $user->company;

        $count = InventoryCount::where('company_id', $company->id)->findOrFail($id);

        if ($count->status == 'approved') {
            return redirect()->back()
                ->with('error', 'لا يمكن إلغاء جلسة الجرد بعد الموافقة عليها');
        }

        $count->status = 'cancelled';
        $count->cancelled_by = $user->id;
        $count->cancelled_at = Carbon::now();
        $count->save();

        return redirect()->route('inventory.counts.index')
            ->with('success', 'تم إلغاء جلسة الجرد بنجاح');
    }

    /**
     * طباعة تقرير جلسة الجرد
     */
    public function print($id)
    {
        $user = Auth::user();
        $company = $user->company;

        $count = InventoryCount::with(['warehouse', 'createdBy', 'completedBy', 'approvedBy', 'items.item'])
            ->where('company_id', $company->id)
            ->findOrFail($id);

        return view('inventory.counts.print', compact('count'));
    }
}
