<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InventoryAlert;
use App\Models\InventoryItem;
use App\Models\InventoryBatch;
use App\Models\User;
use App\Models\Company;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Mail\LowStockAlert;
use App\Mail\ExpiryAlert;
use Illuminate\Support\Facades\Validator;

class InventoryAlertController extends Controller
{
    /**
     * عرض إعدادات التنبيهات
     */
    public function settings()
    {
        $user = Auth::user();
        $company = $user->company;

        $alert = InventoryAlert::where('company_id', $company->id)->first();

        if (!$alert) {
            $alert = InventoryAlert::create([
                'company_id' => $company->id,
                'low_stock_enabled' => true,
                'low_stock_threshold' => 20,
                'expiry_enabled' => true,
                'expiry_threshold_days' => 30,
                'notification_emails' => [$company->email],
                'notification_users' => [$user->id],
            ]);
        }

        $users = User::where('company_id', $company->id)->get();

        return view('inventory.alerts.settings', compact('alert', 'users'));
    }

    /**
     * تحديث إعدادات التنبيهات
     */
    public function updateSettings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'low_stock_enabled' => 'boolean',
            'low_stock_threshold' => 'required_if:low_stock_enabled,1|numeric|min:1|max:100',
            'expiry_enabled' => 'boolean',
            'expiry_threshold_days' => 'required_if:expiry_enabled,1|numeric|min:1|max:365',
            'notification_emails' => 'nullable|array',
            'notification_emails.*' => 'email',
            'notification_users' => 'nullable|array',
            'notification_users.*' => 'exists:users,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = Auth::user();
        $company = $user->company;

        $alert = InventoryAlert::where('company_id', $company->id)->first();

        if (!$alert) {
            $alert = new InventoryAlert();
            $alert->company_id = $company->id;
        }

        $alert->low_stock_enabled = $request->has('low_stock_enabled');
        $alert->low_stock_threshold = $request->input('low_stock_threshold', 20);
        $alert->expiry_enabled = $request->has('expiry_enabled');
        $alert->expiry_threshold_days = $request->input('expiry_threshold_days', 30);
        $alert->notification_emails = $request->input('notification_emails', []);
        $alert->notification_users = $request->input('notification_users', []);

        $alert->save();

        return redirect()->route('inventory.alerts.settings')
            ->with('success', 'تم تحديث إعدادات التنبيهات بنجاح');
    }

    /**
     * عرض العناصر منخفضة المخزون
     */
    public function lowStock()
    {
        $user = Auth::user();
        $company = $user->company;

        $alert = InventoryAlert::where('company_id', $company->id)->first();

        if (!$alert) {
            $alert = InventoryAlert::create([
                'company_id' => $company->id,
                'low_stock_enabled' => true,
                'low_stock_threshold' => 20,
                'expiry_enabled' => true,
                'expiry_threshold_days' => 30,
                'notification_emails' => [$company->email],
                'notification_users' => [$user->id],
            ]);
        }

        $items = InventoryItem::with(['stocks.warehouse', 'category'])
            ->where('company_id', $company->id)
            ->where('is_active', true)
            ->get()
            ->filter(function ($item) use ($alert) {
                if ($item->min_stock_level <= 0) {
                    return false;
                }

                $percentage = ($item->available_stock / $item->min_stock_level) * 100;
                return $percentage <= $alert->low_stock_threshold;
            });

        return view('inventory.alerts.low-stock', compact('items', 'alert'));
    }

    /**
     * عرض العناصر التي ستنتهي قريباً
     */
    public function expiry()
    {
        $user = Auth::user();
        $company = $user->company;

        $alert = InventoryAlert::where('company_id', $company->id)->first();

        if (!$alert) {
            $alert = InventoryAlert::create([
                'company_id' => $company->id,
                'low_stock_enabled' => true,
                'low_stock_threshold' => 20,
                'expiry_enabled' => true,
                'expiry_threshold_days' => 30,
                'notification_emails' => [$company->email],
                'notification_users' => [$user->id],
            ]);
        }

        $expiryDate = now()->addDays($alert->expiry_threshold_days);

        $batches = InventoryBatch::with(['item', 'warehouse'])
            ->where('company_id', $company->id)
            ->where('expiry_date', '<=', $expiryDate)
            ->where('expiry_date', '>=', now())
            ->where('quantity', '>', 0)
            ->orderBy('expiry_date')
            ->get();

        return view('inventory.alerts.expiry', compact('batches', 'alert'));
    }

    /**
     * إرسال تنبيهات المخزون المنخفض يدوياً
     */
    public function sendLowStockAlerts()
    {
        $user = Auth::user();
        $company = $user->company;

        $alert = InventoryAlert::where('company_id', $company->id)->first();

        if (!$alert) {
            return redirect()->back()->with('error', 'لم يتم العثور على إعدادات التنبيهات');
        }

        $items = InventoryItem::with(['stocks.warehouse', 'category'])
            ->where('company_id', $company->id)
            ->where('is_active', true)
            ->get()
            ->filter(function ($item) use ($alert) {
                if ($item->min_stock_level <= 0) {
                    return false;
                }

                $percentage = ($item->available_stock / $item->min_stock_level) * 100;
                return $percentage <= $alert->low_stock_threshold;
            });

        if ($items->isEmpty()) {
            return redirect()->back()->with('info', 'لا توجد عناصر منخفضة المخزون حالياً');
        }

        // إرسال بريد إلكتروني
        if (!empty($alert->notification_emails)) {
            foreach ($alert->notification_emails as $email) {
                Mail::to($email)->send(new LowStockAlert($items, $alert));
            }
        }

        // إرسال إشعارات للمستخدمين
        if (!empty($alert->notification_users)) {
            $users = User::whereIn('id', $alert->notification_users)->get();
            foreach ($users as $notifyUser) {
                $notifyUser->notify(new \App\Notifications\LowStockAlert($items, $alert));
            }
        }

        return redirect()->back()->with('success', 'تم إرسال تنبيهات المخزون المنخفض بنجاح');
    }

    /**
     * إرسال تنبيهات المنتجات التي ستنتهي قريباً يدوياً
     */
    public function sendExpiryAlerts()
    {
        $user = Auth::user();
        $company = $user->company;

        $alert = InventoryAlert::where('company_id', $company->id)->first();

        if (!$alert) {
            return redirect()->back()->with('error', 'لم يتم العثور على إعدادات التنبيهات');
        }

        $expiryDate = now()->addDays($alert->expiry_threshold_days);

        $batches = InventoryBatch::with(['item', 'warehouse'])
            ->where('company_id', $company->id)
            ->where('expiry_date', '<=', $expiryDate)
            ->where('expiry_date', '>=', now())
            ->where('quantity', '>', 0)
            ->orderBy('expiry_date')
            ->get();

        if ($batches->isEmpty()) {
            return redirect()->back()->with('info', 'لا توجد منتجات ستنتهي قريباً');
        }

        // إرسال بريد إلكتروني
        if (!empty($alert->notification_emails)) {
            foreach ($alert->notification_emails as $email) {
                Mail::to($email)->send(new ExpiryAlert($batches, $alert));
            }
        }

        // إرسال إشعارات للمستخدمين
        if (!empty($alert->notification_users)) {
            $users = User::whereIn('id', $alert->notification_users)->get();
            foreach ($users as $notifyUser) {
                $notifyUser->notify(new \App\Notifications\ExpiryAlert($batches, $alert));
            }
        }

        return redirect()->back()->with('success', 'تم إرسال تنبيهات المنتجات التي ستنتهي قريباً بنجاح');
    }
}
