<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use App\Models\AutoPurchaseRule;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\Supplier;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class AutoPurchaseController extends Controller
{
    /**
     * عرض قائمة قواعد الشراء التلقائي.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_auto_purchase')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        $rules = AutoPurchaseRule::with(['item', 'supplier', 'warehouse'])
            ->where('company_id', Auth::user()->company_id)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('inventory.auto_purchase.index', compact('rules'));
    }

    /**
     * عرض نموذج إنشاء قاعدة شراء تلقائي جديدة.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('create_auto_purchase')) {
            return redirect()->route('inventory.auto-purchase.index')->with('error', 'ليس لديك صلاحية لإنشاء قاعدة شراء تلقائي');
        }

        $items = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $suppliers = Supplier::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $warehouses = Warehouse::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('inventory.auto_purchase.create', compact('items', 'suppliers', 'warehouses'));
    }

    /**
     * تخزين قاعدة شراء تلقائي جديدة.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('create_auto_purchase')) {
            return redirect()->route('inventory.auto-purchase.index')->with('error', 'ليس لديك صلاحية لإنشاء قاعدة شراء تلقائي');
        }

        // التحقق من البيانات
        $validated = $request->validate([
            'item_id' => 'required|exists:inventory_items,id',
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'min_quantity' => 'required|numeric|min:0',
            'order_quantity' => 'required|numeric|min:0.01',
            'frequency_days' => 'nullable|integer|min:0',
            'is_active' => 'sometimes|boolean',
        ]);

        // التحقق من عدم وجود قاعدة مشابهة
        $exists = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
            ->where('item_id', $request->item_id)
            ->where('warehouse_id', $request->warehouse_id)
            ->exists();

        if ($exists) {
            return redirect()->back()->withInput()->with('error', 'توجد قاعدة شراء تلقائي لهذا العنصر في هذا المستودع بالفعل');
        }

        // إنشاء القاعدة
        $rule = new AutoPurchaseRule();
        $rule->company_id = Auth::user()->company_id;
        $rule->item_id = $request->item_id;
        $rule->supplier_id = $request->supplier_id;
        $rule->warehouse_id = $request->warehouse_id;
        $rule->min_quantity = $request->min_quantity;
        $rule->order_quantity = $request->order_quantity;
        $rule->frequency_days = $request->frequency_days ?? 0;
        $rule->is_active = $request->has('is_active');
        $rule->created_by = Auth::id();
        $rule->next_check_date = now();
        $rule->save();

        return redirect()->route('inventory.auto-purchase.index')->with('success', 'تم إنشاء قاعدة الشراء التلقائي بنجاح');
    }

    /**
     * عرض تفاصيل قاعدة شراء تلقائي.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_auto_purchase')) {
            return redirect()->route('inventory.auto-purchase.index')->with('error', 'ليس لديك صلاحية لعرض تفاصيل قاعدة الشراء التلقائي');
        }

        $rule = AutoPurchaseRule::with(['item', 'supplier', 'warehouse', 'creator', 'updater', 'purchaseOrders'])
            ->where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // الحصول على مستوى المخزون الحالي
        $currentStock = InventoryStock::where('item_id', $rule->item_id)
            ->where('warehouse_id', $rule->warehouse_id)
            ->sum('quantity');

        // الحصول على طلبات الشراء المرتبطة
        $purchaseOrders = PurchaseOrder::where('auto_purchase_rule_id', $rule->id)
            ->orderBy('created_at', 'desc')
            ->paginate(5);

        return view('inventory.auto_purchase.show', compact('rule', 'currentStock', 'purchaseOrders'));
    }

    /**
     * عرض نموذج تعديل قاعدة شراء تلقائي.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('edit_auto_purchase')) {
            return redirect()->route('inventory.auto-purchase.index')->with('error', 'ليس لديك صلاحية لتعديل قاعدة الشراء التلقائي');
        }

        $rule = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $items = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $suppliers = Supplier::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $warehouses = Warehouse::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('inventory.auto_purchase.edit', compact('rule', 'items', 'suppliers', 'warehouses'));
    }

    /**
     * تحديث قاعدة شراء تلقائي.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('edit_auto_purchase')) {
            return redirect()->route('inventory.auto-purchase.index')->with('error', 'ليس لديك صلاحية لتعديل قاعدة الشراء التلقائي');
        }

        $rule = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // التحقق من البيانات
        $validated = $request->validate([
            'item_id' => 'required|exists:inventory_items,id',
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'min_quantity' => 'required|numeric|min:0',
            'order_quantity' => 'required|numeric|min:0.01',
            'frequency_days' => 'nullable|integer|min:0',
            'is_active' => 'sometimes|boolean',
        ]);

        // التحقق من عدم وجود قاعدة مشابهة (إذا تم تغيير العنصر أو المستودع)
        if ($rule->item_id != $request->item_id || $rule->warehouse_id != $request->warehouse_id) {
            $exists = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
                ->where('item_id', $request->item_id)
                ->where('warehouse_id', $request->warehouse_id)
                ->where('id', '!=', $id)
                ->exists();

            if ($exists) {
                return redirect()->back()->withInput()->with('error', 'توجد قاعدة شراء تلقائي لهذا العنصر في هذا المستودع بالفعل');
            }
        }

        // تحديث القاعدة
        $rule->item_id = $request->item_id;
        $rule->supplier_id = $request->supplier_id;
        $rule->warehouse_id = $request->warehouse_id;
        $rule->min_quantity = $request->min_quantity;
        $rule->order_quantity = $request->order_quantity;
        $rule->frequency_days = $request->frequency_days ?? 0;
        $rule->is_active = $request->has('is_active');
        $rule->updated_by = Auth::id();
        $rule->save();

        return redirect()->route('inventory.auto-purchase.index')->with('success', 'تم تحديث قاعدة الشراء التلقائي بنجاح');
    }

    /**
     * حذف قاعدة شراء تلقائي.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('delete_auto_purchase')) {
            return redirect()->route('inventory.auto-purchase.index')->with('error', 'ليس لديك صلاحية لحذف قاعدة الشراء التلقائي');
        }

        $rule = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        // التحقق من عدم وجود طلبات شراء مرتبطة
        $hasOrders = PurchaseOrder::where('auto_purchase_rule_id', $id)->exists();
        if ($hasOrders) {
            return redirect()->route('inventory.auto-purchase.index')->with('error', 'لا يمكن حذف القاعدة لوجود طلبات شراء مرتبطة بها');
        }

        $rule->delete();

        return redirect()->route('inventory.auto-purchase.index')->with('success', 'تم حذف قاعدة الشراء التلقائي بنجاح');
    }

    /**
     * تشغيل فحص الشراء التلقائي يدوياً.
     *
     * @return \Illuminate\Http\Response
     */
    public function runCheck()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('run_auto_purchase_check')) {
            return redirect()->route('inventory.auto-purchase.index')->with('error', 'ليس لديك صلاحية لتشغيل فحص الشراء التلقائي');
        }

        $count = $this->processAutoPurchaseRules();

        return redirect()->route('inventory.auto-purchase.index')->with('success', "تم تشغيل فحص الشراء التلقائي بنجاح. تم إنشاء {$count} طلب شراء.");
    }

    /**
     * معالجة قواعد الشراء التلقائي وإنشاء طلبات الشراء.
     *
     * @return int عدد طلبات الشراء التي تم إنشاؤها
     */
    private function processAutoPurchaseRules()
    {
        $count = 0;
        $companyId = Auth::user()->company_id;

        // الحصول على القواعد النشطة التي يجب فحصها
        $rules = AutoPurchaseRule::where('company_id', $companyId)
            ->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('next_check_date')
                    ->orWhere('next_check_date', '<=', now());
            })
            ->get();

        foreach ($rules as $rule) {
            // الحصول على مستوى المخزون الحالي
            $currentStock = InventoryStock::where('item_id', $rule->item_id)
                ->where('warehouse_id', $rule->warehouse_id)
                ->sum('quantity');

            // التحقق مما إذا كان المخزون أقل من الحد الأدنى
            if ($currentStock <= $rule->min_quantity) {
                // إنشاء طلب شراء
                DB::beginTransaction();
                try {
                    $purchaseOrder = new PurchaseOrder();
                    $purchaseOrder->company_id = $companyId;
                    $purchaseOrder->supplier_id = $rule->supplier_id;
                    $purchaseOrder->warehouse_id = $rule->warehouse_id;
                    $purchaseOrder->auto_purchase_rule_id = $rule->id;
                    $purchaseOrder->is_auto_generated = true;
                    $purchaseOrder->order_date = now();
                    $purchaseOrder->expected_delivery_date = now()->addDays(7);
                    $purchaseOrder->status = 'pending';
                    $purchaseOrder->created_by = Auth::id();
                    $purchaseOrder->notes = 'تم إنشاء هذا الطلب تلقائياً بواسطة نظام الشراء التلقائي';
                    $purchaseOrder->save();

                    // إضافة عنصر الطلب
                    $item = InventoryItem::find($rule->item_id);
                    $orderItem = new PurchaseOrderItem();
                    $orderItem->purchase_order_id = $purchaseOrder->id;
                    $orderItem->item_id = $rule->item_id;
                    $orderItem->quantity = $rule->order_quantity;
                    $orderItem->unit_price = $item->last_purchase_price ?? 0;
                    $orderItem->total_price = $orderItem->quantity * $orderItem->unit_price;
                    $orderItem->save();

                    // تحديث إجمالي الطلب
                    $purchaseOrder->total_amount = $orderItem->total_price;
                    $purchaseOrder->save();

                    // تحديث القاعدة
                    $rule->last_triggered_at = now();
                    $rule->next_check_date = now()->addDays($rule->frequency_days ?: 1);
                    $rule->save();

                    DB::commit();
                    $count++;
                } catch (\Exception $e) {
                    DB::rollBack();
                    \Log::error('خطأ في إنشاء طلب شراء تلقائي: ' . $e->getMessage());
                }
            } else {
                // تحديث تاريخ الفحص التالي فقط
                $rule->next_check_date = now()->addDays($rule->frequency_days ?: 1);
                $rule->save();
            }
        }

        return $count;
    }
}
