<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use App\Models\AutoPurchaseRule;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\InventoryTransaction;
use App\Models\PurchaseOrder;
use App\Models\Supplier;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AutoPurchaseAnalyticsController extends Controller
{
    /**
     * عرض صفحة التوصيات الذكية للشراء التلقائي
     *
     * @return \Illuminate\Http\Response
     */
    public function smartRecommendations()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_smart_recommendations')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // العناصر التي تحتاج إلى اهتمام عاجل (المخزون منخفض جداً)
        $urgentItems = $this->getUrgentItems();

        // العناصر ذات الاستهلاك الموسمي
        $seasonalItems = $this->getSeasonalItems();

        // توصيات تحسين التكلفة
        $costSavingRecommendations = $this->getCostSavingRecommendations();

        // إحصائيات لوحة المعلومات
        $dashboardStats = [
            'total_items' => InventoryItem::where('company_id', Auth::user()->company_id)->where('is_active', true)->count(),
            'low_stock_items' => count($urgentItems),
            'seasonal_items' => count($seasonalItems),
            'potential_savings' => array_sum(array_column($costSavingRecommendations, 'cost_savings')),
        ];

        // بيانات الرسوم البيانية
        $chartData = $this->getChartData();

        return view('inventory.auto_purchase.smart_recommendations', compact(
            'urgentItems',
            'seasonalItems',
            'costSavingRecommendations',
            'dashboardStats',
            'chartData'
        ));
    }

    /**
     * الحصول على العناصر التي تحتاج إلى اهتمام عاجل
     *
     * @return array
     */
    private function getUrgentItems()
    {
        $urgentItems = [];

        // الحصول على جميع العناصر النشطة
        $items = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->get();

        foreach ($items as $item) {
            // الحصول على المخزون الحالي لكل مستودع
            $warehouses = Warehouse::where('company_id', Auth::user()->company_id)
                ->where('is_active', true)
                ->get()
                ->map(function ($warehouse) use ($item) {
                    $warehouse->current_stock = InventoryStock::where('item_id', $item->id)
                        ->where('warehouse_id', $warehouse->id)
                        ->sum('quantity');

                    // حساب معدل الاستهلاك اليومي
                    $consumptionRate = $this->calculateDailyConsumptionRate($item->id, $warehouse->id);
                    $warehouse->consumption_rate = $consumptionRate;

                    // حساب الأيام المتبقية
                    $warehouse->days_remaining = $consumptionRate > 0 ? $warehouse->current_stock / $consumptionRate : null;

                    return $warehouse;
                })
                ->filter(function ($warehouse) {
                    // تصفية المستودعات التي بها مخزون ومعدل استهلاك
                    return $warehouse->current_stock > 0 && $warehouse->consumption_rate > 0;
                });

            // البحث عن المستودعات التي بها مخزون منخفض (أقل من 14 يوم)
            foreach ($warehouses as $warehouse) {
                if ($warehouse->days_remaining !== null && $warehouse->days_remaining < 14) {
                    // التحقق مما إذا كان هناك قاعدة شراء تلقائي
                    $rule = AutoPurchaseRule::where('item_id', $item->id)
                        ->where('warehouse_id', $warehouse->id)
                        ->where('company_id', Auth::user()->company_id)
                        ->first();

                    // الحصول على وقت التوريد
                    $leadTime = 7; // افتراضي: أسبوع

                    if ($rule && $rule->supplier_id) {
                        $supplierLeadTime = $this->getAverageLeadTime($item->id, $rule->supplier_id);
                        if ($supplierLeadTime !== null) {
                            $leadTime = $supplierLeadTime;
                        }
                    }

                    // الحصول على المورد المحتمل
                    $potentialSupplier = null;
                    if (!$rule) {
                        $potentialSupplier = InventoryTransaction::where('item_id', $item->id)
                            ->whereIn('transaction_type', ['purchase'])
                            ->where('created_at', '>=', now()->subDays(180))
                            ->join('purchase_orders', function ($join) {
                                $join->on('inventory_transactions.reference_id', '=', 'purchase_orders.id')
                                    ->where('inventory_transactions.reference_type', '=', 'purchase_order');
                            })
                            ->join('suppliers', 'purchase_orders.supplier_id', '=', 'suppliers.id')
                            ->select('suppliers.id', 'suppliers.name')
                            ->groupBy('suppliers.id', 'suppliers.name')
                            ->orderBy(DB::raw('COUNT(*)'), 'desc')
                            ->first();
                    }

                    // حساب الكميات الموصى بها
                    $safetyStock = $warehouse->consumption_rate * 3; // مخزون أمان لمدة 3 أيام
                    $recommendedMinQuantity = ($warehouse->consumption_rate * $leadTime) + $safetyStock;

                    // حساب الكمية الاقتصادية للطلب
                    $annualDemand = $warehouse->consumption_rate * 365;
                    $orderingCost = 50; // تكلفة إصدار أمر شراء (افتراضية)
                    $holdingCost = $item->purchase_price * 0.2; // تكلفة تخزين (20% من سعر الشراء)

                    $recommendedOrderQuantity = $holdingCost > 0 ?
                        sqrt((2 * $annualDemand * $orderingCost) / $holdingCost) :
                        $warehouse->consumption_rate * 30;

                    $urgentItems[] = [
                        'item' => $item,
                        'warehouse' => $warehouse,
                        'current_stock' => $warehouse->current_stock,
                        'consumption_rate' => $warehouse->consumption_rate,
                        'days_remaining' => $warehouse->days_remaining,
                        'lead_time' => $leadTime,
                        'has_rule' => $rule ? true : false,
                        'rule_id' => $rule ? $rule->id : null,
                        'potential_supplier' => $potentialSupplier,
                        'recommended_min_quantity' => $recommendedMinQuantity,
                        'recommended_order_quantity' => $recommendedOrderQuantity
                    ];
                }
            }
        }

        // ترتيب العناصر حسب الأيام المتبقية (تصاعدياً)
        usort($urgentItems, function ($a, $b) {
            return $a['days_remaining'] <=> $b['days_remaining'];
        });

        return $urgentItems;
    }

    /**
     * الحصول على العناصر ذات الاستهلاك الموسمي
     *
     * @return array
     */
    private function getSeasonalItems()
    {
        $seasonalItems = [];

        // الحصول على جميع العناصر النشطة
        $items = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->get();

        foreach ($items as $item) {
            // الحصول على المستودعات
            $warehouses = Warehouse::where('company_id', Auth::user()->company_id)
                ->where('is_active', true)
                ->get();

            foreach ($warehouses as $warehouse) {
                // حساب معدل الاستهلاك العادي
                $normalRate = $this->calculateDailyConsumptionRate($item->id, $warehouse->id);

                // حساب معدل الاستهلاك الموسمي
                $seasonalRate = $this->calculateSeasonalConsumptionRate($item->id, $warehouse->id);

                // التحقق مما إذا كان هناك زيادة موسمية كبيرة (أكثر من 30%)
                if ($normalRate > 0 && $seasonalRate > $normalRate * 1.3) {
                    // التحقق مما إذا كان هناك قاعدة شراء تلقائي
                    $rule = AutoPurchaseRule::where('item_id', $item->id)
                        ->where('warehouse_id', $warehouse->id)
                        ->where('company_id', Auth::user()->company_id)
                        ->first();

                    // حساب نسبة الزيادة
                    $increasePercentage = (($seasonalRate - $normalRate) / $normalRate) * 100;

                    // تحديد الفترة الموسمية
                    $currentMonth = now()->month;
                    $seasonalPeriod = '';

                    // تحديد الفترة الموسمية بناءً على الشهر الحالي
                    switch ($currentMonth) {
                        case 1:
                        case 2:
                        case 12:
                            $seasonalPeriod = 'فصل الشتاء (ديسمبر - فبراير)';
                            $seasonalEndDate = now()->month == 12 ?
                                now()->addMonths(2)->endOfMonth()->format('Y-m-d') :
                                now()->setMonth(2)->endOfMonth()->format('Y-m-d');
                            break;
                        case 3:
                        case 4:
                        case 5:
                            $seasonalPeriod = 'فصل الربيع (مارس - مايو)';
                            $seasonalEndDate = now()->setMonth(5)->endOfMonth()->format('Y-m-d');
                            break;
                        case 6:
                        case 7:
                        case 8:
                            $seasonalPeriod = 'فصل الصيف (يونيو - أغسطس)';
                            $seasonalEndDate = now()->setMonth(8)->endOfMonth()->format('Y-m-d');
                            break;
                        case 9:
                        case 10:
                        case 11:
                            $seasonalPeriod = 'فصل الخريف (سبتمبر - نوفمبر)';
                            $seasonalEndDate = now()->setMonth(11)->endOfMonth()->format('Y-m-d');
                            break;
                    }

                    // حساب الكميات الموصى بها
                    $leadTime = 7; // افتراضي: أسبوع

                    if ($rule && $rule->supplier_id) {
                        $supplierLeadTime = $this->getAverageLeadTime($item->id, $rule->supplier_id);
                        if ($supplierLeadTime !== null) {
                            $leadTime = $supplierLeadTime;
                        }
                    }

                    $safetyStock = $seasonalRate * 3; // مخزون أمان لمدة 3 أيام
                    $recommendedMinQuantity = ($seasonalRate * $leadTime) + $safetyStock;

                    // حساب الكمية الاقتصادية للطلب
                    $annualDemand = $seasonalRate * 365;
                    $orderingCost = 50; // تكلفة إصدار أمر شراء (افتراضية)
                    $holdingCost = $item->purchase_price * 0.2; // تكلفة تخزين (20% من سعر الشراء)

                    $recommendedOrderQuantity = $holdingCost > 0 ?
                        sqrt((2 * $annualDemand * $orderingCost) / $holdingCost) :
                        $seasonalRate * 30;

                    $seasonalItems[] = [
                        'item' => $item,
                        'warehouse' => $warehouse,
                        'normal_rate' => $normalRate,
                        'seasonal_rate' => $seasonalRate,
                        'increase_percentage' => $increasePercentage,
                        'seasonal_period' => $seasonalPeriod,
                        'seasonal_end_date' => $seasonalEndDate,
                        'has_rule' => $rule ? true : false,
                        'rule_id' => $rule ? $rule->id : null,
                        'current_min_quantity' => $rule ? $rule->min_quantity : 0,
                        'current_order_quantity' => $rule ? $rule->order_quantity : 0,
                        'recommended_min_quantity' => $recommendedMinQuantity,
                        'recommended_order_quantity' => $recommendedOrderQuantity
                    ];
                }
            }
        }

        // ترتيب العناصر حسب نسبة الزيادة (تنازلياً)
        usort($seasonalItems, function ($a, $b) {
            return $b['increase_percentage'] <=> $a['increase_percentage'];
        });

        return $seasonalItems;
    }

    /**
     * الحصول على توصيات تحسين التكلفة
     *
     * @return array
     */
    private function getCostSavingRecommendations()
    {
        $recommendations = [];

        // الحصول على جميع قواعد الشراء التلقائي النشطة
        $rules = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->with(['item', 'warehouse', 'supplier'])
            ->get();

        foreach ($rules as $rule) {
            // حساب معدل الاستهلاك اليومي
            $consumptionRate = $this->calculateDailyConsumptionRate($rule->item_id, $rule->warehouse_id);

            if ($consumptionRate > 0) {
                // حساب الكمية الاقتصادية للطلب
                $annualDemand = $consumptionRate * 365;
                $orderingCost = 50; // تكلفة إصدار أمر شراء (افتراضية)
                $holdingCost = $rule->item->purchase_price * 0.2; // تكلفة تخزين (20% من سعر الشراء)

                $eoq = $holdingCost > 0 ? sqrt((2 * $annualDemand * $orderingCost) / $holdingCost) : $consumptionRate * 30;

                // حساب التكلفة الحالية
                $currentAnnualCost = $this->calculateAnnualCost(
                    $rule->min_quantity,
                    $rule->order_quantity,
                    $annualDemand,
                    $orderingCost,
                    $holdingCost
                );

                // حساب التكلفة المقترحة
                $recommendedAnnualCost = $this->calculateAnnualCost(
                    $rule->min_quantity, // نفس الحد الأدنى
                    $eoq, // كمية الطلب الاقتصادية
                    $annualDemand,
                    $orderingCost,
                    $holdingCost
                );

                // حساب التوفير
                $costSavings = $currentAnnualCost - $recommendedAnnualCost;
                $savingsPercentage = $currentAnnualCost > 0 ? ($costSavings / $currentAnnualCost) * 100 : 0;

                // إضافة التوصية إذا كان التوفير كبيراً (أكثر من 10%)
                if ($savingsPercentage > 10) {
                    // تحديد نص التوصية
                    $recommendationText = '';

                    if (abs($rule->order_quantity - $eoq) / $rule->order_quantity > 0.2) {
                        if ($rule->order_quantity > $eoq) {
                            $recommendationText = 'تقليل كمية الطلب من ' . number_format($rule->order_quantity, 2) . ' إلى ' . number_format($eoq, 2);
                        } else {
                            $recommendationText = 'زيادة كمية الطلب من ' . number_format($rule->order_quantity, 2) . ' إلى ' . number_format($eoq, 2);
                        }
                    }

                    $recommendations[] = [
                        'rule_id' => $rule->id,
                        'item' => $rule->item,
                        'warehouse' => $rule->warehouse,
                        'current_order_quantity' => $rule->order_quantity,
                        'recommended_order_quantity' => $eoq,
                        'cost_savings' => $costSavings,
                        'savings_percentage' => $savingsPercentage,
                        'recommendation_text' => $recommendationText
                    ];
                }
            }
        }

        // ترتيب التوصيات حسب نسبة التوفير (تنازلياً)
        usort($recommendations, function ($a, $b) {
            return $b['savings_percentage'] <=> $a['savings_percentage'];
        });

        return $recommendations;
    }

    /**
     * الحصول على بيانات الرسوم البيانية
     *
     * @return array
     */
    private function getChartData()
    {
        // بيانات المخزون حسب الفئة
        $categoryData = DB::table('inventory_items')
            ->join('inventory_categories', 'inventory_items.category_id', '=', 'inventory_categories.id')
            ->join('inventory_stocks', 'inventory_items.id', '=', 'inventory_stocks.item_id')
            ->where('inventory_items.company_id', Auth::user()->company_id)
            ->where('inventory_items.is_active', true)
            ->select('inventory_categories.name', DB::raw('SUM(inventory_stocks.quantity) as total_stock'))
            ->groupBy('inventory_categories.name')
            ->orderBy('total_stock', 'desc')
            ->limit(10)
            ->get();

        $categoryLabels = $categoryData->pluck('name')->toArray();
        $categoryValues = $categoryData->pluck('total_stock')->toArray();

        // بيانات الاستهلاك الشهري
        $monthlyConsumption = DB::table('inventory_transactions')
            ->where('company_id', Auth::user()->company_id)
            ->whereIn('transaction_type', ['sale', 'maintenance_usage', 'write_off'])
            ->where('created_at', '>=', now()->subMonths(6))
            ->select(
                DB::raw('YEAR(created_at) as year'),
                DB::raw('MONTH(created_at) as month'),
                DB::raw('SUM(quantity) as total_consumption')
            )
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        $monthLabels = [];
        $consumptionData = [];

        foreach ($monthlyConsumption as $record) {
            $date = \Carbon\Carbon::createFromDate($record->year, $record->month, 1);
            $monthLabels[] = $date->format('M Y');
            $consumptionData[] = $record->total_consumption;
        }

        return [
            'category_labels' => $categoryLabels,
            'category_data' => $categoryValues,
            'month_labels' => $monthLabels,
            'consumption_data' => $consumptionData
        ];
    }

    /**
     * عرض صفحة تحليلات الشراء التلقائي
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_auto_purchase_analytics')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // إحصائيات عامة
        $stats = [
            'total_rules' => AutoPurchaseRule::where('company_id', Auth::user()->company_id)->count(),
            'active_rules' => AutoPurchaseRule::where('company_id', Auth::user()->company_id)
                ->where('is_active', true)
                ->count(),
            'auto_orders' => PurchaseOrder::where('company_id', Auth::user()->company_id)
                ->where('is_auto_generated', true)
                ->count(),
            'auto_orders_value' => PurchaseOrder::where('company_id', Auth::user()->company_id)
                ->where('is_auto_generated', true)
                ->sum('total_amount'),
        ];

        // أكثر العناصر طلباً بشكل تلقائي
        $topItems = DB::table('purchase_order_items')
            ->join('purchase_orders', 'purchase_orders.id', '=', 'purchase_order_items.purchase_order_id')
            ->join('inventory_items', 'inventory_items.id', '=', 'purchase_order_items.item_id')
            ->where('purchase_orders.company_id', Auth::user()->company_id)
            ->where('purchase_orders.is_auto_generated', true)
            ->select(
                'inventory_items.id',
                'inventory_items.name',
                'inventory_items.sku',
                DB::raw('COUNT(purchase_order_items.id) as order_count'),
                DB::raw('SUM(purchase_order_items.quantity) as total_quantity'),
                DB::raw('SUM(purchase_order_items.total_amount) as total_amount')
            )
            ->groupBy('inventory_items.id', 'inventory_items.name', 'inventory_items.sku')
            ->orderBy('order_count', 'desc')
            ->limit(10)
            ->get();

        // أكثر الموردين استخداماً في الشراء التلقائي
        $topSuppliers = DB::table('purchase_orders')
            ->join('suppliers', 'suppliers.id', '=', 'purchase_orders.supplier_id')
            ->where('purchase_orders.company_id', Auth::user()->company_id)
            ->where('purchase_orders.is_auto_generated', true)
            ->select(
                'suppliers.id',
                'suppliers.name',
                DB::raw('COUNT(purchase_orders.id) as order_count'),
                DB::raw('SUM(purchase_orders.total_amount) as total_amount')
            )
            ->groupBy('suppliers.id', 'suppliers.name')
            ->orderBy('order_count', 'desc')
            ->limit(10)
            ->get();

        // توزيع حالات طلبات الشراء التلقائية
        $orderStatusDistribution = DB::table('purchase_orders')
            ->where('company_id', Auth::user()->company_id)
            ->where('is_auto_generated', true)
            ->select(
                'status',
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('status')
            ->get();

        // العناصر التي تحتاج إلى مراجعة قواعد الشراء التلقائي
        $itemsNeedingReview = $this->getItemsNeedingReview();

        return view('inventory.auto_purchase.analytics', compact(
            'stats',
            'topItems',
            'topSuppliers',
            'orderStatusDistribution',
            'itemsNeedingReview'
        ));
    }

    /**
     * عرض تقرير توقعات الشراء التلقائي
     *
     * @return \Illuminate\Http\Response
     */
    public function forecast()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_auto_purchase_analytics')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // الحصول على جميع القواعد النشطة
        $rules = AutoPurchaseRule::with(['item', 'warehouse'])
            ->where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->get();

        $forecastItems = [];

        foreach ($rules as $rule) {
            // الحصول على مستوى المخزون الحالي
            $currentStock = InventoryStock::where('item_id', $rule->item_id)
                ->where('warehouse_id', $rule->warehouse_id)
                ->sum('quantity');

            // حساب معدل الاستهلاك اليومي
            $consumptionRate = $this->calculateDailyConsumptionRate($rule->item_id, $rule->warehouse_id);

            // حساب الأيام المتبقية قبل الوصول إلى الحد الأدنى
            $daysRemaining = $consumptionRate > 0 ? floor(($currentStock - $rule->min_quantity) / $consumptionRate) : null;

            // تحديد تاريخ الطلب المتوقع
            $expectedOrderDate = $daysRemaining !== null ? now()->addDays($daysRemaining) : null;

            $forecastItems[] = [
                'rule' => $rule,
                'current_stock' => $currentStock,
                'consumption_rate' => $consumptionRate,
                'days_remaining' => $daysRemaining,
                'expected_order_date' => $expectedOrderDate,
                'needs_attention' => $daysRemaining !== null && $daysRemaining <= 7,
            ];
        }

        // ترتيب العناصر حسب الأيام المتبقية (الأقل أولاً)
        usort($forecastItems, function ($a, $b) {
            if ($a['days_remaining'] === null) return 1;
            if ($b['days_remaining'] === null) return -1;
            return $a['days_remaining'] <=> $b['days_remaining'];
        });

        return view('inventory.auto_purchase.forecast', compact('forecastItems'));
    }

    /**
     * عرض تقرير أداء قواعد الشراء التلقائي
     *
     * @return \Illuminate\Http\Response
     */
    public function performance()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_auto_purchase_analytics')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // الحصول على جميع القواعد
        $rules = AutoPurchaseRule::with(['item', 'warehouse', 'purchaseOrders'])
            ->where('company_id', Auth::user()->company_id)
            ->get();

        $performanceData = [];

        foreach ($rules as $rule) {
            // عدد مرات تشغيل القاعدة
            $triggerCount = $rule->purchaseOrders->count();

            // متوسط الوقت بين عمليات الشراء
            $avgTimeBetweenOrders = null;
            if ($triggerCount > 1) {
                $orderDates = $rule->purchaseOrders->pluck('created_at')->sort();
                $totalDays = 0;
                for ($i = 1; $i < count($orderDates); $i++) {
                    $totalDays += $orderDates[$i]->diffInDays($orderDates[$i - 1]);
                }
                $avgTimeBetweenOrders = $totalDays / (count($orderDates) - 1);
            }

            // عدد مرات نفاد المخزون
            $stockoutCount = InventoryTransaction::where('item_id', $rule->item_id)
                ->where('warehouse_id', $rule->warehouse_id)
                ->where('notes', 'like', '%نفاد المخزون%')
                ->count();

            // تقييم أداء القاعدة
            $performance = $this->evaluateRulePerformance($rule, $triggerCount, $avgTimeBetweenOrders, $stockoutCount);

            $performanceData[] = [
                'rule' => $rule,
                'trigger_count' => $triggerCount,
                'avg_time_between_orders' => $avgTimeBetweenOrders,
                'stockout_count' => $stockoutCount,
                'performance' => $performance,
            ];
        }

        return view('inventory.auto_purchase.performance', compact('performanceData'));
    }

    /**
     * عرض صفحة توصيات تحسين قواعد الشراء التلقائي
     *
     * @return \Illuminate\Http\Response
     */
    public function recommendations()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_auto_purchase_analytics')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // الحصول على جميع القواعد
        $rules = AutoPurchaseRule::with(['item', 'warehouse', 'supplier'])
            ->where('company_id', Auth::user()->company_id)
            ->get();

        $recommendations = [];

        foreach ($rules as $rule) {
            // الحصول على مستوى المخزون الحالي
            $currentStock = InventoryStock::where('item_id', $rule->item_id)
                ->where('warehouse_id', $rule->warehouse_id)
                ->sum('quantity');

            // حساب معدل الاستهلاك اليومي
            $consumptionRate = $this->calculateDailyConsumptionRate($rule->item_id, $rule->warehouse_id);

            // حساب معدل الاستهلاك الموسمي إذا كان متاحاً
            $seasonalConsumptionRate = $this->calculateSeasonalConsumptionRate($rule->item_id, $rule->warehouse_id);

            // استخدام المعدل الموسمي إذا كان متاحاً وأعلى من المعدل العادي
            $effectiveRate = $seasonalConsumptionRate > $consumptionRate ? $seasonalConsumptionRate : $consumptionRate;

            // توصيات لتحسين الحد الأدنى للمخزون (مع مراعاة وقت التوريد)
            $leadTime = $this->getAverageLeadTime($rule->item_id, $rule->supplier_id) ?? 7; // افتراضي: أسبوع
            $safetyStock = $effectiveRate * 3; // مخزون أمان لمدة 3 أيام
            $recommendedMinQuantity = ($effectiveRate * $leadTime) + $safetyStock;

            // توصيات لتحسين كمية الطلب (باستخدام نموذج الكمية الاقتصادية للطلب EOQ)
            $annualDemand = $effectiveRate * 365;
            $orderingCost = 50; // تكلفة إصدار أمر شراء (افتراضية)
            $holdingCost = $rule->item->purchase_price * 0.2; // تكلفة تخزين (20% من سعر الشراء)

            // حساب الكمية الاقتصادية للطلب
            $eoq = $holdingCost > 0 ? sqrt((2 * $annualDemand * $orderingCost) / $holdingCost) : $effectiveRate * 30;

            // تعديل الكمية الاقتصادية بناءً على الحد الأدنى للكمية من المورد (إن وجد)
            $supplierMinOrderQty = $rule->supplier->min_order_quantity ?? 0;
            $recommendedOrderQuantity = max($eoq, $supplierMinOrderQty);

            // تحديد ما إذا كانت هناك حاجة لتعديل القاعدة
            $minQuantityDiff = $rule->min_quantity > 0 ?
                abs($rule->min_quantity - $recommendedMinQuantity) / $rule->min_quantity : 1;

            $orderQuantityDiff = $rule->order_quantity > 0 ?
                abs($rule->order_quantity - $recommendedOrderQuantity) / $rule->order_quantity : 1;

            $needsAdjustment = $minQuantityDiff > 0.2 || $orderQuantityDiff > 0.2;

            if ($needsAdjustment) {
                // حساب التأثير المالي للتغيير
                $currentAnnualCost = $this->calculateAnnualCost(
                    $rule->min_quantity,
                    $rule->order_quantity,
                    $annualDemand,
                    $orderingCost,
                    $holdingCost
                );

                $recommendedAnnualCost = $this->calculateAnnualCost(
                    $recommendedMinQuantity,
                    $recommendedOrderQuantity,
                    $annualDemand,
                    $orderingCost,
                    $holdingCost
                );

                $costSavings = $currentAnnualCost - $recommendedAnnualCost;
                $savingsPercentage = $currentAnnualCost > 0 ? ($costSavings / $currentAnnualCost) * 100 : 0;

                // تحديد أولوية التوصية
                $priority = 'متوسطة';
                if ($savingsPercentage > 15 || $minQuantityDiff > 0.5) {
                    $priority = 'عالية';
                } elseif ($savingsPercentage < 5 && $minQuantityDiff < 0.3) {
                    $priority = 'منخفضة';
                }

                $recommendations[] = [
                    'rule' => $rule,
                    'current_stock' => $currentStock,
                    'consumption_rate' => $effectiveRate,
                    'is_seasonal' => $seasonalConsumptionRate > $consumptionRate,
                    'lead_time' => $leadTime,
                    'current_min_quantity' => $rule->min_quantity,
                    'recommended_min_quantity' => $recommendedMinQuantity,
                    'current_order_quantity' => $rule->order_quantity,
                    'recommended_order_quantity' => $recommendedOrderQuantity,
                    'cost_savings' => $costSavings,
                    'savings_percentage' => $savingsPercentage,
                    'priority' => $priority,
                    'reasons' => $this->getAdjustmentReasons($rule, $recommendedMinQuantity, $recommendedOrderQuantity, $leadTime, $effectiveRate)
                ];
            }
        }

        // ترتيب التوصيات حسب الأولوية والتوفير المالي
        usort($recommendations, function ($a, $b) {
            // ترتيب حسب الأولوية أولاً
            $priorityOrder = ['عالية' => 1, 'متوسطة' => 2, 'منخفضة' => 3];
            $priorityCompare = $priorityOrder[$a['priority']] <=> $priorityOrder[$b['priority']];

            if ($priorityCompare !== 0) {
                return $priorityCompare;
            }

            // ثم حسب نسبة التوفير (تنازلياً)
            return $b['savings_percentage'] <=> $a['savings_percentage'];
        });

        // العناصر التي ليس لها قواعد شراء تلقائي ولكن يجب أن يكون لها
        $itemsNeedingRules = $this->getItemsNeedingRules();

        // الحصول على إحصائيات التوصيات
        $recommendationStats = [
            'total_recommendations' => count($recommendations),
            'high_priority' => count(array_filter($recommendations, function ($r) {
                return $r['priority'] === 'عالية';
            })),
            'medium_priority' => count(array_filter($recommendations, function ($r) {
                return $r['priority'] === 'متوسطة';
            })),
            'low_priority' => count(array_filter($recommendations, function ($r) {
                return $r['priority'] === 'منخفضة';
            })),
            'total_potential_savings' => array_sum(array_column($recommendations, 'cost_savings')),
            'new_rules_suggested' => count($itemsNeedingRules)
        ];

        return view('inventory.auto_purchase.recommendations', compact(
            'recommendations',
            'itemsNeedingRules',
            'recommendationStats'
        ));
    }

    /**
     * حساب التكلفة السنوية للمخزون
     *
     * @param float $minQuantity
     * @param float $orderQuantity
     * @param float $annualDemand
     * @param float $orderingCost
     * @param float $holdingCost
     * @return float
     */
    private function calculateAnnualCost($minQuantity, $orderQuantity, $annualDemand, $orderingCost, $holdingCost)
    {
        // عدد الطلبات سنوياً
        $ordersPerYear = $annualDemand / $orderQuantity;

        // تكلفة الطلبات السنوية
        $annualOrderingCost = $ordersPerYear * $orderingCost;

        // متوسط المخزون
        $averageInventory = ($minQuantity + $orderQuantity) / 2;

        // تكلفة التخزين السنوية
        $annualHoldingCost = $averageInventory * $holdingCost;

        // إجمالي التكلفة السنوية
        return $annualOrderingCost + $annualHoldingCost;
    }

    /**
     * الحصول على أسباب تعديل قاعدة الشراء التلقائي
     *
     * @param AutoPurchaseRule $rule
     * @param float $recommendedMinQuantity
     * @param float $recommendedOrderQuantity
     * @param int $leadTime
     * @param float $consumptionRate
     * @return array
     */
    private function getAdjustmentReasons($rule, $recommendedMinQuantity, $recommendedOrderQuantity, $leadTime, $consumptionRate)
    {
        $reasons = [];

        // أسباب تعديل الحد الأدنى
        if ($rule->min_quantity < $recommendedMinQuantity * 0.8) {
            $reasons[] = "الحد الأدنى الحالي منخفض جداً مما قد يؤدي إلى نفاد المخزون خلال فترة التوريد ({$leadTime} يوم)";
        } elseif ($rule->min_quantity > $recommendedMinQuantity * 1.2) {
            $reasons[] = "الحد الأدنى الحالي مرتفع جداً مما يؤدي إلى تجميد رأس المال في المخزون";
        }

        // أسباب تعديل كمية الطلب
        if ($rule->order_quantity < $recommendedOrderQuantity * 0.8) {
            $daysSupply = $rule->order_quantity / $consumptionRate;
            $reasons[] = "كمية الطلب الحالية صغيرة جداً وتكفي فقط لـ {$daysSupply} يوم، مما يزيد من تكاليف الطلب";
        } elseif ($rule->order_quantity > $recommendedOrderQuantity * 1.2) {
            $reasons[] = "كمية الطلب الحالية كبيرة جداً مما يزيد من تكاليف التخزين وخطر التقادم";
        }

        // أسباب أخرى
        if ($consumptionRate > 0 && $rule->frequency_days > 7) {
            $reasons[] = "تكرار الفحص ({$rule->frequency_days} يوم) طويل جداً مقارنة بمعدل الاستهلاك";
        }

        return $reasons;
    }

    /**
     * تطبيق التوصيات على قاعدة شراء تلقائي
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function applyRecommendation(Request $request, $id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('edit_auto_purchase')) {
            return redirect()->route('inventory.auto-purchase.recommendations')->with('error', 'ليس لديك صلاحية لتعديل قاعدة الشراء التلقائي');
        }

        $rule = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $validated = $request->validate([
            'min_quantity' => 'required|numeric|min:0',
            'order_quantity' => 'required|numeric|min:0.01',
        ]);

        // تحديث القاعدة
        $rule->min_quantity = $validated['min_quantity'];
        $rule->order_quantity = $validated['order_quantity'];
        $rule->updated_by = Auth::id();
        $rule->save();

        return redirect()->route('inventory.auto-purchase.recommendations')
            ->with('success', 'تم تطبيق التوصيات على قاعدة الشراء التلقائي بنجاح');
    }

    /**
     * إنشاء قاعدة شراء تلقائي جديدة بناءً على التوصيات
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function createRecommendedRule(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('create_recommended_rules')) {
            return redirect()->route('inventory.auto-purchase.smart-recommendations')->with('error', 'ليس لديك صلاحية لإنشاء قواعد شراء تلقائي من التوصيات');
        }

        $validated = $request->validate([
            'item_id' => 'required|exists:inventory_items,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'supplier_id' => 'required|exists:suppliers,id',
            'min_quantity' => 'required|numeric|min:0',
            'order_quantity' => 'required|numeric|min:0.01',
            'frequency_days' => 'required|integer|min:1',
            'create_purchase_order' => 'nullable|boolean',
        ]);

        // التحقق من أن العناصر تنتمي للشركة الحالية
        $itemBelongsToCompany = \App\Models\InventoryItem::where('id', $validated['item_id'])
            ->where('company_id', Auth::user()->company_id)
            ->exists();

        $warehouseBelongsToCompany = \App\Models\Warehouse::where('id', $validated['warehouse_id'])
            ->where('company_id', Auth::user()->company_id)
            ->exists();

        $supplierBelongsToCompany = \App\Models\Supplier::where('id', $validated['supplier_id'])
            ->where('company_id', Auth::user()->company_id)
            ->exists();

        if (!$itemBelongsToCompany || !$warehouseBelongsToCompany || !$supplierBelongsToCompany) {
            return redirect()->route('inventory.auto-purchase.smart-recommendations')
                ->with('error', 'البيانات المدخلة غير صحيحة');
        }

        // إنشاء قاعدة شراء تلقائي جديدة
        $rule = new \App\Models\AutoPurchaseRule();
        $rule->company_id = Auth::user()->company_id;
        $rule->item_id = $validated['item_id'];
        $rule->warehouse_id = $validated['warehouse_id'];
        $rule->supplier_id = $validated['supplier_id'];
        $rule->min_quantity = $validated['min_quantity'];
        $rule->order_quantity = $validated['order_quantity'];
        $rule->frequency_days = $validated['frequency_days'];
        $rule->is_active = true;
        $rule->created_by = Auth::id();
        $rule->updated_by = Auth::id();
        $rule->save();

        // إنشاء طلب شراء فوري إذا تم طلب ذلك
        if ($request->has('create_purchase_order') && $request->create_purchase_order) {
            // استدعاء خدمة إنشاء طلب الشراء
            $purchaseOrderService = new \App\Services\AutoPurchaseService();
            $purchaseOrderService->createPurchaseOrderForRule($rule);

            return redirect()->route('inventory.auto-purchase.smart-recommendations')
                ->with('success', 'تم إنشاء قاعدة الشراء التلقائي وطلب الشراء بنجاح');
        }

        return redirect()->route('inventory.auto-purchase.smart-recommendations')
            ->with('success', 'تم إنشاء قاعدة الشراء التلقائي بنجاح');
    }

    /**
     * تطبيق تعديل موسمي على قاعدة شراء تلقائي
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function adjustSeasonal(Request $request, $id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('apply_seasonal_adjustments')) {
            return redirect()->route('inventory.auto-purchase.smart-recommendations')->with('error', 'ليس لديك صلاحية لتطبيق التعديلات الموسمية');
        }

        $rule = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $validated = $request->validate([
            'seasonal_min_quantity' => 'required|numeric|min:0',
            'seasonal_order_quantity' => 'required|numeric|min:0.01',
            'seasonal_end_date' => 'required|date',
        ]);

        // تحديث القاعدة بالقيم الموسمية
        $rule->min_quantity = $validated['seasonal_min_quantity'];
        $rule->order_quantity = $validated['seasonal_order_quantity'];

        // حفظ معلومات التعديل الموسمي
        $rule->seasonal_adjustment = true;
        $rule->seasonal_end_date = $validated['seasonal_end_date'];
        $rule->original_min_quantity = $rule->min_quantity;
        $rule->original_order_quantity = $rule->order_quantity;

        $rule->updated_by = Auth::id();
        $rule->save();

        return redirect()->route('inventory.auto-purchase.smart-recommendations')
            ->with('success', 'تم تطبيق التعديل الموسمي بنجاح');
    }

    /**
     * إنشاء قاعدة شراء تلقائي جديدة بناءً على التوصيات (النسخة الجديدة)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function createRecommendedRuleNew(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('create_auto_purchase')) {
            return redirect()->route('inventory.auto-purchase.recommendations')->with('error', 'ليس لديك صلاحية لإنشاء قاعدة شراء تلقائي');
        }

        $validated = $request->validate([
            'item_id' => 'required|exists:inventory_items,id',
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'min_quantity' => 'required|numeric|min:0',
            'order_quantity' => 'required|numeric|min:0.01',
        ]);

        // التحقق من عدم وجود قاعدة مشابهة
        $exists = AutoPurchaseRule::where('company_id', Auth::user()->company_id)
            ->where('item_id', $request->item_id)
            ->where('warehouse_id', $request->warehouse_id)
            ->exists();

        if ($exists) {
            return redirect()->back()->withInput()->with('error', 'توجد قاعدة شراء تلقائي لهذا العنصر في هذا المستودع بالفعل');
        }

        // إنشاء القاعدة
        $rule = new AutoPurchaseRule();
        $rule->company_id = Auth::user()->company_id;
        $rule->item_id = $validated['item_id'];
        $rule->supplier_id = $validated['supplier_id'];
        $rule->warehouse_id = $validated['warehouse_id'];
        $rule->min_quantity = $validated['min_quantity'];
        $rule->order_quantity = $validated['order_quantity'];
        $rule->frequency_days = 7; // افتراضي: فحص أسبوعي
        $rule->is_active = true;
        $rule->created_by = Auth::id();
        $rule->next_check_date = now();
        $rule->save();

        return redirect()->route('inventory.auto-purchase.recommendations')
            ->with('success', 'تم إنشاء قاعدة الشراء التلقائي الموصى بها بنجاح');
    }

    /**
     * حساب معدل الاستهلاك اليومي لعنصر في مستودع
     *
     * @param int $itemId
     * @param int $warehouseId
     * @return float
     */
    private function calculateDailyConsumptionRate($itemId, $warehouseId)
    {
        // الحصول على معاملات الاستهلاك (البيع، الاستخدام في الصيانة، إلخ) خلال آخر 90 يوم
        $startDate = now()->subDays(90);

        $consumptionTransactions = InventoryTransaction::where('item_id', $itemId)
            ->where('warehouse_id', $warehouseId)
            ->whereIn('transaction_type', ['sale', 'maintenance_usage', 'write_off'])
            ->where('created_at', '>=', $startDate)
            ->get();

        if ($consumptionTransactions->isEmpty()) {
            return 0;
        }

        // حساب إجمالي الاستهلاك
        $totalConsumption = $consumptionTransactions->sum('quantity');

        // حساب عدد الأيام الفعلية التي حدث فيها استهلاك
        $uniqueDays = $consumptionTransactions->pluck('created_at')->map(function ($date) {
            return $date->format('Y-m-d');
        })->unique()->count();

        // استخدام عدد الأيام الفعلية إذا كان أكبر من 0، وإلا استخدام 90 يوم
        $days = $uniqueDays > 0 ? $uniqueDays : 90;

        // حساب المعدل اليومي
        return $totalConsumption / $days;
    }

    /**
     * تقييم أداء قاعدة الشراء التلقائي
     *
     * @param AutoPurchaseRule $rule
     * @param int $triggerCount
     * @param float|null $avgTimeBetweenOrders
     * @param int $stockoutCount
     * @return array
     */
    private function evaluateRulePerformance($rule, $triggerCount, $avgTimeBetweenOrders, $stockoutCount)
    {
        // تقييم أولي
        $score = 100;
        $issues = [];
        $recommendations = [];

        // خصم نقاط لكل مرة نفاد مخزون
        if ($stockoutCount > 0) {
            $score -= $stockoutCount * 20;
            $issues[] = "حدث نفاد للمخزون {$stockoutCount} مرة";
            $recommendations[] = "زيادة الحد الأدنى للمخزون";
        }

        // تقييم تكرار الطلبات
        if ($avgTimeBetweenOrders !== null) {
            if ($avgTimeBetweenOrders < 7) {
                $score -= 15;
                $issues[] = "تكرار الطلبات بشكل متقارب جداً (كل {$avgTimeBetweenOrders} يوم)";
                $recommendations[] = "زيادة كمية الطلب لتقليل تكرار الطلبات";
            } elseif ($avgTimeBetweenOrders > 90) {
                $score -= 10;
                $issues[] = "فترات طويلة جداً بين الطلبات (كل {$avgTimeBetweenOrders} يوم)";
                $recommendations[] = "تقليل كمية الطلب لتحسين تدوير المخزون";
            }
        }

        // تقييم عدد مرات التشغيل
        if ($rule->created_at->diffInDays(now()) > 30 && $triggerCount == 0) {
            $score -= 25;
            $issues[] = "لم يتم تشغيل القاعدة أبداً رغم مرور أكثر من شهر على إنشائها";
            $recommendations[] = "مراجعة الحد الأدنى للمخزون، قد يكون مرتفعاً جداً";
        }

        // تحديد مستوى الأداء
        $level = 'ممتاز';
        if ($score < 90) $level = 'جيد جداً';
        if ($score < 80) $level = 'جيد';
        if ($score < 70) $level = 'متوسط';
        if ($score < 60) $level = 'ضعيف';
        if ($score < 40) $level = 'سيء';

        return [
            'score' => max(0, $score),
            'level' => $level,
            'issues' => $issues,
            'recommendations' => $recommendations,
        ];
    }

    /**
     * الحصول على العناصر التي تحتاج إلى مراجعة قواعد الشراء التلقائي
     *
     * @return array
     */
    private function getItemsNeedingReview()
    {
        $itemsNeedingReview = [];

        // الحصول على القواعد النشطة
        $rules = AutoPurchaseRule::with(['item', 'warehouse'])
            ->where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->get();

        foreach ($rules as $rule) {
            $issues = [];

            // الحصول على مستوى المخزون الحالي
            $currentStock = InventoryStock::where('item_id', $rule->item_id)
                ->where('warehouse_id', $rule->warehouse_id)
                ->sum('quantity');

            // حساب معدل الاستهلاك
            $consumptionRate = $this->calculateDailyConsumptionRate($rule->item_id, $rule->warehouse_id);

            // التحقق من مشاكل محتملة

            // 1. الحد الأدنى منخفض جداً مقارنة بمعدل الاستهلاك
            if ($consumptionRate > 0 && $rule->min_quantity < ($consumptionRate * 3)) {
                $issues[] = "الحد الأدنى منخفض جداً، قد يؤدي إلى نفاد المخزون قبل وصول الطلب";
            }

            // 2. الحد الأدنى مرتفع جداً مقارنة بمعدل الاستهلاك
            if ($consumptionRate > 0 && $rule->min_quantity > ($consumptionRate * 30)) {
                $issues[] = "الحد الأدنى مرتفع جداً، قد يؤدي إلى تراكم المخزون";
            }

            // 3. كمية الطلب منخفضة جداً مقارنة بمعدل الاستهلاك
            if ($consumptionRate > 0 && $rule->order_quantity < ($consumptionRate * 7)) {
                $issues[] = "كمية الطلب منخفضة جداً، قد تؤدي إلى تكرار الطلبات بشكل متقارب";
            }

            // 4. كمية الطلب مرتفعة جداً مقارنة بمعدل الاستهلاك
            if ($consumptionRate > 0 && $rule->order_quantity > ($consumptionRate * 90)) {
                $issues[] = "كمية الطلب مرتفعة جداً، قد تؤدي إلى تجميد رأس المال";
            }

            // إضافة العنصر إلى القائمة إذا كانت هناك مشاكل
            if (!empty($issues)) {
                $itemsNeedingReview[] = [
                    'rule' => $rule,
                    'current_stock' => $currentStock,
                    'consumption_rate' => $consumptionRate,
                    'issues' => $issues,
                ];
            }
        }

        return $itemsNeedingReview;
    }

    /**
     * حساب معدل الاستهلاك الموسمي لعنصر في مستودع
     *
     * @param int $itemId
     * @param int $warehouseId
     * @return float
     */
    private function calculateSeasonalConsumptionRate($itemId, $warehouseId)
    {
        // الحصول على الشهر الحالي
        $currentMonth = now()->month;

        // تحديد الفترة الموسمية (الشهر الحالي والشهرين السابقين من العام الماضي)
        $startDate = now()->subYear()->setMonth($currentMonth)->startOfMonth()->subMonths(1);
        $endDate = now()->subYear()->setMonth($currentMonth)->endOfMonth()->addMonths(1);

        // الحصول على معاملات الاستهلاك خلال الفترة الموسمية من العام الماضي
        $seasonalTransactions = InventoryTransaction::where('item_id', $itemId)
            ->where('warehouse_id', $warehouseId)
            ->whereIn('transaction_type', ['sale', 'maintenance_usage', 'write_off'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->get();

        if ($seasonalTransactions->isEmpty()) {
            return 0;
        }

        // حساب إجمالي الاستهلاك الموسمي
        $totalConsumption = $seasonalTransactions->sum('quantity');

        // حساب عدد الأيام في الفترة الموسمية
        $days = $startDate->diffInDays($endDate) + 1;

        // حساب المعدل اليومي الموسمي
        return $totalConsumption / $days;
    }

    /**
     * الحصول على متوسط وقت التوريد لعنصر من مورد
     *
     * @param int $itemId
     * @param int $supplierId
     * @return float|null
     */
    private function getAverageLeadTime($itemId, $supplierId)
    {
        // الحصول على طلبات الشراء المكتملة للعنصر من المورد
        $purchaseOrders = PurchaseOrder::where('supplier_id', $supplierId)
            ->where('status', 'completed')
            ->where('created_at', '>=', now()->subYear())
            ->whereHas('items', function ($query) use ($itemId) {
                $query->where('item_id', $itemId);
            })
            ->get();

        if ($purchaseOrders->isEmpty()) {
            return null;
        }

        // حساب متوسط الفترة بين تاريخ الطلب وتاريخ الاستلام
        $totalDays = 0;
        $count = 0;

        foreach ($purchaseOrders as $order) {
            if ($order->completed_at) {
                $totalDays += $order->created_at->diffInDays($order->completed_at);
                $count++;
            }
        }

        return $count > 0 ? $totalDays / $count : null;
    }

    /**
     * الحصول على العناصر التي تحتاج إلى قواعد شراء تلقائي
     *
     * @return array
     */
    private function getItemsNeedingRules()
    {
        $itemsNeedingRules = [];

        // الحصول على العناصر النشطة التي ليس لها قواعد شراء تلقائي
        $items = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->whereNotIn('id', function ($query) {
                $query->select('item_id')
                    ->from('auto_purchase_rules')
                    ->where('company_id', Auth::user()->company_id);
            })
            ->get();

        foreach ($items as $item) {
            // الحصول على معاملات الاستهلاك خلال آخر 90 يوم
            $consumptionTransactions = InventoryTransaction::where('item_id', $item->id)
                ->whereIn('transaction_type', ['sale', 'maintenance_usage', 'write_off'])
                ->where('created_at', '>=', now()->subDays(90))
                ->get();

            // تخطي العناصر التي ليس لها استهلاك
            if ($consumptionTransactions->isEmpty()) {
                continue;
            }

            // حساب إجمالي الاستهلاك
            $totalConsumption = $consumptionTransactions->sum('quantity');

            // حساب معدل الاستهلاك اليومي
            $dailyConsumptionRate = $totalConsumption / 90;

            // الحصول على المستودعات التي تم استهلاك العنصر منها
            $warehouses = $consumptionTransactions->pluck('warehouse_id')->unique()
                ->map(function ($warehouseId) use ($item, $dailyConsumptionRate) {
                    $warehouse = Warehouse::find($warehouseId);
                    if ($warehouse) {
                        // حساب معدل الاستهلاك لهذا المستودع
                        $warehouseConsumption = InventoryTransaction::where('item_id', $item->id)
                            ->where('warehouse_id', $warehouseId)
                            ->whereIn('transaction_type', ['sale', 'maintenance_usage', 'write_off'])
                            ->where('created_at', '>=', now()->subDays(90))
                            ->sum('quantity');

                        $warehouse->consumption_rate = $warehouseConsumption / 90;

                        // الحصول على المخزون الحالي
                        $warehouse->current_stock = InventoryStock::where('item_id', $item->id)
                            ->where('warehouse_id', $warehouseId)
                            ->sum('quantity');
                    }
                    return $warehouse;
                })
                ->filter();

            // الحصول على الموردين المحتملين مع معلومات إضافية
            $potentialSuppliers = InventoryTransaction::where('item_id', $item->id)
                ->whereIn('transaction_type', ['purchase'])
                ->where('created_at', '>=', now()->subDays(180))
                ->join('purchase_orders', function ($join) {
                    $join->on('inventory_transactions.reference_id', '=', 'purchase_orders.id')
                        ->where('inventory_transactions.reference_type', '=', 'purchase_order');
                })
                ->join('suppliers', 'purchase_orders.supplier_id', '=', 'suppliers.id')
                ->select(
                    'suppliers.id',
                    'suppliers.name',
                    DB::raw('COUNT(*) as purchase_count'),
                    DB::raw('AVG(inventory_transactions.unit_price) as avg_price'),
                    DB::raw('MIN(DATEDIFF(inventory_transactions.created_at, purchase_orders.created_at)) as min_lead_time'),
                    DB::raw('MAX(DATEDIFF(inventory_transactions.created_at, purchase_orders.created_at)) as max_lead_time'),
                    DB::raw('AVG(DATEDIFF(inventory_transactions.created_at, purchase_orders.created_at)) as avg_lead_time')
                )
                ->groupBy('suppliers.id', 'suppliers.name')
                ->orderBy('purchase_count', 'desc')
                ->first();

            // إضافة العنصر إلى القائمة إذا كان هناك استهلاك كافٍ
            if ($totalConsumption > 0 && $warehouses->isNotEmpty()) {
                // حساب الحد الأدنى وكمية الطلب الموصى بها بناءً على معدل الاستهلاك
                $leadTime = $potentialSuppliers->avg_lead_time ?? 7; // افتراضي: أسبوع
                $safetyStock = $dailyConsumptionRate * 3; // مخزون أمان لمدة 3 أيام
                $recommendedMinQuantity = ($dailyConsumptionRate * $leadTime) + $safetyStock;

                // حساب الكمية الاقتصادية للطلب
                $annualDemand = $dailyConsumptionRate * 365;
                $orderingCost = 50; // تكلفة إصدار أمر شراء (افتراضية)
                $holdingCost = $item->purchase_price * 0.2; // تكلفة تخزين (20% من سعر الشراء)

                $eoq = $holdingCost > 0 ? sqrt((2 * $annualDemand * $orderingCost) / $holdingCost) : $dailyConsumptionRate * 30;

                // تحديد أولوية التوصية
                $priority = 'متوسطة';

                // حساب الأيام المتبقية قبل نفاد المخزون
                $currentStock = InventoryStock::where('item_id', $item->id)->sum('quantity');
                $daysRemaining = $dailyConsumptionRate > 0 ? $currentStock / $dailyConsumptionRate : null;

                if ($daysRemaining !== null && $daysRemaining < 7) {
                    $priority = 'عالية';
                } elseif ($daysRemaining !== null && $daysRemaining > 30) {
                    $priority = 'منخفضة';
                }

                $itemsNeedingRules[] = [
                    'item' => $item,
                    'total_consumption' => $totalConsumption,
                    'daily_consumption_rate' => $dailyConsumptionRate,
                    'warehouses' => $warehouses,
                    'potential_supplier' => $potentialSuppliers,
                    'recommended_min_quantity' => $recommendedMinQuantity,
                    'recommended_order_quantity' => $eoq,
                    'days_remaining' => $daysRemaining,
                    'priority' => $priority,
                    'current_stock' => $currentStock,
                    'lead_time' => $leadTime
                ];
            }
        }

        // ترتيب العناصر حسب الأولوية
        usort($itemsNeedingRules, function ($a, $b) {
            // ترتيب حسب الأولوية أولاً
            $priorityOrder = ['عالية' => 1, 'متوسطة' => 2, 'منخفضة' => 3];
            $priorityCompare = $priorityOrder[$a['priority']] <=> $priorityOrder[$b['priority']];

            if ($priorityCompare !== 0) {
                return $priorityCompare;
            }

            // ثم حسب الأيام المتبقية (تصاعدياً)
            if ($a['days_remaining'] === null) return 1;
            if ($b['days_remaining'] === null) return -1;
            return $a['days_remaining'] <=> $b['days_remaining'];
        });

        return $itemsNeedingRules;
    }
}
