<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use App\Models\BarcodeTemplate;
use App\Models\InventoryItem;
use App\Models\InventorySerial;
use App\Models\InventoryBatch;
use App\Models\InventoryTransaction;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Milon\Barcode\Facades\DNS1D;
use Milon\Barcode\Facades\DNS2D;

class AdvancedBarcodeController extends Controller
{
    /**
     * عرض صفحة نظام الباركود المتقدم
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('view_inventory_barcodes')) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        $items = InventoryItem::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $warehouses = Warehouse::where('company_id', Auth::user()->company_id)
            ->where('is_active', true)
            ->get();

        // الحصول على قوالب الباركود المحفوظة
        $templates = BarcodeTemplate::where('company_id', Auth::user()->company_id)
            ->orderBy('name')
            ->get();

        return view('inventory.barcodes.advanced', compact('items', 'warehouses', 'templates'));
    }

    /**
     * إنشاء باركود مخصص
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function generateCustom(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('generate_inventory_barcodes')) {
            return redirect()->route('inventory.barcodes.advanced')->with('error', 'ليس لديك صلاحية لإنشاء الباركود');
        }

        $validated = $request->validate([
            'barcode_data' => 'required|string|max:255',
            'barcode_type' => 'required|in:ean13,code128,qrcode,datamatrix',
        ]);

        // إنشاء ملف الباركود
        $barcodeImage = $this->generateBarcodeImage($validated['barcode_data'], $validated['barcode_type']);
        $barcodeFilename = 'custom_barcode_' . time() . '.png';
        Storage::disk('public')->put('barcodes/' . $barcodeFilename, $barcodeImage);

        return view('inventory.barcodes.custom_result', [
            'barcode_data' => $validated['barcode_data'],
            'barcode_type' => $validated['barcode_type'],
            'barcode_image' => 'storage/barcodes/' . $barcodeFilename,
        ]);
    }

    /**
     * حفظ قالب باركود جديد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function saveTemplate(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('manage_barcode_templates')) {
            return redirect()->route('inventory.barcodes.advanced')->with('error', 'ليس لديك صلاحية لإدارة قوالب الباركود');
        }

        $validated = $request->validate([
            'template_name' => 'required|string|max:100',
            'template_width' => 'required|numeric|min:20|max:200',
            'template_height' => 'required|numeric|min:10|max:200',
            'show_name' => 'sometimes|boolean',
            'show_sku' => 'sometimes|boolean',
            'show_price' => 'sometimes|boolean',
            'show_company' => 'sometimes|boolean',
        ]);

        // إنشاء قالب جديد
        $template = new BarcodeTemplate();
        $template->company_id = Auth::user()->company_id;
        $template->name = $validated['template_name'];
        $template->width = $validated['template_width'];
        $template->height = $validated['template_height'];
        $template->settings = [
            'show_name' => $request->has('show_name'),
            'show_sku' => $request->has('show_sku'),
            'show_price' => $request->has('show_price'),
            'show_company' => $request->has('show_company'),
        ];
        $template->created_by = Auth::id();
        $template->save();

        return redirect()->route('inventory.barcodes.advanced')->with('success', 'تم حفظ قالب الباركود بنجاح');
    }

    /**
     * عرض صفحة تعديل قالب باركود
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function editTemplate($id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('manage_barcode_templates')) {
            return redirect()->route('inventory.barcodes.advanced')->with('error', 'ليس لديك صلاحية لإدارة قوالب الباركود');
        }

        $template = BarcodeTemplate::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        return view('inventory.barcodes.edit_template', compact('template'));
    }

    /**
     * تحديث قالب باركود
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateTemplate(Request $request, $id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('manage_barcode_templates')) {
            return redirect()->route('inventory.barcodes.advanced')->with('error', 'ليس لديك صلاحية لإدارة قوالب الباركود');
        }

        $template = BarcodeTemplate::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $validated = $request->validate([
            'template_name' => 'required|string|max:100',
            'template_width' => 'required|numeric|min:20|max:200',
            'template_height' => 'required|numeric|min:10|max:200',
            'show_name' => 'sometimes|boolean',
            'show_sku' => 'sometimes|boolean',
            'show_price' => 'sometimes|boolean',
            'show_company' => 'sometimes|boolean',
        ]);

        // تحديث القالب
        $template->name = $validated['template_name'];
        $template->width = $validated['template_width'];
        $template->height = $validated['template_height'];
        $template->settings = [
            'show_name' => $request->has('show_name'),
            'show_sku' => $request->has('show_sku'),
            'show_price' => $request->has('show_price'),
            'show_company' => $request->has('show_company'),
        ];
        $template->updated_by = Auth::id();
        $template->save();

        return redirect()->route('inventory.barcodes.advanced')->with('success', 'تم تحديث قالب الباركود بنجاح');
    }

    /**
     * حذف قالب باركود
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function deleteTemplate($id)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('manage_barcode_templates')) {
            return response()->json(['error' => 'ليس لديك صلاحية لإدارة قوالب الباركود'], 403);
        }

        $template = BarcodeTemplate::where('company_id', Auth::user()->company_id)
            ->findOrFail($id);

        $template->delete();

        return response()->json(['success' => 'تم حذف القالب بنجاح']);
    }

    /**
     * طباعة باركود متعدد بقالب مخصص
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function printMultiple(Request $request)
    {
        // التحقق من الصلاحيات
        if (!Auth::user()->can('generate_inventory_barcodes')) {
            return redirect()->route('inventory.barcodes.advanced')->with('error', 'ليس لديك صلاحية لطباعة الباركود');
        }

        $validated = $request->validate([
            'items' => 'required|array',
            'items.*' => 'exists:inventory_items,id',
            'barcode_type' => 'required|in:ean13,code128,qrcode',
            'label_template' => 'required|in:standard,small,large,detailed',
            'quantity_per_item' => 'required|integer|min:1|max:100',
            'include_price' => 'sometimes|boolean',
        ]);

        $items = InventoryItem::whereIn('id', $validated['items'])
            ->where('company_id', Auth::user()->company_id)
            ->get();

        $barcodeItems = [];
        foreach ($items as $item) {
            // تحديث رمز الباركود للعنصر إذا لم يكن موجودًا
            if (empty($item->barcode)) {
                $item->barcode = $this->generateBarcodeNumber();
                $item->save();
            }

            // إنشاء ملف الباركود
            $barcodeImage = $this->generateBarcodeImage($item->barcode, $validated['barcode_type']);
            $barcodeFilename = 'barcode_' . $item->id . '_' . time() . '.png';
            Storage::disk('public')->put('barcodes/' . $barcodeFilename, $barcodeImage);

            $barcodeItems[] = [
                'item' => $item,
                'quantity' => $validated['quantity_per_item'],
                'barcode_image' => 'storage/barcodes/' . $barcodeFilename,
            ];
        }

        return view('inventory.barcodes.print_multiple_result', [
            'barcode_items' => $barcodeItems,
            'barcode_type' => $validated['barcode_type'],
            'label_template' => $validated['label_template'],
            'include_price' => $request->has('include_price'),
            'company' => Auth::user()->company,
        ]);
    }

    /**
     * إنشاء رقم باركود فريد
     *
     * @return string
     */
    private function generateBarcodeNumber()
    {
        // إنشاء رقم EAN-13 فريد
        $prefix = '200'; // رمز مخصص للشركة
        $randomPart = mt_rand(100000000, 999999999);
        $barcode = $prefix . $randomPart;

        // حساب رقم التحقق
        $sum = 0;
        for ($i = 0; $i < 12; $i++) {
            $sum += (int)$barcode[$i] * ($i % 2 == 0 ? 1 : 3);
        }
        $checkDigit = (10 - ($sum % 10)) % 10;

        return $barcode . $checkDigit;
    }

    /**
     * إنشاء صورة الباركود
     *
     * @param string $barcode
     * @param string $type
     * @return string
     */
    private function generateBarcodeImage($barcode, $type)
    {
        switch ($type) {
            case 'ean13':
                return base64_decode(DNS1D::getBarcodePNG($barcode, 'EAN13', 2, 60));
            case 'code128':
                return base64_decode(DNS1D::getBarcodePNG($barcode, 'C128', 2, 60));
            case 'qrcode':
                return QrCode::format('png')
                    ->size(200)
                    ->margin(1)
                    ->generate($barcode);
            case 'datamatrix':
                return base64_decode(DNS2D::getBarcodePNG($barcode, 'DATAMATRIX', 5, 5));
            default:
                return base64_decode(DNS1D::getBarcodePNG($barcode, 'C128', 2, 60));
        }
    }
}
