<?php

namespace App\Http\Controllers;

use App\Models\InstallationProject;
use App\Models\Customer;
use App\Models\Location;
use App\Models\User;
use App\Models\Branch;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class InstallationProjectController extends Controller
{
    /**
     * استرجاع قائمة مديري المشاريع
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    private function getProjectManagers()
    {
        $user = Auth::user();
        $query = User::where('is_active', true)
            ->whereHas('roles', function ($q) {
                $q->whereIn('slug', ['admin', 'super_admin', 'project_manager']);
            });

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        return $query->get();
    }
    /**
     * عرض قائمة المشاريع
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        // التحقق من الصلاحيات
        if (!$user->hasPermission('view_installation_projects') && !$user->isSystemAdmin()) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        $query = InstallationProject::with(['customer', 'location', 'projectManager']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        // تطبيق الفلاتر
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('customer_id') && $request->customer_id != '') {
            $query->where('customer_id', $request->customer_id);
        }

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('project_number', 'like', "%{$search}%")
                    ->orWhere('title', 'like', "%{$search}%")
                    ->orWhereHas('customer', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            });
        }

        // الفرز
        $sortField = $request->sort_by ?? 'created_at';
        $sortDirection = $request->sort_direction ?? 'desc';
        $query->orderBy($sortField, $sortDirection);

        $projects = $query->paginate(10);
        
        $customerQuery = Customer::query();
        if (!$user->isSystemAdmin()) {
            $customerQuery->where('company_id', $user->company_id);
        }
        $customers = $customerQuery->get();

        return view('installation-projects.index', compact('projects', 'customers'));
    }

    /**
     * عرض نموذج إنشاء مشروع جديد
     */
    public function create()
    {
        $user = Auth::user();
        // التحقق من الصلاحيات
        if (!$user->hasPermission('create_installation_project') && !$user->isSystemAdmin()) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية لإنشاء مشروع جديد');
        }

        $customerQuery = Customer::query();
        $locationQuery = Location::query();
        $branchQuery = Branch::query();

        if (!$user->isSystemAdmin()) {
            $customerQuery->where('company_id', $user->company_id);
            $locationQuery->where('company_id', $user->company_id);
            $branchQuery->where('company_id', $user->company_id);
        }

        $customers = $customerQuery->get();
        $locations = $locationQuery->get();
        $projectManagers = $this->getProjectManagers();
        $branches = $branchQuery->get();

        // إنشاء رقم المشروع
        $latestProject = InstallationProject::latest()->first();
        $projectNumber = 'PRJ-' . date('Ymd') . '-' . sprintf('%04d', ($latestProject ? ($latestProject->id + 1) : 1));

        return view('installation-projects.create', compact('customers', 'locations', 'projectManagers', 'branches', 'projectNumber'));
    }

    /**
     * حفظ مشروع جديد
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        // التحقق من الصلاحيات
        if (!$user->hasPermission('create_installation_project') && !$user->isSystemAdmin()) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية لإنشاء مشروع جديد');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'location_id' => 'required|exists:locations,id',
            'branch_id' => 'required|exists:branches,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'expected_completion_date' => 'required|date|after_or_equal:start_date',
            'total_budget' => 'required|numeric|min:0',
            'project_manager_id' => 'nullable|exists:users,id',
            'status' => 'required|string',
            'payment_status' => 'required|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // إنشاء رقم المشروع
            $latestProject = InstallationProject::latest()->first();
            $projectNumber = 'PRJ-' . date('Ymd') . '-' . sprintf('%04d', ($latestProject ? ($latestProject->id + 1) : 1));

            // الحصول على الفرع للتأكد من الشركة
            $branch = Branch::findOrFail($request->branch_id);

            // إعداد بيانات المشروع
            $projectData = [
                'company_id' => $user->company_id ?? $branch->company_id,
                'branch_id' => $request->branch_id,
                'customer_id' => $request->customer_id,
                'location_id' => $request->location_id,
                'project_number' => $projectNumber,
                'title' => $request->title,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'expected_completion_date' => $request->expected_completion_date,
                'total_budget' => $request->total_budget,
                'status' => $request->status,
                'payment_status' => $request->payment_status,
                'notes' => $request->notes,
            ];

            // إضافة مدير المشروع فقط إذا تم تحديده
            if ($request->filled('project_manager_id')) {
                $projectData['project_manager_id'] = $request->project_manager_id;
            } else {
                // تعيين مدير المشروع كمستخدم حالي إذا كان لديه دور مناسب
                $hasManagerRole = $user->hasRole(['admin', 'manager', 'project_manager']);

                if ($hasManagerRole) {
                    $projectData['project_manager_id'] = $user->id;
                }
            }

            // إنشاء المشروع
            $project = InstallationProject::create($projectData);

            DB::commit();

            return redirect()->route('installation-projects.show', $project->id)
                ->with('success', 'تم إنشاء المشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();

            // التحقق مما إذا كان الخطأ متعلقًا بعمود project_manager_id
            if (strpos($e->getMessage(), 'project_manager_id') !== false && strpos($e->getMessage(), 'cannot be null') !== false) {
                // إذا كان الخطأ بسبب قيد قاعدة البيانات على عمود project_manager_id
                return redirect()->back()->with('error', 'يجب تحديد مدير المشروع. يرجى اختيار مدير مشروع من القائمة.')->withInput();
            }

            // رسالة خطأ عامة لأي خطأ آخر
            return redirect()->back()->with('error', 'حدث خطأ أثناء إنشاء المشروع: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * عرض تفاصيل المشروع
     */
    public function show(InstallationProject $installationProject)
    {
        $user = Auth::user();
        // التحقق من الصلاحيات
        if (!$user->hasPermission('view_installation_projects') && !$user->isSystemAdmin()) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if (!$user->isSystemAdmin() && $installationProject->company_id !== $user->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $installationProject->load([
            'customer',
            'location',
            'projectManager',
            'elevators',
            'phases.tasks',
            'teamMembers.user',
            'milestones',
            'documents',
            'expenses',
            'payments',
            'issues'
        ]);

        // حساب الإحصائيات
        $stats = [
            'progress' => $installationProject->calculateProgress(),
            'total_cost' => $installationProject->calculateTotalCost(),
            'total_payments' => $installationProject->calculateTotalPayments(),
            'remaining_budget' => $installationProject->getRemainingBudget(),
            'days_remaining' => $installationProject->getDaysRemaining(),
            'is_overdue' => $installationProject->isOverdue(),
        ];

        return view('installation-projects.show', compact('installationProject', 'stats'));
    }

    /**
     * عرض نموذج تعديل المشروع
     */
    public function edit(InstallationProject $installationProject)
    {
        $user = Auth::user();
        // التحقق من الصلاحيات
        if (!$user->hasPermission('edit_installation_project') && !$user->isSystemAdmin()) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية لتعديل المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if (!$user->isSystemAdmin() && $installationProject->company_id !== $user->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $customerQuery = Customer::query();
        $locationQuery = Location::query();
        $branchQuery = Branch::query();

        if (!$user->isSystemAdmin()) {
            $customerQuery->where('company_id', $user->company_id);
            $locationQuery->where('company_id', $user->company_id);
            $branchQuery->where('company_id', $user->company_id);
        }

        $customers = $customerQuery->get();
        $locations = $locationQuery->get();
        $projectManagers = $this->getProjectManagers();
        $branches = $branchQuery->get();

        return view('installation-projects.edit', compact('installationProject', 'customers', 'locations', 'projectManagers', 'branches'));
    }

    /**
     * تحديث المشروع
     */
    public function update(Request $request, InstallationProject $installationProject)
    {
        $user = Auth::user();
        // التحقق من الصلاحيات
        if (!$user->hasPermission('edit_installation_project') && !$user->isSystemAdmin()) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية لتعديل المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if (!$user->isSystemAdmin() && $installationProject->company_id !== $user->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'location_id' => 'required|exists:locations,id',
            'branch_id' => 'nullable|exists:branches,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'expected_completion_date' => 'required|date|after_or_equal:start_date',
            'actual_completion_date' => 'nullable|date|after_or_equal:start_date',
            'total_budget' => 'required|numeric|min:0',
            'status' => 'required|in:draft,planning,in_progress,on_hold,completed,cancelled',
            'payment_status' => 'required|in:not_started,partially_paid,fully_paid',
            'project_manager_id' => 'required|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // تحديث المشروع
            $installationProject->update([
                'branch_id' => $request->branch_id,
                'customer_id' => $request->customer_id,
                'location_id' => $request->location_id,
                'title' => $request->title,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'expected_completion_date' => $request->expected_completion_date,
                'actual_completion_date' => $request->actual_completion_date,
                'total_budget' => $request->total_budget,
                'status' => $request->status,
                'payment_status' => $request->payment_status,
                'project_manager_id' => $request->project_manager_id,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('installation-projects.show', $installationProject->id)
                ->with('success', 'تم تحديث المشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث المشروع: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * حذف المشروع
     */
    public function destroy(InstallationProject $installationProject)
    {
        $user = Auth::user();
        // التحقق من الصلاحيات
        if (!$user->hasPermission('delete_installation_project') && !$user->isSystemAdmin()) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية لحذف المشروع');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if (!$user->isSystemAdmin() && $installationProject->company_id !== $user->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        try {
            DB::beginTransaction();

            // حذف المشروع (سيتم استخدام الحذف الناعم بسبب استخدام SoftDeletes)
            $installationProject->delete();

            DB::commit();

            return redirect()->route('installation-projects.index')
                ->with('success', 'تم حذف المشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف المشروع: ' . $e->getMessage());
        }
    }

    /**
     * عرض لوحة معلومات المشروع
     */
    public function dashboard(InstallationProject $installationProject)
    {
        $user = Auth::user();
        // التحقق من الصلاحيات
        if (!$user->hasPermission('view_installation_projects') && !$user->isSystemAdmin()) {
            return redirect()->route('dashboard')->with('error', 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if (!$user->isSystemAdmin() && $installationProject->company_id !== $user->company_id) {
            return redirect()->route('installation-projects.index')->with('error', 'ليس لديك صلاحية للوصول إلى هذا المشروع');
        }

        $project = $installationProject->load([
            'customer',
            'location',
            'projectManager',
            'elevators',
            'phases.tasks',
            'teamMembers.user',
            'milestones',
            'expenses',
            'payments',
            'issues'
        ]);

        // حساب الإحصائيات
        $stats = [
            'progress' => $project->calculateProgress(),
            'total_cost' => $project->calculateTotalCost(),
            'total_payments' => $project->calculateTotalPayments(),
            'remaining_budget' => $project->getRemainingBudget(),
            'days_remaining' => $project->getDaysRemaining(),
            'is_overdue' => $project->isOverdue(),
            'total_elevators' => $project->elevators->count(),
            'total_tasks' => $project->phases->flatMap(function ($phase) {
                return $phase->tasks;
            })->count(),
            'completed_tasks' => $project->phases->flatMap(function ($phase) {
                return $phase->tasks;
            })->where('status', 'completed')->count(),
            'total_phases' => $project->phases->count(),
            'completed_phases' => $project->phases->where('status', 'completed')->count(),
            'open_issues' => $project->issues->whereIn('status', ['open', 'in_progress', 'reopened'])->count(),
        ];

        // الحصول على المهام المتأخرة
        $overdueTasks = collect();
        foreach ($project->phases as $phase) {
            $overdueTasks = $overdueTasks->merge($phase->tasks->filter(function ($task) {
                return $task->isOverdue();
            }));
        }

        // الحصول على المشكلات المفتوحة
        $openIssues = $project->issues->filter(function ($issue) {
            return in_array($issue->status, ['open', 'in_progress', 'reopened']);
        });

        // الحصول على المدفوعات القادمة
        $upcomingMilestones = $project->milestones->filter(function ($milestone) {
            return $milestone->status === 'pending' && $milestone->due_date->diffInDays(now()) <= 30;
        })->sortBy('due_date');

        return view('installation-projects.dashboard', compact('project', 'stats', 'overdueTasks', 'openIssues', 'upcomingMilestones'));
    }

    /**
     * عرض مخطط جانت للمشروع
     */
    public function ganttChart(InstallationProject $installationProject)
    {
        $user = Auth::user();
        // تسجيل معلومات التصحيح
        \Log::info('محاولة الوصول إلى مخطط جانت', [
            'user_id' => $user->id,
            'user_name' => $user->name,
            'project_id' => $installationProject->id,
            'company_id' => $user->company_id,
            'project_company_id' => $installationProject->company_id
        ]);

        // التحقق من الصلاحيات
        if (!$user->hasPermission('view_installation_projects') && !$user->isSystemAdmin()) {
            $errorMessage = 'ليس لديك صلاحية "view_installation_projects" المطلوبة للوصول إلى مخطط جانت';
            \Log::warning($errorMessage, ['user_id' => $user->id, 'user_name' => $user->name]);
            return redirect()->route('dashboard')->with('error', $errorMessage);
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if (!$user->isSystemAdmin() && $installationProject->company_id !== $user->company_id) {
            $errorMessage = 'هذا المشروع ينتمي إلى شركة مختلفة عن شركتك (شركة المشروع: ' . $installationProject->company_id . '، شركتك: ' . $user->company_id . ')';
            \Log::warning($errorMessage, [
                'user_id' => $user->id,
                'user_name' => $user->name,
                'project_id' => $installationProject->id
            ]);
            return redirect()->route('installation-projects.index')->with('error', $errorMessage);
        }

        $project = $installationProject->load(['phases.tasks']);

        // إعداد بيانات مخطط جانت
        $ganttData = [];

        // إضافة المشروع كمهمة رئيسية
        $ganttData[] = [
            'id' => 'project-' . $project->id,
            'text' => $project->title,
            'start_date' => $project->start_date->format('Y-m-d'),
            'end_date' => $project->expected_completion_date->format('Y-m-d'),
            'progress' => $project->calculateProgress() / 100,
            'type' => 'project',
            'parent' => 0
        ];

        // إضافة المراحل
        foreach ($project->phases as $phase) {
            $ganttData[] = [
                'id' => 'phase-' . $phase->id,
                'text' => $phase->name,
                'start_date' => $phase->start_date->format('Y-m-d'),
                'end_date' => $phase->end_date->format('Y-m-d'),
                'progress' => $phase->completion_percentage / 100,
                'parent' => 'project-' . $project->id,
                'type' => 'phase'
            ];

            // إضافة المهام
            foreach ($phase->tasks as $task) {
                $ganttData[] = [
                    'id' => 'task-' . $task->id,
                    'text' => $task->name,
                    'start_date' => $task->start_date->format('Y-m-d'),
                    'end_date' => $task->due_date->format('Y-m-d'),
                    'progress' => $task->completion_percentage / 100,
                    'parent' => 'phase-' . $phase->id,
                    'type' => 'task'
                ];
            }
        }

        return view('installation-projects.gantt', compact('project', 'ganttData'));
    }

    /**
     * عرض مخطط جانت للمشروع بطريقة آمنة مع معالجة الأخطاء
     * 
     * @param int $id معرف المشروع
     * @return \Illuminate\Http\Response
     */
    public function ganttChartSafe($id)
    {
        $user = Auth::user();
        // تسجيل معلومات التصحيح
        \Log::info('محاولة الوصول إلى مخطط جانت (الطريقة الآمنة)', [
            'user_id' => $user->id,
            'user_name' => $user->name,
            'requested_project_id' => $id
        ]);

        // التحقق من الصلاحيات
        if (!$user->hasPermission('view_installation_projects') && !$user->isSystemAdmin()) {
            $errorMessage = 'ليس لديك صلاحية "view_installation_projects" المطلوبة للوصول إلى مخطط جانت';
            \Log::warning($errorMessage, ['user_id' => $user->id, 'user_name' => $user->name]);
            return redirect()->route('dashboard')->with('error', $errorMessage);
        }

        // البحث عن المشروع
        $installationProject = InstallationProject::find($id);

        // التحقق من وجود المشروع
        if (!$installationProject) {
            $errorMessage = 'المشروع المطلوب غير موجود. الرجاء التحقق من معرف المشروع.';
            \Log::warning($errorMessage, ['user_id' => $user->id, 'requested_project_id' => $id]);
            return response()->view('errors.installation-project-not-found', [], 404);
        }

        // التحقق من أن المشروع ينتمي لنفس الشركة
        if (!$user->isSystemAdmin() && $installationProject->company_id !== $user->company_id) {
            $errorMessage = 'هذا المشروع ينتمي إلى شركة مختلفة عن شركتك (شركة المشروع: ' . $installationProject->company_id . '، شركتك: ' . $user->company_id . ')';
            \Log::warning($errorMessage, [
                'user_id' => $user->id,
                'user_name' => $user->name,
                'project_id' => $installationProject->id
            ]);
            return redirect()->route('installation-projects.index')->with('error', $errorMessage);
        }

        $project = $installationProject->load(['phases.tasks']);

        // التحقق من وجود مراحل ومهام
        if ($project->phases->isEmpty()) {
            \Log::info('المشروع لا يحتوي على مراحل', ['project_id' => $project->id]);
        }

        // إعداد بيانات مخطط جانت
        $ganttData = [];

        // إضافة المشروع كمهمة رئيسية
        $ganttData[] = [
            'id' => 'project-' . $project->id,
            'text' => $project->title,
            'start_date' => $project->start_date->format('Y-m-d'),
            'end_date' => $project->expected_completion_date->format('Y-m-d'),
            'progress' => $project->calculateProgress() / 100,
            'type' => 'project',
            'parent' => 0
        ];

        // إضافة المراحل
        foreach ($project->phases as $phase) {
            $ganttData[] = [
                'id' => 'phase-' . $phase->id,
                'text' => $phase->name,
                'start_date' => $phase->start_date->format('Y-m-d'),
                'end_date' => $phase->end_date->format('Y-m-d'),
                'progress' => $phase->completion_percentage / 100,
                'parent' => 'project-' . $project->id,
                'type' => 'phase'
            ];

            // إضافة المهام
            foreach ($phase->tasks as $task) {
                $ganttData[] = [
                    'id' => 'task-' . $task->id,
                    'text' => $task->name,
                    'start_date' => $task->start_date->format('Y-m-d'),
                    'end_date' => $task->due_date->format('Y-m-d'),
                    'progress' => $task->completion_percentage / 100,
                    'parent' => 'phase-' . $phase->id,
                    'type' => 'task'
                ];
            }
        }

        return view('installation-projects.gantt', compact('project', 'ganttData'));
    }
}
