<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\Company;
use App\Models\HRM\Employee;
use App\Models\HRM\Training;
use App\Models\HRM\TrainingParticipant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class TrainingController extends Controller
{
    /**
     * عرض قائمة التدريبات
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $query = Training::with(['participants.employee'])
            ->where('company_id', $companyId);

        // تطبيق الفلاتر إذا وجدت
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('start_date') && $request->start_date != '') {
            $query->whereDate('start_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date != '') {
            $query->whereDate('end_date', '<=', $request->end_date);
        }

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('trainer_name', 'like', "%{$search}%")
                    ->orWhere('trainer_organization', 'like', "%{$search}%")
                    ->orWhere('location', 'like', "%{$search}%");
            });
        }

        $trainings = $query->orderBy('start_date', 'desc')->paginate(10);

        return view('hrm.trainings.index', compact('trainings'));
    }

    /**
     * عرض نموذج إنشاء تدريب جديد
     */
    public function create()
    {
        $companyId = session('company_id');
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        return view('hrm.trainings.create', compact('employees'));
    }

    /**
     * حفظ تدريب جديد
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'trainer_name' => 'nullable|string|max:255',
            'trainer_organization' => 'nullable|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'location' => 'nullable|string|max:255',
            'training_type' => 'nullable|string|max:255',
            'cost' => 'nullable|numeric|min:0',
            'status' => 'required|in:planned,in_progress,completed,cancelled',
            'notes' => 'nullable|string',
            'employees' => 'nullable|array',
            'employees.*' => 'exists:employees,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = session('company_id');
        $data = $request->all();
        $data['company_id'] = $companyId;
        $data['created_by'] = Auth::id();

        $training = Training::create($data);

        // إضافة المشاركين إذا تم تحديدهم
        if ($request->has('employees') && is_array($request->employees)) {
            foreach ($request->employees as $employeeId) {
                TrainingParticipant::create([
                    'company_id' => $companyId,
                    'training_id' => $training->id,
                    'employee_id' => $employeeId,
                    'status' => 'invited',
                    'created_by' => Auth::id(),
                ]);
            }
        }

        return redirect()->route('hrm.trainings.index')
            ->with('success', 'تم إنشاء التدريب بنجاح');
    }

    /**
     * عرض تفاصيل التدريب
     */
    public function show($id)
    {
        $companyId = session('company_id');
        $training = Training::with(['participants.employee'])
            ->where('company_id', $companyId)
            ->findOrFail($id);

        return view('hrm.trainings.show', compact('training'));
    }

    /**
     * عرض نموذج تعديل التدريب
     */
    public function edit($id)
    {
        $companyId = session('company_id');
        $training = Training::where('company_id', $companyId)->findOrFail($id);

        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        $participants = TrainingParticipant::where('training_id', $id)
            ->pluck('employee_id')
            ->toArray();

        return view('hrm.trainings.edit', compact('training', 'employees', 'participants'));
    }

    /**
     * تحديث التدريب
     */
    public function update(Request $request, $id)
    {
        $companyId = session('company_id');
        $training = Training::where('company_id', $companyId)->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'trainer_name' => 'nullable|string|max:255',
            'trainer_organization' => 'nullable|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'location' => 'nullable|string|max:255',
            'training_type' => 'nullable|string|max:255',
            'cost' => 'nullable|numeric|min:0',
            'status' => 'required|in:planned,in_progress,completed,cancelled',
            'notes' => 'nullable|string',
            'employees' => 'nullable|array',
            'employees.*' => 'exists:employees,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->all();
        $data['updated_by'] = Auth::id();

        $training->update($data);

        // تحديث المشاركين
        if ($request->has('employees')) {
            // حذف المشاركين الحاليين غير الموجودين في القائمة الجديدة
            TrainingParticipant::where('training_id', $id)
                ->whereNotIn('employee_id', $request->employees)
                ->delete();

            // إضافة المشاركين الجدد
            $existingParticipants = TrainingParticipant::where('training_id', $id)
                ->pluck('employee_id')
                ->toArray();

            foreach ($request->employees as $employeeId) {
                if (!in_array($employeeId, $existingParticipants)) {
                    TrainingParticipant::create([
                        'company_id' => $companyId,
                        'training_id' => $id,
                        'employee_id' => $employeeId,
                        'status' => 'invited',
                        'created_by' => Auth::id(),
                    ]);
                }
            }
        } else {
            // إذا لم يتم تحديد أي مشاركين، قم بحذف جميع المشاركين الحاليين
            TrainingParticipant::where('training_id', $id)->delete();
        }

        return redirect()->route('hrm.trainings.index')
            ->with('success', 'تم تحديث التدريب بنجاح');
    }

    /**
     * حذف التدريب
     */
    public function destroy($id)
    {
        $companyId = session('company_id');
        $training = Training::where('company_id', $companyId)->findOrFail($id);

        // حذف المشاركين أولاً
        TrainingParticipant::where('training_id', $id)->delete();

        // ثم حذف التدريب
        $training->delete();

        return redirect()->route('hrm.trainings.index')
            ->with('success', 'تم حذف التدريب بنجاح');
    }

    /**
     * تحديث حالة مشارك في التدريب
     */
    public function updateParticipantStatus(Request $request, $id, $participantId)
    {
        $companyId = session('company_id');
        $participant = TrainingParticipant::where('company_id', $companyId)
            ->where('training_id', $id)
            ->where('id', $participantId)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'status' => 'required|in:invited,confirmed,declined,completed',
            'attendance_status' => 'nullable|in:present,absent,late,excused',
            'score' => 'nullable|numeric|min:0|max:100',
            'feedback' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->only(['status', 'attendance_status', 'score', 'feedback', 'notes']);
        $data['updated_by'] = Auth::id();

        $participant->update($data);

        return redirect()->route('hrm.trainings.show', $id)
            ->with('success', 'تم تحديث حالة المشارك بنجاح');
    }

    /**
     * إصدار شهادة للمشارك
     */
    public function issueCertificate(Request $request, $id, $participantId)
    {
        $companyId = session('company_id');
        $participant = TrainingParticipant::where('company_id', $companyId)
            ->where('training_id', $id)
            ->where('id', $participantId)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'certificate_number' => 'required|string|max:255',
            'certificate_date' => 'required|date',
            'certificate_file' => 'required|file|mimes:pdf,jpg,jpeg,png|max:5120',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // تخزين ملف الشهادة
        $path = $request->file('certificate_file')->store('certificates', 'public');

        $participant->update([
            'certificate_issued' => true,
            'certificate_number' => $request->certificate_number,
            'certificate_date' => $request->certificate_date,
            'certificate_file' => $path,
            'updated_by' => Auth::id(),
        ]);

        return redirect()->route('hrm.trainings.show', $id)
            ->with('success', 'تم إصدار الشهادة بنجاح');
    }
}
