<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\Department;
use App\Models\Employee;
use App\Models\Leave;
use App\Models\Salary;
use App\Models\SalaryComponent;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class SalaryController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->session()->get('company_id');
        $query = Salary::with(['employee', 'createdBy'])
            ->where('company_id', $companyId);

        // Apply filters if provided
        if ($request->has('employee_id') && $request->employee_id != '') {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->has('month') && $request->month != '') {
            $query->where('month', $request->month);
        }

        if ($request->has('year') && $request->year != '') {
            $query->where('year', $request->year);
        }

        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('department_id') && $request->department_id != '') {
            $query->whereHas('employee', function ($q) use ($request) {
                $q->where('department_id', $request->department_id);
            });
        }

        $salaries = $query->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->paginate(10);

        // Get all employees for filter dropdown
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Get all departments for filter dropdown
        $departments = Department::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Get available months and years for filter dropdowns
        $availableMonths = Salary::where('company_id', $companyId)
            ->select('month')
            ->distinct()
            ->orderBy('month')
            ->pluck('month');

        $availableYears = Salary::where('company_id', $companyId)
            ->select('year')
            ->distinct()
            ->orderBy('year', 'desc')
            ->pluck('year');

        return view('hrm.salaries.index', compact('salaries', 'employees', 'departments', 'availableMonths', 'availableYears'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $companyId = $request->session()->get('company_id');

        // Get all employees for dropdown
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Get all salary components for the company
        $salaryComponents = SalaryComponent::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Default to current month and year
        $month = date('m');
        $year = date('Y');

        return view('hrm.salaries.create', compact('employees', 'salaryComponents', 'month', 'year'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'month' => 'required|integer|min:1|max:12',
            'year' => 'required|integer|min:2000|max:2100',
            'basic_salary' => 'required|numeric|min:0',
            'total_allowances' => 'required|numeric|min:0',
            'total_deductions' => 'required|numeric|min:0',
            'net_salary' => 'required|numeric|min:0',
            'payment_method' => 'required|in:bank_transfer,cash,check,other',
            'payment_details' => 'nullable|string',
            'comments' => 'nullable|string',
            'attachment' => 'nullable|file|max:10240', // 10MB max
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');

        // Check if employee belongs to the company
        $employee = Employee::findOrFail($request->employee_id);
        if ($employee->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Check if salary for this employee and month/year already exists
        $existingSalary = Salary::where('employee_id', $request->employee_id)
            ->where('month', $request->month)
            ->where('year', $request->year)
            ->first();

        if ($existingSalary) {
            return redirect()->back()
                ->with('error', 'Salary record already exists for this employee for the selected month and year.')
                ->withInput();
        }

        // Handle attachment upload
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('salaries/attachments', 'public');
        }

        // Start a database transaction
        DB::beginTransaction();

        try {
            // Create the salary record
            $salary = new Salary();
            $salary->company_id = $companyId;
            $salary->employee_id = $request->employee_id;
            $salary->month = $request->month;
            $salary->year = $request->year;
            $salary->basic_salary = $request->basic_salary;
            $salary->total_allowances = $request->total_allowances;
            $salary->total_deductions = $request->total_deductions;
            $salary->net_salary = $request->net_salary;
            $salary->payment_method = $request->payment_method;
            $salary->payment_details = $request->payment_details;
            $salary->comments = $request->comments;
            $salary->attachment = $attachmentPath;
            $salary->status = 'pending';
            $salary->created_by = Auth::id();
            $salary->save();

            // Save salary components if provided
            if ($request->has('components') && is_array($request->components)) {
                foreach ($request->components as $componentId => $amount) {
                    if ($amount > 0) {
                        $component = SalaryComponent::findOrFail($componentId);

                        // Check if component belongs to the company
                        if ($component->company_id != $companyId) {
                            continue; // Skip this component
                        }

                        $salary->components()->attach($componentId, [
                            'amount' => $amount,
                            'type' => $component->type, // allowance or deduction
                        ]);
                    }
                }
            }

            DB::commit();

            return redirect()->route('hrm.salaries.index')
                ->with('success', 'Salary record created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'An error occurred while creating the salary record: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $salary = Salary::with(['employee', 'components', 'createdBy'])->findOrFail($id);

        // Check if salary belongs to the current company
        $companyId = session('company_id');
        if ($salary->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Get attendance data for the month
        $startDate = Carbon::createFromDate($salary->year, $salary->month, 1);
        $endDate = $startDate->copy()->endOfMonth();

        $attendanceData = Attendance::where('employee_id', $salary->employee_id)
            ->whereBetween('date', [$startDate, $endDate])
            ->get();

        $totalWorkDays = $attendanceData->where('status', 'present')->count();
        $totalAbsentDays = $attendanceData->where('status', 'absent')->count();
        $totalLateDays = $attendanceData->where('status', 'late')->count();
        $totalHalfDays = $attendanceData->where('status', 'half_day')->count();
        $totalLeaveDays = Leave::where('employee_id', $salary->employee_id)
            ->where('status', 'approved')
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate]);
            })
            ->sum('total_days');

        return view('hrm.salaries.show', compact('salary', 'totalWorkDays', 'totalAbsentDays', 'totalLateDays', 'totalHalfDays', 'totalLeaveDays'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        $companyId = $request->session()->get('company_id');
        $salary = Salary::with('components')->findOrFail($id);

        // Check if salary belongs to the current company
        if ($salary->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only pending salaries can be edited
        if ($salary->status != 'pending') {
            return redirect()->route('hrm.salaries.show', $id)
                ->with('error', 'Only pending salary records can be edited.');
        }

        // Get all salary components for the company
        $salaryComponents = SalaryComponent::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Prepare component amounts for the form
        $componentAmounts = [];
        foreach ($salary->components as $component) {
            $componentAmounts[$component->id] = $component->pivot->amount;
        }

        return view('hrm.salaries.edit', compact('salary', 'salaryComponents', 'componentAmounts'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $salary = Salary::findOrFail($id);

        // Check if salary belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($salary->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only pending salaries can be updated
        if ($salary->status != 'pending') {
            return redirect()->route('hrm.salaries.show', $id)
                ->with('error', 'Only pending salary records can be updated.');
        }

        $validator = Validator::make($request->all(), [
            'basic_salary' => 'required|numeric|min:0',
            'total_allowances' => 'required|numeric|min:0',
            'total_deductions' => 'required|numeric|min:0',
            'net_salary' => 'required|numeric|min:0',
            'payment_method' => 'required|in:bank_transfer,cash,check,other',
            'payment_details' => 'nullable|string',
            'comments' => 'nullable|string',
            'attachment' => 'nullable|file|max:10240', // 10MB max
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Handle attachment upload
        if ($request->hasFile('attachment')) {
            // Delete old attachment if exists
            if ($salary->attachment) {
                Storage::disk('public')->delete($salary->attachment);
            }

            $attachmentPath = $request->file('attachment')->store('salaries/attachments', 'public');
            $salary->attachment = $attachmentPath;
        }

        // Start a database transaction
        DB::beginTransaction();

        try {
            // Update the salary record
            $salary->basic_salary = $request->basic_salary;
            $salary->total_allowances = $request->total_allowances;
            $salary->total_deductions = $request->total_deductions;
            $salary->net_salary = $request->net_salary;
            $salary->payment_method = $request->payment_method;
            $salary->payment_details = $request->payment_details;
            $salary->comments = $request->comments;
            $salary->updated_by = Auth::id();
            $salary->save();

            // Update salary components
            $salary->components()->detach(); // Remove all existing components

            if ($request->has('components') && is_array($request->components)) {
                foreach ($request->components as $componentId => $amount) {
                    if ($amount > 0) {
                        $component = SalaryComponent::findOrFail($componentId);

                        // Check if component belongs to the company
                        if ($component->company_id != $companyId) {
                            continue; // Skip this component
                        }

                        $salary->components()->attach($componentId, [
                            'amount' => $amount,
                            'type' => $component->type, // allowance or deduction
                        ]);
                    }
                }
            }

            DB::commit();

            return redirect()->route('hrm.salaries.show', $id)
                ->with('success', 'Salary record updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'An error occurred while updating the salary record: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        $salary = Salary::findOrFail($id);

        // Check if salary belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($salary->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only pending salaries can be deleted
        if ($salary->status != 'pending') {
            return redirect()->route('hrm.salaries.show', $id)
                ->with('error', 'Only pending salary records can be deleted.');
        }

        // Delete attachment if exists
        if ($salary->attachment) {
            Storage::disk('public')->delete($salary->attachment);
        }

        // Start a database transaction
        DB::beginTransaction();

        try {
            // Delete salary components
            $salary->components()->detach();

            // Delete the salary record
            $salary->delete();

            DB::commit();

            return redirect()->route('hrm.salaries.index')
                ->with('success', 'Salary record deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'An error occurred while deleting the salary record: ' . $e->getMessage());
        }
    }

    /**
     * Approve a salary record.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function approve(Request $request, $id)
    {
        $salary = Salary::findOrFail($id);

        // Check if salary belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($salary->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only pending salaries can be approved
        if ($salary->status != 'pending') {
            return redirect()->route('hrm.salaries.show', $id)
                ->with('error', 'Only pending salary records can be approved.');
        }

        $salary->status = 'approved';
        $salary->approved_by = Auth::id();
        $salary->approved_at = now();
        $salary->save();

        return redirect()->route('hrm.salaries.show', $id)
            ->with('success', 'Salary record approved successfully.');
    }

    /**
     * Mark a salary record as paid.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function markAsPaid(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'payment_date' => 'required|date',
            'payment_reference' => 'nullable|string|max:255',
            'payment_notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $salary = Salary::findOrFail($id);

        // Check if salary belongs to the current company
        $companyId = $request->session()->get('company_id');
        if ($salary->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Only approved salaries can be marked as paid
        if ($salary->status != 'approved') {
            return redirect()->route('hrm.salaries.show', $id)
                ->with('error', 'Only approved salary records can be marked as paid.');
        }

        $salary->status = 'paid';
        $salary->payment_date = $request->payment_date;
        $salary->payment_reference = $request->payment_reference;
        $salary->payment_notes = $request->payment_notes;
        $salary->paid_by = Auth::id();
        $salary->save();

        return redirect()->route('hrm.salaries.show', $id)
            ->with('success', 'Salary record marked as paid successfully.');
    }

    /**
     * Generate salary records for multiple employees.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function generateBulk(Request $request)
    {
        $companyId = $request->session()->get('company_id');

        // Get all employees for filter dropdown
        $employees = Employee::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Get all departments for filter dropdown
        $departments = Department::where('company_id', $companyId)
            ->where('status', 'active')
            ->get();

        // Default to current month and year
        $month = date('m');
        $year = date('Y');

        return view('hrm.salaries.generate_bulk', compact('employees', 'departments', 'month', 'year'));
    }

    /**
     * Process bulk salary generation.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function processBulk(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'month' => 'required|integer|min:1|max:12',
            'year' => 'required|integer|min:2000|max:2100',
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'exists:employees,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = $request->session()->get('company_id');
        $month = $request->month;
        $year = $request->year;
        $employeeIds = $request->employee_ids;

        // Start a database transaction
        DB::beginTransaction();

        try {
            $successCount = 0;
            $skipCount = 0;
            $errorCount = 0;

            foreach ($employeeIds as $employeeId) {
                // Check if employee belongs to the company
                $employee = Employee::findOrFail($employeeId);
                if ($employee->company_id != $companyId) {
                    $errorCount++;
                    continue; // Skip this employee
                }

                // Check if salary for this employee and month/year already exists
                $existingSalary = Salary::where('employee_id', $employeeId)
                    ->where('month', $month)
                    ->where('year', $year)
                    ->first();

                if ($existingSalary) {
                    $skipCount++;
                    continue; // Skip this employee
                }

                // Calculate salary components based on employee's contract
                $basicSalary = $employee->basic_salary ?? 0;
                $totalAllowances = 0;
                $totalDeductions = 0;

                // Get all active salary components for this employee
                $employeeComponents = $employee->salaryComponents()
                    ->where('status', 'active')
                    ->get();

                $componentData = [];

                foreach ($employeeComponents as $component) {
                    $amount = $component->pivot->amount;

                    if ($component->type == 'allowance') {
                        $totalAllowances += $amount;
                    } else {
                        $totalDeductions += $amount;
                    }

                    $componentData[$component->id] = [
                        'amount' => $amount,
                        'type' => $component->type,
                    ];
                }

                // Calculate net salary
                $netSalary = $basicSalary + $totalAllowances - $totalDeductions;

                // Create the salary record
                $salary = new Salary();
                $salary->company_id = $companyId;
                $salary->employee_id = $employeeId;
                $salary->month = $month;
                $salary->year = $year;
                $salary->basic_salary = $basicSalary;
                $salary->total_allowances = $totalAllowances;
                $salary->total_deductions = $totalDeductions;
                $salary->net_salary = $netSalary;
                $salary->payment_method = $employee->payment_method ?? 'bank_transfer';
                $salary->payment_details = $employee->payment_details ?? '';
                $salary->status = 'pending';
                $salary->created_by = Auth::id();
                $salary->save();

                // Save salary components
                foreach ($componentData as $componentId => $data) {
                    $salary->components()->attach($componentId, [
                        'amount' => $data['amount'],
                        'type' => $data['type'],
                    ]);
                }

                $successCount++;
            }

            DB::commit();

            $message = "Salary generation completed: {$successCount} records created, {$skipCount} skipped (already exist), {$errorCount} errors.";

            return redirect()->route('hrm.salaries.index', ['month' => $month, 'year' => $year])
                ->with('success', $message);
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'An error occurred while generating salary records: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Generate salary slip PDF.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function generateSlip($id)
    {
        $salary = Salary::with(['employee', 'employee.department', 'components', 'createdBy'])->findOrFail($id);

        // Check if salary belongs to the current company
        $companyId = session('company_id');
        if ($salary->company_id != $companyId) {
            abort(403, 'Unauthorized action.');
        }

        // Get attendance data for the month
        $startDate = Carbon::createFromDate($salary->year, $salary->month, 1);
        $endDate = $startDate->copy()->endOfMonth();

        $attendanceData = Attendance::where('employee_id', $salary->employee_id)
            ->whereBetween('date', [$startDate, $endDate])
            ->get();

        $totalWorkDays = $attendanceData->where('status', 'present')->count();
        $totalAbsentDays = $attendanceData->where('status', 'absent')->count();
        $totalLateDays = $attendanceData->where('status', 'late')->count();
        $totalHalfDays = $attendanceData->where('status', 'half_day')->count();
        $totalLeaveDays = Leave::where('employee_id', $salary->employee_id)
            ->where('status', 'approved')
            ->where(function ($query) use ($startDate, $endDate) {
                $query->whereBetween('start_date', [$startDate, $endDate])
                    ->orWhereBetween('end_date', [$startDate, $endDate]);
            })
            ->sum('total_days');

        // Group components by type
        $allowances = $salary->components->where('type', 'allowance');
        $deductions = $salary->components->where('type', 'deduction');

        // Generate PDF
        $pdf = PDF::loadView('hrm.salaries.slip', compact(
            'salary',
            'allowances',
            'deductions',
            'totalWorkDays',
            'totalAbsentDays',
            'totalLateDays',
            'totalHalfDays',
            'totalLeaveDays'
        ));

        return $pdf->download('salary-slip-' . $salary->employee->employee_id . '-' . $salary->month . '-' . $salary->year . '.pdf');
    }
}
